﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <string>
#include <vector>
#include <map>

namespace qcit
{
    // 本テスト用のテストデータレコード定義
    struct ContentRecord
    {
        std::string id;
        std::string nspFileName;
        std::string type;
        std::string digest;
        std::string dataSize;
        std::string applicationId;
    };

    struct SyncListRecord
    {
        std::string type;
        std::string id;
        std::string index;
    };

    class ContentDataContainer
    {
        NN_DISALLOW_COPY(ContentDataContainer);
        NN_DISALLOW_MOVE(ContentDataContainer);
    public:
        ContentDataContainer() NN_NOEXCEPT;
        virtual ~ContentDataContainer() NN_NOEXCEPT;

        // テストデータの元になる一覧データの読み込み
        // m_TestDataList に展開する
        nn::Result ReadFromContentInfoCSV(const std::string& inInfoFilePath) NN_NOEXCEPT;
        nn::Result ReadFromSyncListCSV(const std::string& inSyncListFilePath) NN_NOEXCEPT;

        std::vector<ContentRecord>& GetDataList() NN_NOEXCEPT
        {
            return m_ContentDataList;
        }

        std::map<nn::Bit64, ContentRecord>& GetInfoMap() NN_NOEXCEPT
        {
            return m_ContentInfoMap;
        }

        std::vector<SyncListRecord>& GetSyncList() NN_NOEXCEPT
        {
            return m_SyncList;
        }

    private:
        using CsvRecord = std::vector<std::string>;
        using CsvDataTable = std::vector<CsvRecord>;

        CsvDataTable ParseTable(const std::string& inData, const std::string& delim) NN_NOEXCEPT;

        void ToContentDataContainer(CsvDataTable& inCSVData) NN_NOEXCEPT;
        void ToSyncListContainer(CsvDataTable& inCSVData) NN_NOEXCEPT;
        void SetCsvRecordData(ContentRecord& outRecord, CsvRecord& inCSVRecordData) NN_NOEXCEPT;
        void SetSyncListCsvRecordData(SyncListRecord& outRecord, CsvRecord& inCSVRecordData) NN_NOEXCEPT;

        void InitializeTitleIndexMap() NN_NOEXCEPT;
        int FindIndexVector(CsvRecord& inFindList, const std::string& inFindValue) NN_NOEXCEPT;
        std::string SafeAt(CsvRecord& vec, const std::string& titleStr) NN_NOEXCEPT;

    private:
        std::vector<ContentRecord> m_ContentDataList;
        std::map<nn::Bit64, ContentRecord> m_ContentInfoMap;

        std::vector<SyncListRecord> m_SyncList;

        std::map<std::string, int> m_TitleIndexMap;
    };

} // ~namespace qcit
