﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/repair.h>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/manu/manu_Api.h>
#include <nn/repair/repair_Authentication.h>
#include <nn/repair/repair_LabelButton.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_Sdcard.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_CommandLineOption.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace repair {

/**
* @brief  セーブデータを Raw バイナリで抽出します。
*
* @param[in]   path        抜き出したセーブデータを保存するディレクトリのパス
* @param[in]   pCallback   メッセージ表示用の構造体
*
* @return      処理の結果を返します。
* @retval      nn::ResultSuccess   セーブデータの抜き出しに成功しました。
*
* @details
*/
    nn::Result UnsafeExtract(
        const char* path,
        const ReportCallbackStruct* pCallback) NN_NOEXCEPT;
}}

#define RETURN_IF_FAILURE(result, ...) \
    do { \
    if (result.IsFailure()){SendString(__VA_ARGS__); return result;} \
    } while(NN_STATIC_CONDITION(false))

namespace {

static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;
const size_t messageBufferSize = 1024;
char messageBuffer[messageBufferSize] = {};

static nn::os::MutexType s_Mutex;

static const int TOOL_MAJOR_VERSION = 5;
static const int TOOL_MINOR_VERSION = 0;

static glv::GLV           s_GlvRootView;
static nn::repair::StreamTextView*    s_pTextView;
static nn::os::ThreadType s_SdWaiterThreadType;
static nn::os::ThreadType s_ProcessingThreadType;
static nn::os::EventType  s_SdInsertionEvent;

static nn::repair::LabelButton* s_StartButton;
static nn::repair::LabelButton* s_ExitButton;

static const char* const s_AutoExitOption = "--autoexit";
static const std::string s_MountName = "Target";
static const std::string s_ProjectPath = s_MountName + ":/" + "UnsafeExtract/";

static void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

static void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

static void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

static void SetupPeripherals() NN_NOEXCEPT
{
    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}

void SendString(const char* pFormat, ...) NN_NOEXCEPT
{
    nn::os::LockMutex(&s_Mutex);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::UnlockMutex(&s_Mutex);
    };

    va_list args;
    va_start(args, pFormat);
    auto len = vsnprintf(messageBuffer, messageBufferSize, pFormat, args);
    va_end(args);

#if !defined(NN_REPAIR_DEVELOP_MODE)
    size_t writeSize;
    auto result = nn::manu::UsbWrite(&writeSize, messageBuffer, messageBufferSize, len);
    if (result.IsFailure())
    {
        NN_SDK_LOG("%s:%d %016x\n", __FILE__, __LINE__, result.GetInnerValueForDebug());
    }
#else
    // NN_SDK_LOG(messageBuffer);
#endif

    if(len)
    {
        s_pTextView->AppendValue(messageBuffer);
    }
}

void ShowDeviceId() NN_NOEXCEPT
{
    char id[nn::repair::DeviceIdLength];
    nn::repair::GetDeviceId(id);
    SendString("Device Id : %s\n", id);
}

void ShowSerialNumber() NN_NOEXCEPT
{
    char number[nn::repair::SerialNumberLength];
    nn::repair::GetSerialNumber(number);
    SendString("Serial Number : %s\n", number);
}

void ShowInternalInformation() NN_NOEXCEPT
{
    // デバイス ID 表示
    ShowDeviceId();

    // シリアルナンバー表示
    ShowSerialNumber();
}

nn::Result Backup(bool* pOutHasCorruption) NN_NOEXCEPT
{
    auto result = nn::fs::MountSdCard(s_MountName.c_str());
    RETURN_IF_FAILURE(result, "Failed to mount device\n");
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(s_MountName.c_str());
    };

    nn::fs::DirectoryEntryType type;
    result = nn::fs::GetEntryType(&type, s_ProjectPath.c_str());

    if ( result.IsFailure() )
    {
        result = nn::fs::CreateDirectory(s_ProjectPath.c_str());
        RETURN_IF_FAILURE(result, "Failed to create the project directory\n");
    }

    char number[nn::repair::SerialNumberLength];
    nn::repair::GetSerialNumber(number);

    // 保存先のファイルパス
    std::string targetPath = s_ProjectPath + std::string(number) + "/";

    // 同一シリアルの場合はクリーンアップ
    result = nn::fs::GetEntryType(&type, targetPath.c_str());
    if ( result.IsSuccess() )
    {
        result = nn::fs::DeleteDirectoryRecursively(targetPath.c_str());
        RETURN_IF_FAILURE(result, "Failed to delete the target directory\n");
    }

    result = nn::fs::CreateDirectory(targetPath.c_str());
    RETURN_IF_FAILURE(result, "Failed to create the target directory\n");

    nn::repair::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::repair::Finalize();
    };

    SendString("- Backup save data\n");
    {
        nn::repair::ReportCallbackStruct callback = { SendString, nullptr };
        result = nn::repair::UnsafeExtract(targetPath.c_str(), &callback);

        RETURN_IF_FAILURE(result, "Failed to export save data\n");
    }
    SendString("- Backup done\n");

    NN_RESULT_SUCCESS;
}

void ApplicationExit()
{
    // SD 挿入待ちイベントの終了
    nn::os::FinalizeEvent(&s_SdInsertionEvent);

#if !defined(NN_REPAIR_DEVELOP_MODE)
    // 中断時は Repair サービスへの入力待ち状態になっているので適当な値を送って終了させる
    auto message = nn::repair::BackupRequestMessage::MakeZero();
    nn::manu::WriteToHost(&message, sizeof(message), nn::repair::GetOptionArgument(nn::repair::RequestOption), 0, sizeof(message));
    nn::manu::FinalizeUfio();
#endif

    nn::os::FinalizeMutex(&s_Mutex);

    std::quick_exit(0);
}

void ProcessingThread(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    SendString("=== Backup Start ===\n");

    bool hasCorruption = false;
    auto result = Backup(&hasCorruption);
    SendString("M: %d, D: %d, I: %d\n",
        result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());

    if (nn::repair::ResultInMigration().Includes(result))
    {
        SendString("This console is maybe in migration mode.\n");
        SendString("Please run proper tool before backup.\n");
    }

    if (result.IsSuccess() && (!hasCorruption))
    {
        SendString("[[ SUCCESS ]]\n");
    }
    else if (result.IsSuccess() && hasCorruption)
    {
        SendString("[[ LIMITED SUCCESS ]]\n");
    }
    else
    {
        SendString("[[ FAILED ]]\n");
    }

    if (nn::repair::HasOption(s_AutoExitOption))
    {
        ApplicationExit();
    }
}

void BackupMouseDownHandler() NN_NOEXCEPT
{
}

void SdcardWaiter(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::os::InitializeEvent(&s_SdInsertionEvent, false, nn::os::EventClearMode_AutoClear);

    if (nn::repair::HasOption(s_AutoExitOption))
    {
        SendString("=== Auto Exit mode ===\n");
    }

    SendString("Please insert your micro sd card\n");

    std::unique_ptr<nn::repair::Sdcard> sdcard(new nn::repair::Sdcard());
    auto result = sdcard->WaitAttach();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    SendString("Ready to backup !\n");
    SendString("Push START button\n");

    nn::os::SignalEvent(&s_SdInsertionEvent);

    // バックアップツールは SD カードの挿入を検知して動作開始
    nn::os::StartThread(&s_ProcessingThreadType);
}

void BackupWindowCreateHandler() NN_NOEXCEPT
{
    nn::os::InitializeEvent(&s_SdInsertionEvent, false, nn::os::EventClearMode_ManualClear);

    // 本体情報の表示
    ShowInternalInformation();

    // SD カードの挿入待ち
    NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_ThreadStack[0x10000];
    auto result = nn::os::CreateThread(
            &s_SdWaiterThreadType, SdcardWaiter, nullptr, s_ThreadStack,
            sizeof(s_ThreadStack), nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::StartThread(&s_SdWaiterThreadType);
}

void BackupWindowDestroyHandler() NN_NOEXCEPT
{
    nn::os::DestroyThread(&s_SdWaiterThreadType);
}

void ExitMouseDownHandler() NN_NOEXCEPT
{
    // 処理スレッドの破棄
    nn::os::DestroyThread(&s_ProcessingThreadType);

    ApplicationExit();
}

void Main( glv::Window& window ) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    // ログビュー表示
    s_pTextView = new nn::repair::StreamTextView(glv::Rect(1024, 512), 24.f);
    s_pTextView->pos(120, 100);
    *pGlvRootView << s_pTextView;

    // ツール情報ラベル表示
    auto titleLabel = new glv::Label("Unsafe Extract Tool", false);
    nn::repair::GetToolInformationLabel(
            titleLabel, "Unsafe Extract Tool", TOOL_MAJOR_VERSION, TOOL_MINOR_VERSION);
    *pGlvRootView << titleLabel;

    // 開始ボタン表示
    s_StartButton = new nn::repair::LabelButton(
            "START",
            [&]{BackupMouseDownHandler();},
            [&]{BackupWindowCreateHandler();},
            [&]{BackupWindowDestroyHandler();});
    s_StartButton->pos(120, 640);
    *pGlvRootView << s_StartButton;

    // 終了ボタン表示
    s_ExitButton = new nn::repair::LabelButton(
            "EXIT",
            [&]{ExitMouseDownHandler();});
    s_ExitButton->pos(1030, 640);
    *pGlvRootView << s_ExitButton;

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    window.setGLV(*pGlvRootView);
    glv::Application::run();
}

} // namespace

#if !defined(NN_REPAIR_DEVELOP_MODE)
    extern "C" void nninitInitializeSdkModule()
    {
        nn::fs::InitializeWithMultiSessionForTargetTool();
    }

    extern "C" void nninitFinalizeSdkModule()
    {
    }

    extern "C" void nndiagStartup()
    {
    }
#endif

extern "C" void nnMain()
{
    SetupPeripherals();

    #if !defined(NN_REPAIR_DEVELOP_MODE)
      nn::manu::InitializeUfio();
    #endif

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    nn::os::InitializeMutex(&s_Mutex, false, 0);

    // バックアップ処理スレッド
    NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_ProcessingThreadStack[0x10000];
    auto result = nn::os::CreateThread(
            &s_ProcessingThreadType, ProcessingThread, nullptr, s_ProcessingThreadStack,
            sizeof(s_ProcessingThreadStack), nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window);
}

