﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/btm/btm.h>
#include <nn/btm/system/btm_SystemApi.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <sstream>
#include <nn/repair/repair_LabelButton.h>
#include <nn/repair/repair_ShutdownButton.h>
#include <nn/repair/repair_StreamTextView.h>

static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;

static glv::GLV           s_GlvRootView;
static nn::repair::StreamTextView*    s_pTextView;

static void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

static void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

static void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

static void SetupPeripherals() NN_NOEXCEPT
{
    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}

void StartMouseDownHandler() NN_NOEXCEPT
{
    // 1 回だけ実行を認める
    static bool s_IsFirstTry = true;
    if (!s_IsFirstTry)
    {
        return;
    }
    s_IsFirstTry = false;

    s_pTextView->AppendValue("Clear pairing database\n");

    // ペアリング情報削除の動作確認のためにペアリング済みデバイスの数を取得しておきます
    auto count = nn::btm::system::GetPairedGamepadCount();
    std::ostringstream oss;
    oss << count << "devices paired\n";
    s_pTextView->AppendValue(oss.str().c_str());

    nn::btm::system::ClearGamepadPairingDatabase();

    // 処理の完了を待ちます
    for (int i = 0; i < 20; ++i)
    {
        count = nn::btm::system::GetPairedGamepadCount();
        if (count == 0)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    if (count == 0)
    {
        s_pTextView->AppendValue("[[ SUCCESS ]]\n");
    }
    else
    {
        s_pTextView->AppendValue("[[ FAILED ]]\n");
    }
}

void Main( glv::Window& window ) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    // ログビュー表示
    s_pTextView = new nn::repair::StreamTextView(glv::Rect(1024, 512), 32.f);
    s_pTextView->pos(120, 100);
    *pGlvRootView << s_pTextView;

    // ツール情報ラベル表示
    auto titleLabel = new glv::Label("", false);
    nn::repair::GetToolInformationLabel(titleLabel, "Repair Pairing Cutter Tool", 1, 0);
    *pGlvRootView << titleLabel;

    // 開始ボタン表示
    auto startButton = new nn::repair::LabelButton(
            "START",
            [&]{StartMouseDownHandler();});
    startButton->pos(120, 640);
    *pGlvRootView << startButton;

    // シャットダウンボタン(FW 3.0.0 以降で有効)
    glv::Button* shutdownButton = new nn::repair::ShutdownButton();
    shutdownButton->pos(800,640);
    *pGlvRootView << shutdownButton;

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    window.setGLV(*pGlvRootView);
    glv::Application::run();
}

extern "C" void nnMain()
{
    SetupPeripherals();
    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window);
}
