﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cmath>
#include <nn/nn_Log.h>
#include <nn/tc/tc.h>
#include <nn/tc/tc_FanControl.h>
#include "CpuFan.h"
#include "Gpio.h"

namespace ptd { namespace lib {

    CpuFan::CpuFan() : m_gpio(new Gpio())
    {
        nn::tc::Initialize();
        m_isFanControlEnabled = true;
        nn::tc::IsFanControlEnabled(&m_isFanControlEnabled); // 温度制御モジュールによるファンの制御の有効・無効
        if (m_isFanControlEnabled) nn::tc::DisableFanControl(); // 温度制御モジュールによるファンの制御を無効にする

        nn::fan::Initialize();
        nn::fan::OpenController(&m_ctrl, nn::fan::FanName_Cpu);
    }
    CpuFan::~CpuFan()
    {
        nn::fan::CloseController(&m_ctrl);
        nn::fan::Finalize();

        if (m_isFanControlEnabled) nn::tc::EnableFanControl(); // 温度制御モジュールによるファンの制御を有効に戻す
        nn::tc::Finalize();
    }

    bool CpuFan::IsRotated() NN_NOEXCEPT
    {
        if (m_gpio->GetSamplingTimeCpuFan() < 0)
        {
            return false;
        }
        else
        {
            return true;
        }
    }

    void CpuFan::RoundingSpeedLevel(int* plevel)
    {
        if (*plevel < 0)
        {
            *plevel = 0;
        }
        else if (*plevel >= MAX_SPEED)
        {
            *plevel = static_cast<nn::fan::RotationSpeedLevel>(MAX_SPEED);
        }

        NN_LOG("cpu fan rotation speed had set level %d.\n", *plevel);
    }

    void CpuFan::SetSpeed(int level) NN_NOEXCEPT
    {
        RoundingSpeedLevel(&level);
        nn::fan::SetRotationSpeedLevel(&m_ctrl, static_cast<nn::fan::RotationSpeedLevel>(level));
    }

    void CpuFan::SetProvisionalSystemSpeed() NN_NOEXCEPT
    {
        CpuFan::SetSpeed(static_cast<int>(CpuFan::PROVISIONAL_SYSTEM_SPEED));
    }

    void CpuFan::SetMaxSpeed() NN_NOEXCEPT
    {
        CpuFan::SetSpeed(static_cast<int>(CpuFan::MAX_SPEED));
    }

    int CpuFan::GetSpeed() NN_NOEXCEPT
    {
        int level = static_cast<int>(nn::fan::GetRotationSpeedLevel(&m_ctrl));
        NN_LOG("cpu fan rotation speed is level %d.\n", level);

        return level;
    }

    int CpuFan::GetRpm() NN_NOEXCEPT
    {
        const int SAMPLING_PULSE_NUM = 10;
        int64_t span = m_gpio->GetSamplingTimeCpuFan(SAMPLING_PULSE_NUM);
        if (span < 0) return 0;
        int rpm = static_cast<int>(0.5 * 60 * pow(10, 6) * SAMPLING_PULSE_NUM / span);
        //NN_LOG("[%s] %drpm (%dpls, %lldus)\n", __FUNCTION__, rpm, SAMPLING_PULSE_NUM, span);

        return rpm;
    }
}}
