﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <cstring>
#include <list>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_Bis.h>
#include <nn/os.h>

#include <nn/ncm/ncm_SubmissionPackageInstallTask.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ApplyDeltaTask.h>

#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_ApplicationEntityApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_ApplicationVerificationApi.h>

#include "../../../../Eris/Sources/TargetTools/DevMenuCommand/DevMenuCommand_Common.h"
#include "../../../../Eris/Sources/TargetTools/DevMenuCommand/DevMenuCommand_InstallUtil.h"

#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_Manufacture.h>
#include <nn/psm/psm_SystemProcessApi.h>

#include "PreinstallAppWriter_View.h"
#include "PreinstallAppWriter_Status.h"
#include "PreinstallAppWriter_AudioOut.h"
#include "PreinstallAppWriter_Progress.h"
#include "PreinstallAppWriter_CommandLineOption.h"

namespace {

const size_t CacheBufferSize = 1 * 1024 * 1024;
uint8_t g_CacheBuffer[CacheBufferSize];

const size_t           ThreadStackSize = 8192 * 10;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ ThreadStackSize ];
nn::os::ThreadType  g_Thread;
NN_OS_ALIGNAS_THREAD_STACK char  g_ChargerThreadStack[ ThreadStackSize ];
nn::os::ThreadType  g_ChargerThread;
NN_OS_ALIGNAS_THREAD_STACK char  g_ProgressThreadStack[ ThreadStackSize ];
nn::os::ThreadType  g_ProgressThread;

nn::Result g_Result;
PreinstallAppWriter::InstallStatus g_InstallStatus = PreinstallAppWriter::Prepare;
std::string g_PreinstallVersion;
std::string g_ErrorMessage;
bool g_WaitChargingEnabled = true;
PreinstallAppWriter::ProgressInfo g_ProgressInfo;
static const char* const g_IgnoreChargerOption = "--ignoreCharger"; // SDEVで確認するとき
static const char* const g_AutoExitOption = "--autoExit"; // 自動テスト用

}

namespace PreinstallAppWriter {

uint8_t GetRepairSetting()
{
    uint8_t repairSetting;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("rom", g_CacheBuffer, sizeof(g_CacheBuffer)));
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount( "rom" ); };

    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile( &fileHandle, "rom:/RepairSetting.bin", nn::fs::OpenMode_Read ));
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile( fileHandle ); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, &repairSetting, sizeof(repairSetting)));

    NN_LOG("repairSetting = %d\n", repairSetting);

    return repairSetting;
}

bool IsForRepair()
{
    return GetRepairSetting() != 0;
}

bool IsNspuFile(const char* path) NN_NOEXCEPT
{
    nn::util::string_view view(path);

    auto pos = view.find_last_of(".");

    // 拡張子がないファイル
    if ( pos == nn::util::string_view::npos )
    {
        return false;
    }
    // 拡張子が .WXYZ の形でない
    if ( view.size() - pos != 5 )
    {
        return false;
    }

    // 拡張子を取り出し、比較
    nn::util::string_view ext = view.substr(pos + 1, view.size() - pos );
    if ( ext != "nspu" )
    {
        return false;
    }

    return true;
}

nn::Result CalculateContentsSize(int64_t* outCalculatedSize, bool* outValue, const char* nspPath, nn::ncm::StorageId storage)
{
    NN_LOG("Preparing to CalculateContentsSize %s...\n", nspPath);
    nn::fs::FileHandle file;
    NN_RESULT_DO(nn::fs::OpenFile(&file, nspPath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };

    const size_t InstallBufferSize = 16 * 1024 * 1024;
    std::unique_ptr<char[]> buffer(new char[InstallBufferSize]);
    nn::ncm::SubmissionPackageInstallTask task;
    NN_RESULT_DO(task.Initialize(file, storage, buffer.get(), InstallBufferSize));

    NN_RESULT_DO(task.Prepare());
    bool needsCleanup = true;
    NN_UTIL_SCOPE_EXIT
    {
        if (needsCleanup == true)
        {
            task.Cleanup();
        }
    };

    const int MaxContentMetaCount = 2048;
    std::unique_ptr<nn::ncm::StorageContentMetaKey[]> keys(new nn::ncm::StorageContentMetaKey[MaxContentMetaCount]);
    nn::ncm::ApplicationContentMetaKey appkey;

    int contentMetaCount;
    NN_RESULT_DO(task.ListContentMetaKey(&contentMetaCount, keys.get(), MaxContentMetaCount, 0));
    int outAppMetaKeyCount;
    NN_RESULT_DO(task.ListApplicationContentMetaKey(&outAppMetaKeyCount, &appkey, 1, 0));

    if (outAppMetaKeyCount == 0)
    {
        NN_LOG("This file doesn't contain application contents.\n");
        *outValue = false;
        NN_RESULT_SUCCESS;
    }

    for ( int i = 0; i < contentMetaCount; ++i )
    {
        auto key = keys[i];
        NN_RESULT_DO( task.CalculateContentsSize( outCalculatedSize, key.key, key.storageId ) );
        NN_LOG( "calculatedSize = %lld\n", *outCalculatedSize );
    }

    // 必要な情報は取得したので、いったん cleanup
    NN_RESULT_DO(task.Cleanup());
    NN_RESULT_SUCCESS;
}

nn::Result CalculateInstallSize(int64_t* outSize, int* outCount,const char* nspuDirectoryPath, nn::ncm::StorageId storage)
{
    nn::fs::DirectoryHandle dir;
    NN_RESULT_DO(nn::fs::OpenDirectory(&dir, nspuDirectoryPath, nn::fs::OpenDirectoryMode::OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(dir); };

    int targetCount = 0;
    int64_t totalInstallSize = 0;
    while ( NN_STATIC_CONDITION( true ) )
    {
        int64_t count;
        nn::fs::DirectoryEntry entry;
        NN_RESULT_DO( nn::fs::ReadDirectory( &count, &entry, dir, 1 ) );
        if ( count == 0 )
        {
            break;
        }

        auto pathString = std::string( nspuDirectoryPath ) + "/" + entry.name;
        auto path = pathString.c_str();
        if ( entry.directoryEntryType != nn::fs::DirectoryEntryType_File )
        {
            NN_LOG( "Skip %s since it is not file.\n", path );
            continue;
        }

        if ( !( IsNspuFile( path ) || devmenuUtil::IsNspFile( path ) ) )
        {
            NN_LOG( "Skip %s since it is not .nspu file or .nsp file.\n", path );
            continue;
        }

        if ( entry.fileSize <= 0 )
        {
            NN_LOG( "Skip %s since its file size <= 0.\n", path );
            continue;
        }

        targetCount++;
        int64_t calculatedSize;
        bool outValue;
        NN_RESULT_DO(CalculateContentsSize(&calculatedSize, &outValue, path, storage));
        totalInstallSize += calculatedSize;
    }

    *outCount = targetCount;
    *outSize = totalInstallSize;
    NN_RESULT_SUCCESS;
}

nn::Result InstallFromRomFs(bool* outValue, std::list<nn::ncm::ApplicationId>& outIdList, nn::ncm::ContentMetaType type)
{
    auto storage = nn::ncm::StorageId::BuildInUser;
    auto onlyPush = false;
    auto nspuDirectoryPath = "rom:/";

    NN_RESULT_DO(
        nn::fs::MountRom("rom", g_CacheBuffer, sizeof(g_CacheBuffer)));
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount( "rom" ); };

    NN_RESULT_DO( nn::ns::DeleteRedundantApplicationEntity() );

    bool forceUpdate = true;
    bool skipInvalidateCache = false;
    int64_t detachTime = 0;

    {
        nn::fs::DirectoryEntryType entryType;
        NN_RESULT_DO(nn::fs::GetEntryType(&entryType, nspuDirectoryPath));
        if (entryType != nn::fs::DirectoryEntryType_Directory)
        {
            NN_LOG("%s is not directory path. --all option must used with directory.\n", nspuDirectoryPath);
            *outValue = false;
            NN_RESULT_SUCCESS;
        }

        int totalInstallCount;
        int64_t totalInstallSize;
        NN_RESULT_DO(CalculateInstallSize(&totalInstallSize, &totalInstallCount, nspuDirectoryPath, storage));
        NN_LOG("[PreinstallAppWriter] totalInstallCount = %d, totalInstallSize = %lld\n", totalInstallCount, totalInstallSize);
        g_ProgressInfo.SetInstallTotalCount(totalInstallCount);
        g_ProgressInfo.SetInstallTotalSize(totalInstallSize);

        nn::fs::DirectoryHandle dir;
        NN_RESULT_DO(nn::fs::OpenDirectory(&dir, nspuDirectoryPath, nn::fs::OpenDirectoryMode::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(dir); };

        while (NN_STATIC_CONDITION(true))
        {
            int64_t count;
            nn::fs::DirectoryEntry entry;
            NN_RESULT_DO(nn::fs::ReadDirectory(&count, &entry, dir, 1));
            if (count == 0)
            {
                break;
            }

            auto pathString = std::string(nspuDirectoryPath) + "/" + entry.name;
            auto path = pathString.c_str();
            if (entry.directoryEntryType != nn::fs::DirectoryEntryType_File)
            {
                NN_LOG("Skip %s since it is not file.\n", path);
                continue;
            }

            if (!(IsNspuFile(path) || devmenuUtil::IsNspFile(path)))
            {
                NN_LOG("Skip %s since it is not .nspu file or .nsp file.\n", path);
                continue;
            }

            if(entry.fileSize <= 0)
            {
                NN_LOG("Skip %s since its file size <= 0.\n", path);
                continue;
            }

            int readCountOne;
            nn::ncm::ApplicationId outId;
            nn::ncm::StorageContentMetaKey outInstalledKey;
            g_InstallStatus = Installing;
            nn::ncm::SubmissionPackageInstallTask task;
            g_ProgressInfo.SetInstallTask(&task);
            NN_UTIL_SCOPE_EXIT
            {
                g_ProgressInfo.SetInstallStatus(nullptr, g_ProgressInfo.GetInstalledCount() + 1, g_ProgressInfo.GetInstalledSize() + task.GetProgress().installedSize, 0);
            };
            NN_RESULT_DO(devmenuUtil::InstallCommonOne(
                outValue, &outInstalledKey, &readCountOne, &outId, 1, path, storage, type, onlyPush,
                forceUpdate, skipInvalidateCache, detachTime, &task, IsNspuFile(path)));
            if (!*outValue)
            {
                NN_LOG("Failed to install %s.\n", path);
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_DO( nn::ns::SetPreInstalledApplication( outId ) );
            outIdList.push_back(outId);
        }
    }

    *outValue = true;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

nn::Result CalculateVerifySize(int64_t* pOutValue, const std::list<nn::ncm::ApplicationId>& ids)
{
    int64_t totalVerifySize = 0;
    for(auto i : ids)
    {
        const size_t BufSize = 16 * 1024 * 1024;
        const int Alignment = 4 * 1024;
        void* buf = aligned_alloc( Alignment, BufSize );
        NN_UTIL_SCOPE_EXIT
        {
            free( buf );
        };
        nn::ns::AsyncVerifyApplicationResult asyncResult;
        NN_RESULT_DO( nn::ns::RequestVerifyApplication( &asyncResult, i, buf, BufSize ) );
        for(;;)
        {
            nn::ns::AsyncVerifyApplicationProgress progress;
            asyncResult.GetProgress(&progress);
            if(progress.totalSize > 0)
            {
                NN_LOG("AsyncVerifyApplicationProgress.totalSize = %lld\n", progress.totalSize);
                totalVerifySize += progress.totalSize;
                break;
            }
        }
        asyncResult.Cancel();
    }

    *pOutValue = totalVerifySize;

    NN_RESULT_SUCCESS;
}

nn::Result Verify(const std::list<nn::ncm::ApplicationId>& ids)
{
    g_InstallStatus = Verifying;
    const size_t BufSize = 16 * 1024 * 1024;
    const int Alignment = 4 * 1024;
    auto start = std::chrono::system_clock::now();

    int64_t totalVerifySize;
    NN_RESULT_DO(CalculateVerifySize(&totalVerifySize, ids));
    g_ProgressInfo.SetVerifyTotalCount(ids.size());
    g_ProgressInfo.SetVerifyTotalSize(totalVerifySize);

    for(auto i : ids)
    {
        void* buf = aligned_alloc( Alignment, BufSize );
        NN_UTIL_SCOPE_EXIT
        {
            free( buf );
        };
        nn::ns::AsyncVerifyApplicationResult asyncResult;
        NN_RESULT_DO( nn::ns::RequestVerifyApplication( &asyncResult, i, buf, BufSize ) );
        nn::ns::AsyncVerifyApplicationProgress progress;
        g_ProgressInfo.SetAsyncProgress(&progress);
        for(;;)
        {
            asyncResult.GetProgress(&progress);
            if(asyncResult.TryWait())
            {
                NN_RESULT_DO( asyncResult.Get() );
                g_ProgressInfo.SetVerifyStatus(nullptr, g_ProgressInfo.GetVerifiedCount() + 1, g_ProgressInfo.GetVerifiedSize() + progress.totalSize, 0);
                break;
            }
        }
    }

    auto end = std::chrono::system_clock::now();
    auto elapsed = std::chrono::duration_cast<std::chrono::seconds>( end - start ).count();
    NN_LOG( "Verify elapsed = %d sec\n", elapsed );

    NN_RESULT_SUCCESS;
}

nn::Result ReadPreinstallVersionInRom(char* pOutValue, size_t bufSize)
{
    NN_RESULT_DO(
        nn::fs::MountRom("rom", g_CacheBuffer, sizeof(g_CacheBuffer)));
    NN_UTIL_SCOPE_EXIT { nn::fs::Unmount("rom"); };

    nn::fs::FileHandle fileHandle;
    NN_RESULT_DO(nn::fs::OpenFile( &fileHandle, "rom:/PreinstallVersion.txt", nn::fs::OpenMode_Read ));
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile( fileHandle ); };

    NN_RESULT_DO(nn::fs::ReadFile(fileHandle, 0, pOutValue, bufSize));

    NN_RESULT_SUCCESS;
}

nn::Result ReadWrittenPreinstallVersion(char* pOutValue, size_t bufSize)
{
    NN_RESULT_DO(nn::fs::MountBis(nn::fs::BisPartitionId::CalibrationFile, NULL));

    std::string mountName = nn::fs::GetBisMountName(nn::fs::BisPartitionId::CalibrationFile);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(mountName.c_str());
    };
    NN_LOG("name = %s\n", mountName.c_str());

    std::string fileName = ":/ptd/log/PreinstallVer_Finalize.log";
    std::string filePath = mountName + fileName;

    nn::fs::FileHandle fileHandle;
    NN_RESULT_TRY(nn::fs::OpenFile( &fileHandle, filePath.c_str(), nn::fs::OpenMode_Read ))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            g_ErrorMessage = "Written Preinstall Version not found.";
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile( fileHandle ); };

    NN_RESULT_DO(nn::fs::ReadFile(fileHandle, 0, pOutValue, bufSize));

    NN_RESULT_SUCCESS;
}

nn::Result WritePreinstallVersion(const char* versionInfo, size_t versionInfoSize)
{
    NN_RESULT_DO(nn::fs::MountBis(nn::fs::BisPartitionId::CalibrationFile, NULL));

    std::string mountName = nn::fs::GetBisMountName(nn::fs::BisPartitionId::CalibrationFile);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(mountName.c_str());
    };
    NN_LOG("name = %s\n", mountName.c_str());

    std::string fileName = ":/ptd/log/PreinstallVer_Writer.log";
    std::string filePath = mountName + fileName;

    NN_RESULT_TRY(nn::fs::DeleteFile(filePath.c_str()))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound) {}
    NN_RESULT_END_TRY;

    NN_RESULT_DO(nn::fs::CreateFile( filePath.c_str(), versionInfoSize));

    nn::fs::FileHandle fileHandle;
    NN_RESULT_DO(nn::fs::OpenFile( &fileHandle, filePath.c_str(), nn::fs::OpenMode_Write ));
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile( fileHandle ); };

    NN_RESULT_DO(
        nn::fs::WriteFile( fileHandle, 0, versionInfo, versionInfoSize,
                           nn::fs::WriteOption::MakeValue( nn::fs::WriteOptionFlag_Flush ) ) );

    NN_RESULT_SUCCESS;
}

nn::Result InstallApplication(bool* pOutValue)
{
    nn::ncm::Initialize();
    nn::es::Initialize();
    nn::ns::Initialize();

    std::list<nn::ncm::ApplicationId> idList;
    NN_RESULT_DO(InstallFromRomFs(pOutValue, idList, nn::ncm::ContentMetaType::Unknown));
    NN_LOG("idList.size = %d\n", idList.size() );
    NN_RESULT_DO(Verify(idList));

    NN_RESULT_SUCCESS;
}

nn::Result Install(bool* pOutValue)
{
    char writtenVersionInfo[3]; // [A-Z0-9]{3}
    char versionInfoInRom[3]; // [A-Z0-9]{3}
    static_assert(sizeof(writtenVersionInfo) == sizeof(versionInfoInRom), "");
    NN_RESULT_DO(ReadPreinstallVersionInRom(versionInfoInRom, sizeof(versionInfoInRom)));
    g_PreinstallVersion.assign(versionInfoInRom, sizeof(versionInfoInRom));
    if(!IsForRepair())
    {
        NN_RESULT_DO(ReadWrittenPreinstallVersion(writtenVersionInfo, sizeof(writtenVersionInfo)));
        if (std::memcmp(writtenVersionInfo, versionInfoInRom, sizeof(writtenVersionInfo)) != 0)
        {
            *pOutValue = false;
            char message[256];
            auto writtenSize(std::string(writtenVersionInfo, sizeof(writtenVersionInfo)));
            auto toWriteSize(std::string(versionInfoInRom, sizeof(versionInfoInRom)));
            nn::util::SNPrintf(message, sizeof(message),
                               "Preinstall version not matched. written = %s, to write = %s",
                               writtenSize.c_str(), toWriteSize.c_str());
            g_ErrorMessage = message;
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_DO(InstallApplication(pOutValue));
    if(!IsForRepair())
    {
        NN_RESULT_DO(WritePreinstallVersion(versionInfoInRom, sizeof(versionInfoInRom)));
    }
    NN_RESULT_SUCCESS;
}

void PlaySound(bool isOk)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("rom", g_CacheBuffer, sizeof(g_CacheBuffer)));
    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount( "rom" ); };

    std::string filePath = isOk ? "rom:/ok.wav" : "rom:/ng.wav";

    AudioOut aOut;
    NN_ABORT_UNLESS(aOut.Initialize());

    NN_ABORT_UNLESS(aOut.StartPlayWavFile(filePath.c_str()));
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2000));
    aOut.StopPlaySound();
}

void Charger(void* p)
{
    nn::psm::Initialize();
    for(;;)
    {
        auto isChargerEnoughPower =
             (nn::psm::GetChargerType() == nn::psm::ChargerType_EnoughPower) ? true : false;
        if(HasOption(g_IgnoreChargerOption))
        {
            NN_LOG("%s option Enabled\n", g_IgnoreChargerOption);
            isChargerEnoughPower = true;
        }
        if(IsForRepair())
        {
            NN_LOG("IsForRepair == true\n");
            isChargerEnoughPower = true;
        }
        if(isChargerEnoughPower)
        {
            g_WaitChargingEnabled = false;
            break;
        }
        NN_LOG("ChargerEnoughPower = %d\n", isChargerEnoughPower);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    for(;;)
    {
        auto level = nn::psm::GetBatteryChargePercentage();
        if(level > 50)
        {
            nn::psm::DisableBatteryCharging();
        }
        else
        {
            nn::psm::EnableBatteryCharging();
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
}

void WaitChargingEnabled()
{
    while(g_WaitChargingEnabled)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
}

nn::Result GetInstallResult()
{
    return g_Result;
}

InstallStatus GetInstallStatus()
{
    return g_InstallStatus;
}

std::string GetPreinstallVersion()
{
    return g_PreinstallVersion;
}

std::string GetErrorMessage()
{
    return g_ErrorMessage;
}

}

extern "C" void nnMain()
{
    nn::fs::SetEnabledAutoAbort(false);
    auto result = nn::os::CreateThread( &g_Thread, PreinstallAppWriter::View, &g_ProgressInfo, g_ThreadStack,
                                        ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ABORT_UNLESS( result.IsSuccess(), "Cannot create g_Thread." );
    nn::os::StartThread( &g_Thread );
    result = nn::os::CreateThread( &g_ChargerThread, PreinstallAppWriter::Charger, nullptr, g_ChargerThreadStack,
                                        ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ABORT_UNLESS( result.IsSuccess(), "Cannot create g_ChargerThread." );
    nn::os::StartThread( &g_ChargerThread );
    result = nn::os::CreateThread( &g_ProgressThread, PreinstallAppWriter::ProgressInfo::Func, &g_ProgressInfo, g_ProgressThreadStack,
                                        ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ABORT_UNLESS( result.IsSuccess(), "Cannot create g_ProgressThread." );
    nn::os::StartThread( &g_ProgressThread );
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::WaitThread( &g_Thread );
        nn::os::DestroyThread( &g_Thread );
        nn::os::WaitThread( &g_ChargerThread );
        nn::os::DestroyThread( &g_ChargerThread );
        nn::os::WaitThread( &g_ProgressThread );
        nn::os::DestroyThread( &g_ProgressThread );
    };

    PreinstallAppWriter::WaitChargingEnabled();
    bool outValue = true;
    result = PreinstallAppWriter::Install(&outValue);
    if (result.IsSuccess() && outValue)
    {
        NN_LOG("[SUCCESS]\n");
        g_InstallStatus = PreinstallAppWriter::Done;
        PreinstallAppWriter::PlaySound(true);
    }
    else
    {
        g_Result = result;
        NN_LOG("[FAILURE]: result 0x%08x\n", result.GetInnerValueForDebug());
        g_InstallStatus = PreinstallAppWriter::Error;
        PreinstallAppWriter::PlaySound(false);
    }

    if (PreinstallAppWriter::HasOption(g_AutoExitOption))
    {
        NN_LOG("%s option Enabled\n", g_AutoExitOption);
        if(g_InstallStatus == PreinstallAppWriter::Done)
        {
            std::quick_exit(0);
        }
        else
        {
            NN_ABORT();
        }
    }

    for(;;)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
}
