﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>

#include "../graphics/NfpDebugTool_Renderer.h"
#include "../npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_DisplayObject.h"

namespace nfpdebug { namespace ui {

extern const graphics::Color DefaultWindowFrameColor;
extern const graphics::Color DefaultWindowBackColor;
extern const graphics::Color DefaultCaptionColor;
extern const graphics::Color DefaultActiveCaptionBackColor;
extern const graphics::Color DefaultInactiveCaptionBackColor;

/**
 * @brief   ウィンドウの操作に対するコールバック関数型です。
 */
typedef void (*WindowHandlerType)(const nn::hid::NpadIdType&, uintptr_t);

/**
 * @brief   ウィンドウの操作に対するコールバック関数型 (インデックス付き) です。
 */
typedef void (*WindowHandlerTypeWithIndex)(const nn::hid::NpadIdType&, uintptr_t, int index);

/**
 * @brief   簡易ウィンドウ表示を行うクラスです。
 */
class Window : public DisplayObject
{
public:
    Window() NN_NOEXCEPT :
        m_IsActive(false),
        m_IsShowing(false),
        m_IsHiding(false),
        m_IsCursorVisible(false),
        m_Width(360),
        m_Height(180),
        m_DefaultLineHeight(1),
        m_CancelHandler(nullptr),
        m_Caption(nullptr),
        m_FrameColor(),
        m_BackColor(),
        m_ActiveCaptionBackColor(),
        m_InactiveCaptionBackColor(),
        m_CaptionColor(),
        m_CursorFadeDuration(0),
        m_CursorColor(graphics::Colors::Turquoise)
    {
        SetOpacity(0);
    }

    /**
     * @brief   初期化します。
     */
    void Initialize() NN_NOEXCEPT;

    /**
     * @brief   表示します。
     */
    virtual void Show() NN_NOEXCEPT NN_OVERRIDE
    {
        DisplayObject::Show();
        m_IsShowing = true;
        m_IsHiding  = false;
    }

    /**
     * @brief   非表示にします。
     */
    virtual void Hide() NN_NOEXCEPT NN_OVERRIDE
    {
        //DisplayObject::Hide();
        m_IsShowing = false;
        m_IsHiding  = true;
        Deactivate();
    }

    /**
     * @brief   アクティブ状態かどうか取得します。
     */
    bool IsActive() const NN_NOEXCEPT
    {
        return m_IsActive;
    }

    /**
     * @brief   アクティブ状態にします。
     */
    void Activate() NN_NOEXCEPT
    {
        m_IsActive = true;
    }

    /**
     * @brief   非アクティブ状態にします。
     */
    void Deactivate() NN_NOEXCEPT
    {
        m_IsActive = false;
    }

    /**
     * @brief   カーソルの表示状態を設定します。
     */
    void SetCursorVisible(bool isVisible) NN_NOEXCEPT
    {
        m_IsCursorVisible = isVisible;
    }

    /**
     * @brief   カーソルの表示状態を取得します。
     */
    bool IsCursorVisible() const NN_NOEXCEPT
    {
        return m_IsCursorVisible;
    }

    /**
     * @brief   描画サイズを設定します。
     */
    void SetSize(float width, float height) NN_NOEXCEPT
    {
        m_Width  = width;
        m_Height = height;
    }

    /**
     * @brief   描画サイズを取得します。
     */
    void GetSize(float* pOutWidth, float* pOutHeight) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutWidth);
        NN_ASSERT_NOT_NULL(pOutHeight);

        *pOutWidth  = m_Width;
        *pOutHeight = m_Height;
    }

    /**
     * @brief   描画幅を取得します。
     */
    float GetWidth() const NN_NOEXCEPT
    {
        return m_Width;
    }

    /**
     * @brief   描画高を取得します。
     */
    float GetHeight() const NN_NOEXCEPT
    {
        return m_Height;
    }

    /**
     * @brief   キャンセル時のコールバック関数を登録します。
     */
    void SetCancelHandler(WindowHandlerType handler) NN_NOEXCEPT
    {
        m_CancelHandler = handler;
    }

    /**
     * @brief   ウィンドウのタイトル文字列を設定します。
     */
    void SetCaption(const char* caption) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(caption);

        m_Caption = caption;
    }

    /**
     * @brief   ウィンドウタイトルの色を設定します。
     */
    void SetCaptionColor(
        const graphics::Color& activeBackColor,
        const graphics::Color& inactiveBackColor,
        const graphics::Color& textColor) NN_NOEXCEPT
    {
        m_ActiveCaptionBackColor   = activeBackColor;
        m_InactiveCaptionBackColor = inactiveBackColor;
        m_CaptionColor             = textColor;
    }

    /**
     * @brief   描画に使用するレンダラーを指定します。
     */
    virtual void SetRenderer(graphics::Renderer* pRenderer) NN_NOEXCEPT NN_OVERRIDE
    {
        DisplayObject::SetRenderer(pRenderer);

        if (pRenderer == nullptr)
        {
            return;
        }

        nn::util::Float2 prevScale;
        pRenderer->GetTextScale(&prevScale);

        pRenderer->SetTextScale(1.2f, 1.2f);
        m_DefaultLineHeight = pRenderer->GetTextLineHeight();
        pRenderer->SetTextScale(prevScale.x, prevScale.y);
    }

    /**
     * @brief   入力処理を更新します。
     */
    virtual void UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT;

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

protected:
    /**
     * @brief   固定幅モード時の標準テキスト幅です。
     */
    static const float FixedModeTextWidth;

protected:
    /**
     * @brief   描画幅を取得します。
     */
    virtual float GetRenderWidth() const NN_NOEXCEPT
    {
        return m_Width;
    }

    /**
     * @brief   描画高を取得します。
     */
    virtual float GetRenderHeight() const NN_NOEXCEPT
    {
        return m_Height;
    }

    virtual float GetDefaultLineHeight() const NN_NOEXCEPT
    {
        return m_DefaultLineHeight;
    }

    /**
     * @brief   ヘッダー領域の高さを取得します。
     */
    float GetHeaderHeight() const NN_NOEXCEPT
    {
        return GetDefaultLineHeight() * 1.3f + 4;
    }

    /**
     * @brief   クライアント領域の矩形を取得します。
     */
    virtual void GetClientRectangle(graphics::Rectangle* pOutRect) NN_NOEXCEPT;

    /**
     * @brief   不透明度を反映した色を取得します。
     */
    graphics::Color GetDisplayColor(const graphics::Color& color) const NN_NOEXCEPT
    {
        return graphics::Colors::GetAlphaColor(color, GetOpacity());
    }

    /**
     * @brief   フェードアニメーションを反映したカーソル色を取得します。
     */
    graphics::Color GetCursorFadeColor() const NN_NOEXCEPT;

    /**
     * @brief   キャンセル時のコールバック関数を取得します。
     */
    WindowHandlerType GetCancelHandler() NN_NOEXCEPT
    {
        return m_CancelHandler;
    }

    /**
     * @brief   カーソルを描画します。
     */
    void DrawCursor(float x, float y, float width, float height, const graphics::Color& color) NN_NOEXCEPT;
    void DrawCursor(const graphics::Rectangle& rect, const graphics::Color& color) NN_NOEXCEPT
    {
        DrawCursor(rect.x, rect.y, rect.width, rect.height, color);
    }

private:
    bool                m_IsActive;                     //!< アクティブ状態
    bool                m_IsShowing;                    //!< 表示処理中
    bool                m_IsHiding;                     //!< 非表示処理中
    bool                m_IsCursorVisible;              //!< カーソルの表示状態
    float               m_Width;                        //!< 幅
    float               m_Height;                       //!< 高さ
    float               m_DefaultLineHeight;            //!< 行の高さのデフォルト値
    WindowHandlerType   m_CancelHandler;                //!< キャンセル時のコールバック
    const char*         m_Caption;                      //!< ヘッダの文字列
    graphics::Color     m_FrameColor;                   //!< ウィンドウ枠の色
    graphics::Color     m_BackColor;                    //!< クライアント領域の背景色
    graphics::Color     m_ActiveCaptionBackColor;       //!< アクティブ時のヘッダ部分の背景色
    graphics::Color     m_InactiveCaptionBackColor;     //!< 非アクティブ時のヘッダ部分の背景色
    graphics::Color     m_CaptionColor;                 //!< ヘッダ部分の文字色
    int                 m_CursorFadeDuration;           //!< カーソルのフェードアニメーション時間
    graphics::Color     m_CursorColor;                  //!< カーソルの色
};

}}  // nfpdebug::ui
