﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>

#include "NfpDebugTool_MessageBox.h"

namespace nfpdebug { namespace ui {

void MessageBox::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    Window::UpdateInput(controller);

    if (!IsActive() || !IsVisible())
    {
        return;
    }

    if (controller.IsTriggered(nn::hid::NpadButton::A::Mask))
    {
        auto handler = GetCancelHandler();
        if (handler != nullptr)
        {
            handler(controller.GetNpadId(), 0);
        }
    }
}

void MessageBox::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    Window::Render();

    if (m_Text == nullptr)
    {
        return;
    }

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    DrawIcon(rect.x + 6, rect.y + 4, 64);

    renderer.SetTextScale(1.2f, 1.2f);
    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    // 描画サイズを算出
    float textHeight = renderer.CalculateTextHeight(m_Text);

    renderer.DrawText(
        rect.x + 72,
        rect.y + (rect.height - textHeight) / 2,
        "%s",
        m_Text);

    renderer.SetTextScale(prevScale.x, prevScale.y);
}

void MessageBox::DrawIcon(float x, float y, float size) NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    renderer.SetTextScale(3.0f, 3.0f);

    auto drawIconImpl = [&](
        float offsetX,
        float offsetY,
        const char* iconFrame,
        const char* iconText,
        const graphics::Color& color)
    {
        NN_ASSERT_NOT_NULL(iconFrame);
        NN_ASSERT_NOT_NULL(iconText);

        renderer.SetTextColor(GetDisplayColor(color));
        renderer.DrawText(x, y, iconFrame);
        renderer.SetTextScale(2.0f, 2.0f);
        renderer.DrawText(x + offsetX, y + offsetY, iconText);
    };

    switch (m_Icon)
    {
    case Icon::None:
        return;
    case Icon::Information:
        drawIconImpl(20, 12, "○", "i", graphics::Colors::DodgerBlue);
        break;
    case Icon::Error:
        drawIconImpl(14, 10, "○", "×", graphics::Colors::Crimson);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}  // nfpdebug::ui
