﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/vi.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/mem.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_Mutex.h>
#include "NfpDebugTool_GraphicSystem.h"

namespace nfpdebug { namespace graphics {

/**
 * @brief   色を表す型です。
 */
typedef nn::util::Color4u8Type Color;

/**
 * @brief   位置情報を表す 2 次元ベクトルです。
 */
typedef nn::util::Float2 Point2D;

/**
 * @brief   サイズ情報を表す 2 次元ベクトルです。
 */
struct Size
{
    union
    {
        float v[2];     //!< 要素配列
        struct
        {
            float width;
            float height;
        };
    };
};

/**
 * @brief   矩形情報を表す 4 次元ベクトルです。
 */
struct Rectangle
{
    union
    {
        float v[4];     //!< 要素配列
        struct
        {
            float x;
            float y;
            float width;
            float height;
        };
    };
};

/**
 * @brief   汎用的な色を定義した静的クラスです。
 */
class Colors
{
public:
    static const Color Transparent;
    static const Color Black;
    static const Color White;
    static const Color Silver;
    static const Color Gray;
    static const Color DimGray;
    static const Color Red;
    static const Color Orange;
    static const Color Gold;
    static const Color Yellow;
    static const Color Green;
    static const Color Turquoise;
    static const Color Cyan;
    static const Color Blue;
    static const Color DodgerBlue;
    static const Color RoyalBlue;
    static const Color Magenta;
    static const Color Crimson;;

    /**
     * @brief   アルファ値に補正をかけた色を取得します。
     */
    static Color GetAlphaColor(const Color& color, uint8_t alpha) NN_NOEXCEPT
    {
        auto newColor = color;
        newColor.v[3] = static_cast<uint8_t>(color.v[3] * alpha / 255);
        return newColor;
    }

private:
    Colors();   // インスタンス生成を禁止
};

/**
 * @brief   基本的な図形、テキスト描画を行うクラスです。
 */
class Renderer
{
    NN_DISALLOW_COPY(Renderer);
    NN_DISALLOW_MOVE(Renderer);

public:
    static Renderer* GetInstance() NN_NOEXCEPT;
    static void Shutdown() NN_NOEXCEPT;

    nn::gfx::util::DebugFontTextWriter* GetTextWriter() NN_NOEXCEPT
    {
        return m_pTextWriter;
    }

    void Render() NN_NOEXCEPT;

    // テキスト
    void SetTextScale(float h, float v) NN_NOEXCEPT;
    void GetTextScale(Point2D* pOutScale) const NN_NOEXCEPT;
    void SetTextColor(const Color& color) NN_NOEXCEPT;
    void GetTextColor(Color* pOutColor) const NN_NOEXCEPT;

    void SetFixedMode(bool isFixed, float width) NN_NOEXCEPT
    {
        m_pTextWriter->SetFixedWidthEnabled(isFixed);
        m_pTextWriter->SetFixedWidth(width);
    }

    void SetFixedMode(bool isFixed) NN_NOEXCEPT
    {
        m_pTextWriter->SetFixedWidthEnabled(isFixed);
    }

    bool IsFixedMode() const NN_NOEXCEPT
    {
        return m_pTextWriter->IsFixedWidthEnabled();
    }

    float GetTextLineHeight() const NN_NOEXCEPT
    {
        return m_FontContext.scaleVertical * 22.0f;
    }

    float CalculateTextWidth(const char* text) NN_NOEXCEPT
    {
        return m_pTextWriter->CalculateStringWidth(text);
    }

    float CalculateTextHeight(const char* text) NN_NOEXCEPT
    {
        return m_pTextWriter->CalculateStringHeight(text);
    }

    void DrawText(float x, float y, const char* format, ...) NN_NOEXCEPT;

    // 図形
    void DrawHorizontalLine(float x, float y, float length, const Color& color, float weight) NN_NOEXCEPT;
    void DrawHorizontalLine(float x, float y, float length, const Color& color) NN_NOEXCEPT
    {
        DrawHorizontalLine(x, y, length, color, 1.0f);
    }

    void DrawVerticalLine(float x, float y, float length, const Color& color, float weight) NN_NOEXCEPT;
    void DrawVerticalLine(float x, float y, float length, const Color& color) NN_NOEXCEPT
    {
        DrawVerticalLine(x, y, length, color, 1.0f);
    }

    void DrawRect(float x, float y, float width, float height, const Color& color, float lineWeight) NN_NOEXCEPT;
    void DrawRect(float x, float y, float width, float height, const Color& color) NN_NOEXCEPT
    {
        DrawRect(x, y, width, height, color, 1.0f);
    }
    void DrawRect(const Rectangle& rect, const Color& color, float lineWeight) NN_NOEXCEPT
    {
        DrawRect(rect.x, rect.y, rect.width, rect.height, color);
    }
    void DrawRect(const Rectangle& rect, const Color& color) NN_NOEXCEPT
    {
        DrawRect(rect, color, 1.0f);
    }

    void FillRect(float x, float y, float width, float height, const Color& color) NN_NOEXCEPT;
    void FillRect(const Rectangle& rect, const Color& color) NN_NOEXCEPT
    {
        FillRect(rect.x, rect.y, rect.width, rect.height, color);
    }

private:
    struct FontContext
    {
        int   displayWidth;
        int   displayHeight;
        float scaleHorizontal;
        float scaleVertical;
        float x;
        float y;
        Color color;
    };

private:
    Renderer() NN_NOEXCEPT :
        m_Mutex(true),
        m_ApplicationHeap(),
        m_pGraphicsSystem(nullptr),
        m_GraphicsSystemStorage(),
        m_pTextWriter(nullptr),
        m_FontContext()
    {}
    ~Renderer() NN_NOEXCEPT { Finalize(); }

    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

private:
    mutable nn::os::Mutex m_Mutex;

    ApplicationHeap m_ApplicationHeap;
    nfpdebug::graphics::GraphicsSystem* m_pGraphicsSystem;
    char m_GraphicsSystemStorage[sizeof(nfpdebug::graphics::GraphicsSystem)];

    nn::gfx::util::DebugFontTextWriter* m_pTextWriter;

    FontContext m_FontContext;
};

}}  // nfpdebug::graphics
