﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VibrationDemo_VibrationKnob.h"

namespace VibrationDemo
{
    VibrationKnob::VibrationKnob()
    {
        SetDefault();
    }

    void VibrationKnob::SetDefault()
    {
        static_cast<BaseItem*>(this)->SetDefault();

        m_IsTouchedItem = false;

        m_MaxValue = 100;
        m_MinValue = 0;
        m_MaterValue = static_cast<float>((m_MinValue + m_MaxValue) / 2);

        m_BorderColor = nn::util::Color4u8(64, 64, 64, 255);
        m_MainColor = Color::Gray;
        m_SubColor = Color::Green;
        m_EffectColor = Color::Cyan;

        m_InputType = InputType::InputType_None;
    }

    void VibrationKnob::Update()
    {
        ++m_FrameCount;

        // 現在の値を退避
        const auto oldValue = m_MaterValue;

        // 何の入力デバイスで操作を行ったか記憶する
        m_InputType = InputType::InputType_Touch;

        // タッチ操作を最優先で扱う
        if (UpdateTouchEvent() == false)
        {   // タッチ操作が行われていない場合コントローラの入力状態を確認
            m_InputType = InputType::InputType_Controller;
            if (UpdateControllerEvent() == false)
            {   // コントローラによる操作が発生していない場合ジャイロを確認
                m_InputType = InputType::InputType_Gyro;
                if (UpdateGyroEvent() == false)
                {   // 対応した入力操作が行われていない場合
                    m_InputType = InputType::InputType_None;
                }
            }
        }

        if (m_State.Test<nns::hidfw::layout::LayoutState::Selected>() && !m_State.Test<nns::hidfw::layout::LayoutState::OnFocus>() && !m_State.Test<nns::hidfw::layout::LayoutState::Hover>())
        {
            m_State.Reset<nns::hidfw::layout::LayoutState::Selected>();
            m_Option.Reset<nns::hidfw::layout::LayoutOption::ThroughChoose>();
        }

        SetVibrationBalance(0.f);

        // メータ値の上限・下限チェック
        m_MaterValue = std::min(static_cast<float>(m_MaxValue), std::max(static_cast<float>(m_MinValue), m_MaterValue));

        m_State.Reset<nns::hidfw::layout::LayoutState::LimitingValue>();
        m_MasterGain = 1.f;

        if (static_cast<int64_t>(std::roundf(m_MaterValue)) >= m_MaxValue)
        {
            m_State.Set< nns::hidfw::layout::LayoutState::LimitingValue>();
            SetVibrationBalance(2.f);
        }
        if (static_cast<int64_t>(std::roundf(m_MaterValue)) <= m_MinValue)
        {
            m_State.Set< nns::hidfw::layout::LayoutState::LimitingValue>();
            SetVibrationBalance(-2.f);
        }
        if(m_MaxValue - m_MinValue < 8)
        {
            m_State.Set<nns::hidfw::layout::LayoutState::LimitingValue>();
            SetVibrationBalance(0.f);
        }

        // 値が更新されたか否か
        m_State.Set< nns::hidfw::layout::LayoutState::UpdateValue>(static_cast<int64_t>(std::roundf(oldValue)) != static_cast<int64_t>(std::roundf(m_MaterValue)));

        // ジャイロ操作が有効な場合は右ジョイコンのみを振動させる
        if (m_InputType == InputType_Gyro) { SetVibrationBalance(1.f); }

        // 振動の送信
        UpdateVibration();
    }

    bool VibrationKnob::UpdateTouchEvent() NN_NOEXCEPT
    {
        const auto radius = std::min(m_Size.x * m_Scale.x / 2.f, m_Size.y * m_Scale.y / 2.f);
        nn::hid::GestureState state[nn::hid::GestureStateCountMax];
        auto count = m_IsTouchedItem ?
            gTouch.GetGestureStates(state, nn::hid::GestureStateCountMax, nn::util::MakeFloat2(0.f, 0.f), nn::util::MakeFloat2(1280.f, 720.f)) :
            gTouch.GetGestureStates(state, nn::hid::GestureStateCountMax, m_Pos, m_Size);
        bool result = count > 0;

        for (int i = 0; i < count; ++i)
        {
            if (state[i].pointCount == 1)
            {
                switch (state[i].GetGestureType())
                {
                case nn::hid::GestureType_Pan:
                {
                    if (m_IsTouchedItem)
                    {
                        auto prevPos = nn::util::MakeFloat2(
                            static_cast<float>(state[i].x - state[i].deltaX),
                            static_cast<float>(state[i].y - state[i].deltaY));

                        auto rad1 = std::atan2f(prevPos.y - (m_Pos.y + radius), prevPos.x - (m_Pos.x + radius));
                        if (rad1 < 0) { rad1 = rad1 + 2 * nn::util::FloatPi; }
                        auto rad2 = std::atan2f(state[i].y - (m_Pos.y + radius), state[i].x - (m_Pos.x + radius));
                        if (rad2 < 0) { rad2 = rad2 + 2 * nn::util::FloatPi; }

                        auto degree = floorf(rad2 * 360 / (2 * nn::util::FloatPi)) - floorf(rad1 * 360 / (2 * nn::util::FloatPi));

                        if (degree > 180.f || degree < -180.f) { degree = 0.f; }

                        m_MaterValue += (degree / 270.f) * static_cast<float>(m_MaxValue - m_MinValue);
                    }
                    break;
                }
                case nn::hid::GestureType_Complete:
                    m_IsTouchedItem = false;
                    break;
                case nn::hid::GestureType_Cancel:
                    m_IsTouchedItem = false;
                    break;
                case nn::hid::GestureType_Touch:
                    m_IsTouchedItem = true;
                    break;
                default:
                    break;
                }
            }
            else if (state[i].pointCount >= 2)
            {
                switch (state[i].GetGestureType())
                {
                case nn::hid::GestureType_Rotate:
                    m_MaterValue += (state[i].rotationAngle / 270.f) * static_cast<float>(m_MaxValue - m_MinValue);
                    break;
                default:
                    break;
                }
            }
            else
            {
                m_IsTouchedItem = false;
            }
        }

        return result;
    }

    bool VibrationKnob::UpdateControllerEvent() NN_NOEXCEPT
    {
        bool result = false;

        if (
            (m_Option.Test<nns::hidfw::layout::LayoutOption::ThroughChoose>() && m_State.Test<nns::hidfw::layout::LayoutState::Hover>()) ||
            (!m_Option.Test<nns::hidfw::layout::LayoutOption::ThroughChoose>() && m_State.Test<nns::hidfw::layout::LayoutState::Selected>())
            )
        {
            if (m_State.Test<nns::hidfw::layout::LayoutState::Selected>())
            {
                m_Option.Set<nns::hidfw::layout::LayoutOption::ThroughChoose>();
            }
            else
            {
                m_Option.Reset<nns::hidfw::layout::LayoutOption::ThroughChoose>();
            }
            for (
                std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
                it != gController.GetConnectedControllerList().end();
                ++it
                )
            {
                // デジタルキーは小刻みな調整に利用し、最優先で処理します
                if ((*it)->IsHold(nn::hid::NpadButton::Left::Mask | nn::hid::NpadButton::Right::Mask))
                {
                    if ((*it)->IsHold(nn::hid::NpadButton::Left::Mask))
                    {
                        if ((*it)->IsRepeat(nn::hid::NpadButton::Left::Mask))
                        {
                            m_MaterValue = floorf(m_MaterValue - 0.5f);
                        }
                    }
                    else
                    {
                        if ((*it)->IsRepeat(nn::hid::NpadButton::Right::Mask))
                        {
                            m_MaterValue = floorf(m_MaterValue + 1.f);
                        }
                    }
                    result = true;
                    break;
                }
                else if ((*it)->IsHold(nn::hid::NpadButton::StickLLeft::Mask | nn::hid::NpadButton::StickRLeft::Mask | nn::hid::NpadButton::StickLRight::Mask | nn::hid::NpadButton::StickRRight::Mask))
                {
                    result = true;

                    if ((*it)->IsHold(nn::hid::NpadButton::StickLLeft::Mask | nn::hid::NpadButton::StickLRight::Mask))
                    {
                        const auto value = static_cast<float>((*it)->GetLeftAnalogStickState().x) / static_cast<float>(nn::hid::AnalogStickMax);
                        m_MaterValue += (static_cast<float>(m_MaxValue - m_MinValue) / 60.f) * value * value * value;
                        break;
                    }
                    else if ((*it)->IsHold(nn::hid::NpadButton::StickRLeft::Mask | nn::hid::NpadButton::StickRRight::Mask))
                    {
                        const auto value = static_cast<float>((*it)->GetRightAnalogStickState().x) / static_cast<float>(nn::hid::AnalogStickMax);
                        m_MaterValue += (static_cast<float>(m_MaxValue - m_MinValue) / 60.f) * value * value * value;
                        break;
                    }
                }
            }
        }
        return result;
    }

    bool VibrationKnob::UpdateGyroEvent() NN_NOEXCEPT
    {
        if (
            (m_Option.Test<nns::hidfw::layout::LayoutOption::ThroughChoose>() && m_State.Test<nns::hidfw::layout::LayoutState::Hover>()) ||
            (!m_Option.Test<nns::hidfw::layout::LayoutOption::ThroughChoose>() && m_State.Test<nns::hidfw::layout::LayoutState::Selected>())
            )
        {
            for (
                std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
                it != gController.GetConnectedControllerList().end();
                ++it
                )
            {
                if ((*it)->GetMaxHoldCount(nn::hid::NpadButton::A::Mask, nullptr) > 5)
                {
                    nn::hid::SixAxisSensorState state[nn::hid::SixAxisSensorStateCountMax];
                    auto count = gController.GetController(gControllerSequenceManager.GetMasterControllerId())->GetSixAxisSensorStates(nn::hid::NpadJoyDeviceType_Right, state, nn::hid::SixAxisSensorStateCountMax);
                    if (count > 0)
                    {
                        auto rotate = state[0].angle.y - state[count - 1].angle.y;
                        m_MaterValue += (rotate / (180.f / 100.f)) * static_cast<float>(m_MaxValue - m_MinValue);
                        return true;
                    }
                }
            }
        }
        return false;
    }

    void VibrationKnob::UpdateVibration() NN_NOEXCEPT
    {
        if (m_State.Test< nns::hidfw::layout::LayoutState::UpdateValue>())
        {
            int key = m_State.Test< nns::hidfw::layout::LayoutState::LimitingValue>() ? nns::hidfw::layout::LayoutState::LimitingValue::Index : nns::hidfw::layout::LayoutState::UpdateValue::Index;

            auto itr = m_BnvibFileName.find(key);
            if (itr != m_BnvibFileName.end())
            {
                SetBnvibFile(itr->second.fileName.c_str(), itr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            itr = m_SoundFileName.find(key);
            if (itr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(itr->second.fileName.c_str(), itr->second.volume);
            }
            CallFunc();
        }

        if (
            !m_OldState.Test<nns::hidfw::layout::LayoutState::Hover>() &&
            m_State.Test<nns::hidfw::layout::LayoutState::Hover>()
            )
        {
            int key = nns::hidfw::layout::LayoutState::Hover::Index;
            auto itr = m_BnvibFileName.find(key);
            if (itr != m_BnvibFileName.end())
            {
                SetBnvibFile(itr->second.fileName.c_str(), itr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            itr = m_SoundFileName.find(key);
            if (itr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(itr->second.fileName.c_str(), itr->second.volume);
            }
        }

        if (
            !m_OldState.Test<nns::hidfw::layout::LayoutState::Selected>() &&
            m_State.Test<nns::hidfw::layout::LayoutState::Selected>()
            )
        {
            int key = nns::hidfw::layout::LayoutState::Selected::Index;
            auto itr = m_BnvibFileName.find(key);
            if (itr != m_BnvibFileName.end())
            {
                SetBnvibFile(itr->second.fileName.c_str(), itr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            itr = m_SoundFileName.find(key);
            if (itr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(itr->second.fileName.c_str(), itr->second.volume);
            }
        }

        if (
            !m_OldState.Test<nns::hidfw::layout::LayoutState::Canceled>() &&
            m_State.Test<nns::hidfw::layout::LayoutState::Canceled>()
            )
        {
            int key = nns::hidfw::layout::LayoutState::Canceled::Index;
            auto itr = m_BnvibFileName.find(key);
            if (itr != m_BnvibFileName.end())
            {
                SetBnvibFile(itr->second.fileName.c_str(), itr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            itr = m_SoundFileName.find(key);
            if (itr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(itr->second.fileName.c_str(), itr->second.volume);
            }
        }
    }

    void VibrationKnob::SetValue(int value) NN_NOEXCEPT
    {
        m_MaterValue = static_cast<float>(value);
    }

    void VibrationKnob::SetDefaultValue(int value) NN_NOEXCEPT
    {
        m_DefaultValue = value;
    }

    void VibrationKnob::SetMaxValue(int value) NN_NOEXCEPT
    {
        m_MaxValue = value;
        if (m_MaxValue < floorf(m_MaterValue + 0.5f))
        {
            m_MaterValue = static_cast<float>(value);
        }
        m_DefaultValue = std::min(m_DefaultValue, m_MaxValue);
    }

    void VibrationKnob::SetMinValue(int value) NN_NOEXCEPT
    {
        m_MinValue = value;
        if (m_MinValue > floorf(m_MaterValue + 0.5f))
        {
            m_MaterValue = static_cast<float>(value);
        }
        m_DefaultValue = std::max(m_DefaultValue, m_MinValue);
    }

    int VibrationKnob::GetValue() NN_NOEXCEPT
    {
        return static_cast<int64_t>(floorf(m_MaterValue + 0.5f));
    }

    void VibrationKnob::PrintText()
    {
        m_OldState = m_State;
    }

    void VibrationKnob::Draw()
    {
        const auto radius = std::min(m_Size.x * m_Scale.x / 2.f, m_Size.y * m_Scale.y / 2.f);
        const auto centerPos = nn::util::MakeFloat2(m_Pos.x + radius, m_Pos.y + radius);

        const auto effectColor = m_State.Test<nns::hidfw::layout::LayoutState::Selected>() ? Color::Yellow : Color::Cyan;

        const nn::util::Color4u8 transparenceEffectColor = nn::util::Color4u8(effectColor.GetR(), effectColor.GetG(), effectColor.GetB(), 0);

        auto max = (m_MaxValue - m_MinValue != 0) ? static_cast<float>(m_MaxValue - m_MinValue) : 1.f;
        auto len = (std::roundf(m_MaterValue) / max) * 270.f;

        if (m_State.Test<nns::hidfw::layout::LayoutState::Hover>())
        {
            auto effectPos = nn::util::MakeFloat2(m_Pos.x + m_Size.x * 0.25f, m_Pos.y + m_Size.y + 4.f);
            auto effectSize = nn::util::MakeFloat2(m_Size.x * 0.5f, 6.f);
            gDrawer.SetColor(effectColor);
            gDrawer.Draw2DRoundedRect(effectPos, effectSize, 1.f, 32);
            gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_In, effectColor, transparenceEffectColor);
            gDrawer.Draw2DRoundedFrame(effectPos, effectSize, 1.f, 32, 1.f);
        }

        int quality = 54;

        // メーター背面
        gDrawer.SetColor(m_BorderColor);
        gDrawer.Draw2DCircleFrame(centerPos, radius - 6.f, 6.f, quality * 2, 225.f, 270.f);
        // メーター部
        gDrawer.SetColor(m_SubColor);
        gDrawer.Draw2DCircleFrame(centerPos, radius - 6.f, 6.f, quality * 2, 225.f, len);

        // ベース
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Up, m_MainColor, m_BorderColor);
        gDrawer.Draw2DCircle(centerPos, radius * 0.7f, quality, 0);
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, m_MainColor, m_BorderColor);
        gDrawer.Draw2DCircle(centerPos, radius * 0.66f, quality);
        gDrawer.SetColor(m_BorderColor);
        gDrawer.Draw2DCircleFrame(centerPos, radius * 0.66f, 0.5f, quality);

        // ボッチ
        const auto knobVal = std::roundf(m_MaterValue) / max;
        const auto knobPos = nn::util::MakeFloat2(centerPos.x + nn::util::SinEst(nn::util::DegreeToRadian(315.f - knobVal * 270)) * radius * 0.4f, centerPos.y + nn::util::CosEst(nn::util::DegreeToRadian(knobVal * 270 + 45.f)) * radius * 0.4f);
        gDrawer.SetColor(m_MainColor);
        gDrawer.Draw2DCircle(knobPos, radius * 0.2f, quality);
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, m_BorderColor, m_MainColor);
        gDrawer.Draw2DCircle(knobPos, radius * 0.2f - 1.f, quality);

        // 現在位置
        const auto makerVal = std::roundf(m_MaterValue) / max;
        const auto makerPos = nn::util::MakeFloat2(centerPos.x + nn::util::SinEst(nn::util::DegreeToRadian(315.f - makerVal * 270)) * (radius - 4.f), centerPos.y + nn::util::CosEst(nn::util::DegreeToRadian(makerVal * 270 + 45.f)) * (radius - 4.f));
        gDrawer.SetColor(m_SubColor);
        gDrawer.Draw2DCircle(makerPos, 9.f, quality);
    }
}
