﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VibrationDemo_VibrationItem.h"

namespace VibrationDemo
{
    void VibrationItem::SetVibration(const nns::hidfw::layout::LayoutStateSet& state, const char* fileName) NN_NOEXCEPT
    {
        SetVibration(state, fileName, 1.f);
    }

    void VibrationItem::SetVibration(const nns::hidfw::layout::LayoutStateSet& state, const char* fileName, float volume) NN_NOEXCEPT
    {
        auto layoutState = state;
        for (int i = 0; layoutState.CountPopulation() > 0; ++i)
        {
            if (layoutState.Test(i))
            {
                if (m_BnvibFileName.find(i) != m_BnvibFileName.end())
                {
                    m_BnvibFileName.erase(i);
                }
                m_BnvibFileName.emplace(i, FileData(fileName, volume));
                layoutState.Reset(i);
            }
        }
    }

    void VibrationItem::SetSoundEffect(const nns::hidfw::layout::LayoutStateSet& state, const char* fileName) NN_NOEXCEPT
    {
        SetSoundEffect(state, fileName, 1.f);
    }

    void VibrationItem::SetSoundEffect(const nns::hidfw::layout::LayoutStateSet& state, const char* fileName, float volume) NN_NOEXCEPT
    {
        auto layoutState = state;
        for (int i = 0; layoutState.CountPopulation() > 0; ++i)
        {
            if (layoutState.Test(i))
            {
                if (m_SoundFileName.find(i) != m_SoundFileName.end())
                {
                    m_SoundFileName.erase(i);
                }
                m_SoundFileName.emplace(i, FileData(fileName, volume));
                layoutState.Reset(i);
            }
        }
    }

    void VibrationItem::SetVibrationSources(VibrationSource<nn::hid::VibrationPlayer>** pSources, int count) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pSources);
        int sourceCount = static_cast<int>(std::min(NN_ARRAY_SIZE(m_pVibrationSources), (size_t)count));
        m_SourceCount = 0;
        for (int i = 0; i < sourceCount; ++i)
        {
            m_pVibrationSources[i] = pSources[i];
            ++m_SourceCount;
        }
    }

    float VibrationItem::GetGain() NN_NOEXCEPT
    {
        return m_MasterGain;
    }

    void VibrationItem::SetGain(float gain) NN_NOEXCEPT
    {
        m_MasterGain = std::max(0.f, gain);
    }

    void VibrationItem::SetVibrationBalance(const float value) NN_NOEXCEPT
    {
        NN_UNUSED(value);
        // 現時点では無効化
    }

    float VibrationItem::GetVibrationBalance() const NN_NOEXCEPT
    {
        return m_VibrationBalance;
    }

    void VibrationItem::SetBnvibFile(const char* fileName) NN_NOEXCEPT
    {
        SetBnvibFile(fileName, 1.f);
    }

    void VibrationItem::SetBnvibFile(const char* fileName, float volume) NN_NOEXCEPT
    {
        void* address = nullptr;
        size_t fileSize = 0;

        const auto isLoad = gVibrationManager.GetBnvibFile(fileName, &address, &fileSize);

        for (int i = 0; i < m_SourceCount; ++i)
        {
            auto pSource = m_pVibrationSources[i];
            if (pSource != nullptr)
            {
                if (pSource->node.IsPlaying())
                {
                    pSource->node.Stop();
                }
                if (isLoad)
                {
                    NN_ASSERT(pSource->node.Load(address, fileSize).IsSuccess());
                }
                else if(pSource->node.IsLoaded())
                {
                    pSource->node.Unload();
                }

                nn::hid::VibrationModulation modulation = nn::hid::VibrationModulation::Make();
                modulation.gainHigh *= volume; modulation.gainLow *= volume;
                pSource->SetDefaultModulation(modulation, nn::hid::NpadJoyDeviceType_Left);
                pSource->SetDefaultModulation(modulation, nn::hid::NpadJoyDeviceType_Right);
            }
        }
    }

    const std::map<int, VibrationItem::FileData>& VibrationItem::GetVibrationFile() NN_NOEXCEPT
    {
        return m_BnvibFileName;
    }

    const std::map<int, VibrationItem::FileData>& VibrationItem::GetSoundEffectFile() NN_NOEXCEPT
    {
        return m_SoundFileName;
    }

    void VibrationItem::PlayVibration(nn::hid::NpadIdType id) NN_NOEXCEPT
    {
        for (int i = 0; i < m_SourceCount; ++i)
        {
            auto pSource = m_pVibrationSources[i];
            if (pSource != nullptr)
            {
                if (pSource->npadId == id &&
                    pSource->node.IsLoaded() &&
                    pSource->node.GetCurrentPosition() >= 0 &&
                    pSource->node.GetCurrentPosition() < pSource->node.GetFileInfo().sampleLength &&
                    pSource->node.GetLoopStartPosition() >= 0 &&
                    pSource->node.GetLoopStartPosition() < pSource->node.GetLoopEndPosition()
                    )
                {
                    if (pSource->node.IsPlaying())
                    {
                        pSource->node.Stop();
                    }
                    nn::hid::VibrationModulation modulation, leftModulation, rightModulation;
                    pSource->ResetModulation();
                    if (pSource->GetModulation(&leftModulation, nn::hid::NpadJoyDeviceType_Left))
                    {
                        modulation = leftModulation;
                        modulation.gainHigh = (modulation.gainHigh - modulation.gainHigh * std::min(1.f, m_VibrationBalance)) * m_MasterGain;
                        modulation.gainLow = (modulation.gainLow - modulation.gainLow * std::min(1.f, m_VibrationBalance)) * m_MasterGain;
                        pSource->SetModulation(modulation, nn::hid::NpadJoyDeviceType_Left);
                    }
                    if (pSource->GetModulation(&rightModulation, nn::hid::NpadJoyDeviceType_Right))
                    {
                        modulation = rightModulation;
                        modulation.gainHigh = (modulation.gainHigh - modulation.gainHigh * std::min(1.f, -m_VibrationBalance)) * m_MasterGain;
                        modulation.gainLow = (modulation.gainLow - modulation.gainLow * std::min(1.f, -m_VibrationBalance)) * m_MasterGain;
                        pSource->SetModulation(modulation, nn::hid::NpadJoyDeviceType_Right);
                    }
                    pSource->node.Play();
                }
            }
        }
    }

    void VibrationItem::PlayVibration(nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType deviceType) NN_NOEXCEPT
    {
        for (int i = 0; i < m_SourceCount; ++i)
        {
            auto pSource = m_pVibrationSources[i];
            if (pSource != nullptr)
            {
                nn::hid::NpadJoyDeviceType type;

                if (pSource->npadId == id &&
                    pSource->GetConnectedDeviceType(&type) &&
                    type == deviceType &&
                    pSource->node.IsLoaded() &&
                    pSource->node.GetCurrentPosition() >= 0 &&
                    pSource->node.GetCurrentPosition() < pSource->node.GetFileInfo().sampleLength &&
                    pSource->node.GetLoopStartPosition() >= 0 &&
                    pSource->node.GetLoopStartPosition() < pSource->node.GetLoopEndPosition()
                    )
                {
                    if (pSource->node.IsPlaying())
                    {
                        pSource->node.Stop();
                    }
                    nn::hid::VibrationModulation modulation, leftModulation, rightModulation;
                    if (deviceType == nn::hid::NpadJoyDeviceType_Left && pSource->GetModulation(&leftModulation, nn::hid::NpadJoyDeviceType_Left))
                    {
                        modulation = leftModulation;
                        modulation.gainHigh = (modulation.gainHigh - modulation.gainHigh * std::min(1.f, m_VibrationBalance)) * m_MasterGain;
                        modulation.gainLow = (modulation.gainLow - modulation.gainLow * std::min(1.f, m_VibrationBalance)) * m_MasterGain;
                        pSource->SetModulation(modulation, nn::hid::NpadJoyDeviceType_Left);
                    }
                    if (deviceType == nn::hid::NpadJoyDeviceType_Right && pSource->GetModulation(&rightModulation, nn::hid::NpadJoyDeviceType_Right))
                    {
                        modulation = rightModulation;
                        modulation.gainHigh = (modulation.gainHigh - modulation.gainHigh * std::min(1.f, -m_VibrationBalance)) * m_MasterGain;
                        modulation.gainLow = (modulation.gainLow - modulation.gainLow * std::min(1.f, -m_VibrationBalance)) * m_MasterGain;
                        pSource->SetModulation(modulation, nn::hid::NpadJoyDeviceType_Right);
                    }
                    pSource->node.Play();
                }
            }
        }
    }
}
