﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Base.h"
#include "Audio.h"
#include "Button.h"
#include "Drawer.h"
#include "File.h"
#include "Vibration.h"
#include "Command.h"

namespace VibrationCollection
{
    std::string g_HostRootPath = "D://Nintendo/Resources/";
    std::string g_SdRootPath = "Nintendo/Resources/";

    extern "C" void nnMain()
    {
        //======================================================================
        // 初期状態のリソース参照パスを設定
        //======================================================================

        gFileManager.SetMountType(MountType_Sd);
        gFileManager.SetRootPath(g_SdRootPath.c_str());
        gFileManager.ChangePath();

        //======================================================================
        // 起動時引数の処理
        //======================================================================

        StartupArgument();

        //======================================================================
        // 初期化
        //======================================================================

        gGraphics.Initialize();                     //!< グラフィックス機能の初期化を行います
        gInput.Initialize();                        //!< 入力関連の機能の初期化を行います
        gAudio.Initialize();
        gAudio.Enable();

        // 無線1コンか本体装着コントローラのみ使用します
        const nn::hid::NpadIdType ids[] =
        {
            nn::hid::NpadId::No1,
            nn::hid::NpadId::Handheld
        };
        nn::hid::SetSupportedNpadIdType(ids, NN_ARRAY_SIZE(ids));

        // NpadStyleFullKey もしくは NpadStyleHandheld 以外のコントローラを受け付けません
        nn::hid::SetSupportedNpadStyleSet(
            nn::hid::NpadStyleFullKey::Mask |
            nn::hid::NpadStyleHandheld::Mask
        );

        nn::mem::StandardAllocator appAllocator;
        const size_t appMemorySize = 128 * 1024 * 1024;
        nn::Bit8* pAppMemory = new nn::Bit8[appMemorySize];

        appAllocator.Initialize(pAppMemory, appMemorySize);

        InitializeResource(gGraphics.GetApplicationHeap());

        gFileManager.LoadBnvib(true);
        gFileManager.LoadAudio(true);

        int targetCount = 0;
        nn::hid::VibrationDeviceHandle handle[16];
        for (int i = 0; i < gController.GetController(nn::hid::NpadId::Handheld)->GetVibration().GetVibrationDeviceHandle(nn::hid::NpadStyleHandheld::Mask, handle, NN_ARRAY_SIZE(handle)); ++i)
        {
            g_VibrationTarget.VibrationTarget[targetCount].SetVibrationDeviceHandle(handle[i]);
            ++targetCount;
        }
        gController.GetController(nn::hid::NpadId::Handheld)->GetVibration().Disable();
        for (int i = 0; i < gController.GetController(nn::hid::NpadId::No1)->GetVibration().GetVibrationDeviceHandle(nn::hid::NpadStyleFullKey::Mask, handle, NN_ARRAY_SIZE(handle)); ++i)
        {
            g_VibrationTarget.VibrationTarget[targetCount].SetVibrationDeviceHandle(handle[i]);
            ++targetCount;
        }
        gController.GetController(nn::hid::NpadId::No1)->GetVibration().Disable();
        for (int i = 0; i < g_VibrationTarget.MaxTargetCount && i < targetCount; ++i)
        {
            g_VibrationTarget.Connection[i].Connect(&g_VibrationPlayer, &g_VibrationTarget.VibrationTarget[i]);
        }

        nn::Result result;

        VibrationUpdateThread::Initialize();
        VibrationUpdateThread::Start();

        //======================================================================
        // レイアウトアイテムの初期設定
        //======================================================================

        // メインのボタンを初期化します
        InitializeMainButtonGroup(&g_ButtonGroup[ApplicationButtonGroup_Main]);
        // 参照先の設定関係のボタンを初期化します
        InitializeFileButtonGroup(&g_ButtonGroup[ApplicationButtonGroup_File]);
        // 振動選択のボタンを初期化します
        InitializeVibrationButtonGroup(&g_ButtonGroup[ApplicationButtonGroup_Vibration]);
        // 効果音選択のボタンを初期化します
        InitializeAudioButtonGroup(&g_ButtonGroup[ApplicationButtonGroup_Audio]);

        g_ButtonGroup[ApplicationButtonGroup_Main].SetPrevItemSet(&g_ButtonGroup[ApplicationButtonGroup_Audio]);
        g_ButtonGroup[ApplicationButtonGroup_Main].SetNextItemSet(&g_ButtonGroup[ApplicationButtonGroup_Vibration]);

        g_ButtonGroup[ApplicationButtonGroup_Vibration].SetPrevItemSet(&g_ButtonGroup[ApplicationButtonGroup_Main]);
        g_ButtonGroup[ApplicationButtonGroup_Vibration].SetNextItemSet(&g_ButtonGroup[ApplicationButtonGroup_Audio]);

        g_ButtonGroup[ApplicationButtonGroup_Audio].SetPrevItemSet(&g_ButtonGroup[ApplicationButtonGroup_Vibration]);
        g_ButtonGroup[ApplicationButtonGroup_Audio].SetNextItemSet(&g_ButtonGroup[ApplicationButtonGroup_Main]);

        g_ButtonGroup[ApplicationButtonGroup_File].SetNextItemSet(&g_ButtonGroup[ApplicationButtonGroup_Audio]);

        nns::hidfw::layout::ButtonGroup vibrationViewer;

        // 振動の表示関連のコンポーネントを初期化
        InitializeVibrationViewer(&vibrationViewer);

        // 最初のボタンセットにフォーカスをあわせます
        g_ButtonGroup[ApplicationButtonGroup_Main].Focus(true);
        // フォーカス変更後は全てのコマンドが無効化されるため、更新を行います
        g_ButtonGroup[ApplicationButtonGroup_Main].Update();

        //======================================================================
        // 初期設定
        //======================================================================

        // 背景色を白に
        gDrawer.SetClearColor(nn::util::Color4u8::White());
        //gDrawer.SetClearColor(nn::util::Color4u8(2, 3, 4));
        // 現在実行中のスレッドで実行
        nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

        bool                exit        = false;                         // trueになったらループ終了 (サンプルではtrueになることはありません)
        uint64_t            frameCount  = 0;                             // フレームカウント

        SetupNpadButtonIconset();
        SetupNpadStyleIconSet();
        SetupToolIconSet();

        //======================================================================
        // メインループ
        //======================================================================
        do
        {
            // BeginPerfFrame() ～ EndPerfFrame() 間でパフォーマンスを計測します
            gGraphics.GetGraphicsSystem().BeginPerfFrame();

            //======================================================================
            // 入力処理
            //======================================================================
            // コントローラ・タッチスクリーンの入力を更新します
            gInput.Update();
            //gAudio.Update();

            // 接続中のコントローラのボタンの押下げ状況を確認します
            for (
                std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
                it != gController.GetConnectedControllerList().end();
                ++it
                )
            {
                if ((*it)->IsTrigger(nn::hid::NpadButton::Plus::Mask) || (*it)->IsTrigger(nn::hid::NpadButton::Minus::Mask))
                {
                    ShowSetting(nullptr, nullptr);
                }
                else
                {
                    if ((*it)->IsTrigger(nn::hid::NpadButton::X::Mask))
                    {
                        PlayVibration(nullptr, nullptr);
                    }

                    bool isLeftConnected = ((*it)->GetStyleSet() & (nn::hid::NpadStyleJoyLeft::Mask)).IsAnyOn();
                    // 振動の表示先
                    nns::hidfw::layout::LineChart* pLineChart = (nns::hidfw::layout::LineChart*)vibrationViewer.GetItemSet().at(0);
                    nns::hidfw::layout::BarChart* pBarChart = (nns::hidfw::layout::BarChart*)vibrationViewer.GetItemSet().at(1);
                    // 振動値
                    nn::hid::VibrationValue left, right, sendVib[2];
                    // 送信されている振動の期待値
                    (*it)->GetVibration().GetCurVibrationValue(&left, &right);
                    // 実際の振動値
                    size_t vibCount = (*it)->GetVibration().GetCurSendVibrationValue(sendVib, 2);
                    // 振動子が存在する場合、接続されているコントローラ(左優先)の振動値で振動表示を更新します
                    if (vibCount > 0)
                    {
                        pLineChart->AddValue(0, (isLeftConnected ? left : right).amplitudeLow);
                        pLineChart->AddValue(1, (isLeftConnected ? left : right).amplitudeHigh);

                        pBarChart->AddValue((isLeftConnected ? left : right).frequencyLow, (isLeftConnected ? left : right).amplitudeLow);
                        pBarChart->AddValue((isLeftConnected ? left : right).frequencyHigh, (isLeftConnected ? left : right).amplitudeHigh);
                    }
                }
                break;
            }
            //======================================================================
            // アイテムセットの更新処理
            //======================================================================

            for (size_t i = 0; i < NN_ARRAY_SIZE(g_ButtonGroup); ++i)
            {
                g_ButtonGroup[i].Update();                     //!< ボタンセットの更新
            }
            gTextWriter.SetScale(1.f, 1.f);
            vibrationViewer.Update();

            //======================================================================
            // テキスト出力
            //======================================================================

            //======================================================================
            // 描画処理
            //======================================================================

            gDrawer.BeginDraw();            // 描画関数の受付開始
            {
                DrawBackGround();
                DrawMainMenu();
                DrawStatusBar();
                vibrationViewer.Draw();
                g_ButtonGroup[ApplicationButtonGroup_Main].Draw();
                g_ButtonGroup[ApplicationButtonGroup_File].Draw();
            }
            gDrawer.EndDraw();              // 描画終了
            ++frameCount;

            // BeginPerfFrame() ～ EndPerfFrame() 間でパフォーマンスを計測します
            gGraphics.GetGraphicsSystem().EndPerfFrame();

        } while (!exit);

        //======================================================================
        // 終了処理
        //======================================================================

        gInput.Finalize();
        gGraphics.Finalize();

    } // NOLINT(impl/function_size)
}
