﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/system/hid_UniquePad.h>

#include "../sgx/SimpleGfx.h"
#include "../sgx/gui/SimpleGfx_Gui.h"
#include "../util/StringTable.h"
#include "../util/util.h"
#include "UniquePadView.h"

NN_DISABLE_WARNING_ARRAY_DEFAULT_INITIALIZATION_IN_CONSTRUCTOR;

namespace nns { namespace hid { namespace ui {

namespace
{

}  // anonymous

UniquePadView::UniquePadView() NN_NOEXCEPT
    : m_Manager()
    , m_ManagerUpdateEvent()
    , m_ButtonSkin()
    , m_Buttons()
    , m_DecideCallback(nullptr)
    , m_DecideCallbackArg()
{
}

UniquePadView::~UniquePadView() NN_NOEXCEPT
{
    Finalize();
}

bool UniquePadView::IsEmpty() const NN_NOEXCEPT
{
    for (auto& button : m_Buttons)
    {
        // 表示状態のボタンが一つでもあれば空ではない
        if (button.IsVisible())
        {
            return false;
        }
    }

    return true;
}

void UniquePadView::SetDecideCallback(DecideCallbackFunc callback, uintptr_t arg) NN_NOEXCEPT
{
    m_DecideCallback    = callback;
    m_DecideCallbackArg = arg;
}

void UniquePadView::SetTargetImage(const util::FirmwareImage& image) NN_NOEXCEPT
{
    for (auto& button : m_Buttons)
    {
        button.SetTargetDevice(image.GetTargetDeviceType());
    }
}

void UniquePadView::Initialize() NN_NOEXCEPT
{
    if (m_IsInitialized)
    {
        return;
    }

    m_IsInitialized = true;

    SetDirection(nns::sgx::gui::LayoutDirection::Vertical);
    SetItemGap({ { 8, 0 } });

    nn::os::InitializeLightEvent(&m_ManagerUpdateEvent, false, nn::os::EventClearMode_ManualClear);
    m_Manager.Initialize();
    m_Manager.BindUpdateEvent(&m_ManagerUpdateEvent);

    // ボタンスキンの読み込み
    NN_ABORT_UNLESS_EQUAL(
        m_ButtonSkin.Load("rom:/FlatMenuButton.json"),
        nns::sgx::ResultCode::Success
    );

    // ボタンの初期設定
    for (auto& button : m_Buttons)
    {
        button.SetSkin(m_ButtonSkin);
        button.SetButtonMaskForPress(util::DecideButtonMask);
        button.SetManager(&m_Manager);
    }

    Refresh();
}

void UniquePadView::Finalize() NN_NOEXCEPT
{
    if (!m_IsInitialized)
    {
        return;
    }

    m_IsInitialized = false;
    m_Manager.Finalize();
    nn::os::FinalizeLightEvent(&m_ManagerUpdateEvent);
}

void UniquePadView::Update() NN_NOEXCEPT
{
    m_Manager.Update();

    // UniquePad の接続状態が変化したらボタンを再生成
    if (nn::os::TryWaitLightEvent(&m_ManagerUpdateEvent))
    {
        nn::os::ClearLightEvent(&m_ManagerUpdateEvent);
        Refresh();
    }

    FlowLayoutContainer::Update();
}

void UniquePadView::Render() NN_NOEXCEPT
{
    float dx = GetX() - 12;
    float dy = GetY() + 3;
    nns::sgx::DrawGradientLine(
        dx, dy, dx, dy + GetHeight() / 2,
        nns::sgx::Colors::WhiteSmoke(),
        nns::sgx::Colors::LightGray(),
        4
    );
    nns::sgx::DrawGradientLine(
        dx, dy + GetHeight() / 2, dx, dy + GetHeight(),
        nns::sgx::Colors::LightGray(),
        nns::sgx::Colors::WhiteSmoke(),
        4
    );

    FlowLayoutContainer::Render();
}

void UniquePadView::PushButtonHandler(NNS_SGX_GUI_EVENT_HANDLER_ARGS(pSender, arg)) NN_NOEXCEPT
{
    auto& button = *reinterpret_cast<UniquePadButton*>(pSender);
    auto& scene  = *reinterpret_cast<UniquePadView*>(arg);

    if (scene.m_DecideCallback != nullptr)
    {
        scene.m_DecideCallback(button.GetUniquePadId(), scene.m_DecideCallbackArg);
    }
}

void UniquePadView::Refresh() NN_NOEXCEPT
{
    // 直前まで選択していた項目を記憶
    nn::hid::system::UniquePadSerialNumber lastSerial = {};
    int lastIndex = 0;
    auto* pLastFocused = reinterpret_cast<UniquePadButton*>(GetFocusedChild());
    if (pLastFocused != nullptr)
    {
        nn::hid::system::GetUniquePadSerialNumber(&lastSerial, pLastFocused->GetUniquePadId());
        for (int i = 0; i < NN_ARRAY_SIZE(m_Buttons); i++)
        {
            if (&m_Buttons[i] == pLastFocused)
            {
                lastIndex = i;
                break;
            }
        }
    }

    // 単純化のために一旦全ボタンを除去
    for (auto& button : m_Buttons)
    {
        RemoveChild(&button);
        button.SetVisible(false);
    }

    const float buttonWidth = GetClientArea().width - 32;
    const float buttonHeight = 64;

    int padCount = m_Manager.GetPadCount();
    bool isLastSelected = false;
    for (int i = 0; i < padCount; i++)
    {
        nn::hid::system::UniquePadId id;
        if (m_Manager.GetUniquePadId(&id, i).IsFailure())
        {
            continue;
        }

        auto& button = m_Buttons[i];
        button.SetVisible(true);
        button.SetSize(buttonWidth, buttonHeight);
        button.SetUniquePadId(id, i);
        button.SetPushEventHandler(PushButtonHandler, reinterpret_cast<uintptr_t>(this));
        AddChild(&button);

        // 直前のデバイスに一致したら選択
        nn::hid::system::UniquePadSerialNumber serial;
        if (nn::hid::system::GetUniquePadSerialNumber(&serial, id).IsSuccess() &&
            std::memcmp(&serial, &lastSerial, sizeof(serial)) == 0)  // XXX: 等値演算がないので memcmp
        {
            SetFocusedChild(&button);
            isLastSelected = true;
        }
    }

    // 直前のデバイスが見つからなければ、なるべく同じ index を選択
    if (!isLastSelected && padCount > 0)
    {
        SetFocusedChild(&m_Buttons[std::min(lastIndex, padCount - 1)]);
    }
}

}}}  // nns::hid::ui
