﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include "../sgx/SimpleGfx.h"
#include "../sgx/gui/SimpleGfx_Gui.h"
#include "../util/FirmwareImage.h"
#include "../util/StringTable.h"
#include "../util/util.h"
#include "FirmwareUpdateProgressDialog.h"

namespace nns { namespace hid { namespace ui {

namespace
{

const int ProgressAnimFrameCount = 90;

}  // anonymous

FirmwareUpdateProgressDialog::FirmwareUpdateProgressDialog() NN_NOEXCEPT
    : m_IsProcessing(false)
    , m_WaitIcon()
    , m_ProgressBarImage()
    , m_Progress(0)
    , m_ProgressAnimCount(0)
    , m_ControllerNumber(0)
    , m_PadType(nn::hid::system::UniquePadType_Unknown)
    , m_DeviceHandle()
    , m_UpdateResult()
{
}

FirmwareUpdateProgressDialog::~FirmwareUpdateProgressDialog() NN_NOEXCEPT
{
    nns::sgx::DestroyImage(&m_ProgressBarImage);
}

void FirmwareUpdateProgressDialog::Initialize() NN_NOEXCEPT
{
    Dialog::Initialize();

    NN_ABORT_UNLESS_EQUAL(
        nns::sgx::LoadImage(&m_ProgressBarImage, "rom:/ProgressBar.bmp"),
        nns::sgx::ResultCode::Success
    );

    SetButtonCount(0);
    SetSize(640, 320);
    SetRenderTextAreaHandler(RenderHandler, reinterpret_cast<uintptr_t>(this));

    m_WaitIcon.SetPosition((GetTextAreaSize().width - m_WaitIcon.GetWidth()) / 2, 48);
    m_WaitIcon.SetColor(nns::sgx::Colors::RoyalBlue());
    m_WaitIcon.SetTextSize(64.0f);
    AddChild(&m_WaitIcon);

    //m_ProgressBar.SetSize(400, 20);
    //m_ProgressBar.SetPosition(80, 250);
    //m_ProgressBar.SetMinValue(0);
    //m_ProgressBar.SetMaxValue(100);
    //AddChild(&m_ProgressBar);
}

void FirmwareUpdateProgressDialog::Disappear() NN_NOEXCEPT
{
    Dialog::Disappear();

    m_IsProcessing = false;
}

void FirmwareUpdateProgressDialog::StartProgressCheck(
    const nn::hid::system::UniquePadId& id,
    const nn::hid::system::FirmwareUpdateDeviceHandle& handle) NN_NOEXCEPT
{
    if (nn::hid::system::GetUniquePadControllerNumber(&m_ControllerNumber, id).IsFailure())
    {
        // 即失敗
        m_IsProcessing = false;
        m_UpdateResult = nn::hid::system::ResultUniquePadDisconnected();
        return;
    }

    m_PadType           = nn::hid::system::GetUniquePadType(id);
    m_IsProcessing      = true;
    m_DeviceHandle      = handle;
    m_Progress          = 0;
    m_ProgressAnimCount = 0;
    m_UpdateResult      = nn::ResultSuccess();
}

bool FirmwareUpdateProgressDialog::IsFinished() const NN_NOEXCEPT
{
    if (m_UpdateResult.IsFailure())
    {
        return true;
    }
    else if (m_Progress >= 100)
    {
        return true;
    }

    return false;
}

bool FirmwareUpdateProgressDialog::IsSucceeded() const NN_NOEXCEPT
{
    if (m_UpdateResult.IsSuccess() && m_Progress >= 100)
    {
        return true;
    }

    return false;
}

void FirmwareUpdateProgressDialog::Update() NN_NOEXCEPT
{
    NN_UTIL_SCOPE_EXIT
    {
        Dialog::Update();
    };

    if (!m_IsProcessing)
    {
        return;
    }

    // 進捗を取得
    nn::hid::system::FirmwareUpdateState state;
    auto result = nn::hid::system::GetFirmwareUpdateState(&state, m_DeviceHandle);
    if (m_UpdateResult.IsSuccess())
    {
        if (result.IsFailure())
        {
            // 最初に失敗したら結果を更新
            m_IsProcessing = false;
            m_UpdateResult = result;
            NN_LOG("Firmware update failed (0x%08X)\n", result.GetInnerValueForDebug());
            return;
        }
        else if (state.stage == nn::hid::system::FirmwareUpdateStage_Completed)
        {
            // Completed になったら成功
            m_IsProcessing = false;
            m_Progress     = 100;
            return;
        }

        m_Progress = state.progress;
    }

    m_WaitIcon.SetPosition((GetTextAreaSize().width - m_WaitIcon.GetWidth()) / 2, m_WaitIcon.GetY());

    m_ProgressAnimCount = (m_ProgressAnimCount + 1) % ProgressAnimFrameCount;
}

void FirmwareUpdateProgressDialog::RenderHandler(NNS_SGX_GUI_EVENT_HANDLER_ARGS(pSender, arg)) NN_NOEXCEPT
{
    NN_UNUSED(pSender);

    auto& scene = *reinterpret_cast<FirmwareUpdateProgressDialog*>(arg);
    scene.RenderImpl();
}

void FirmwareUpdateProgressDialog::RenderImpl() NN_NOEXCEPT
{
    int opacity = GetDisplayOpacity();
    auto& str = nns::hid::util::StringTable::GetInstance();

    nns::sgx::ScopedFontContextSaver saver;

    nns::sgx::SetTextSize(24);
    nns::sgx::SetTextColor(nns::sgx::Colors::Shadow().BlendAlpha(opacity));

    nns::sgx::DrawText(180, 128, "%s", util::GetUniquePadTypeName(m_PadType));
    if (m_ControllerNumber > 0)
    {
        util::DrawControllerIndicator(166, 114, util::GetControllerIndicatorPattern(m_ControllerNumber), 12);
    }
    else
    {
        nns::sgx::ScopedFontContextSaver saverForIndicator;
        nns::sgx::SetTextSize(16);
        nns::sgx::DrawText(166, 110, "OnRail");
    }

    const float clientWidth = GetTextAreaSize().width;

    char buf[100];
    nn::util::SNPrintf(buf, sizeof(buf), "%s", str.Get("FlashWriter_Updating"));
    nns::sgx::DrawText(0, 176, clientWidth, 32, nns::sgx::TextAlignment::Center, buf);

    DrawProgressBar(70, 236);

    nns::sgx::DrawText(480, 230, "%d%%", m_Progress);
}

void FirmwareUpdateProgressDialog::DrawProgressBar(float x, float y) NN_NOEXCEPT
{
    bool isCompleted = m_Progress >= 100;

    const float gaugeLength = 400;
    float fillLength = gaugeLength * m_Progress / 100;

    // 背景
    nns::sgx::FillRectangle(x, y, gaugeLength + 2, 18, nns::sgx::Colors::Gray());

    // 本体
    float barWidth = m_ProgressBarImage.size.width / 3;
    float barHeight = m_ProgressBarImage.size.height / 2;
    nns::sgx::Rectangle destRect = { { x + 1, y + 1, fillLength, 16 } };
    if (isCompleted)
    {
        nns::sgx::Rectangle srcRect = { { 0, barHeight, barWidth, barHeight } };
        nns::sgx::DrawImage(m_ProgressBarImage, destRect, srcRect);
    }
    else
    {
        float rate = 1.0f - 1.0f * m_ProgressAnimCount / ProgressAnimFrameCount;
        float barX = barWidth * 2 * rate;
        nns::sgx::Rectangle srcRect = { { barX, 0, barWidth, barHeight } };
        nns::sgx::DrawImage(m_ProgressBarImage, destRect, srcRect);
    }
}

}}}  // nns::hid::ui
