﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../SimpleGfx.h"
#include "../detail/SimpleGfx_Json.h"
#include "SimpleGfx_GuiUtil.h"
#include "SimpleGfx_GuiButton.h"

namespace nns { namespace sgx { namespace gui {

struct Pixel32bpp
{
    uint8_t v[4];

    bool IsBlack() const NN_NOEXCEPT
    {
        return v[0] == 0 && v[1] == 0 && v[2] == 0 && v[3] > 0;
    }
};

constexpr Point2D AppearanceOffset[] =
{
    {{  0,  0 }},  // Normal
    {{ 64,  0 }},  // Focused
    {{  0, 64 }},  // Pressed
    {{ 64, 64 }}   // Disabled
};

void ClampVisibility(Rectangle* pRect) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pRect);

    if (pRect->x < 0.0f)
    {
        pRect->width = std::max(pRect->width + pRect->x, 0.0f);
        pRect->x = 0.0f;
    }

    if (pRect->y < 0.0f)
    {
        pRect->height = std::max(pRect->height + pRect->y, 0.0f);
        pRect->y = 0.0f;
    }
}

void Parse9PatchImage(Image9Patch* pOutImage, const ImageData& srcImage) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImage);
    NN_ASSERT(srcImage.IsValid());

    pOutImage->pImage = &srcImage;

    // フィールドの初期化
    for (int i = 0; i < ScalePointMax; i++)
    {
        pOutImage->scalePointX[i] = -1;
        pOutImage->scalePointY[i] = -1;
    }
    pOutImage->contentsAreaHorizontal.offset = 1;
    pOutImage->contentsAreaHorizontal.length = static_cast<int>(srcImage.size.width) - 2;
    pOutImage->contentsAreaVertical.offset = 1;
    pOutImage->contentsAreaVertical.length = static_cast<int>(srcImage.size.height) - 2;

    // X 方向の探索
    {
        int count = 0;
        int width = static_cast<int>(srcImage.size.width);
        int checkWidth = width - 2;
        int yOffset = static_cast<int>(srcImage.size.height) - 1;
        bool isInContents = false;
        auto& contentsArea = pOutImage->contentsAreaHorizontal;
        for (int x = 1; x < checkWidth; x++)
        {
            // スケールポイント
            if (count < ScalePointMax)
            {
                auto pPixel = reinterpret_cast<Pixel32bpp*>(
                    &srcImage.pPixelData[x * 4]);

                if (pPixel->IsBlack())
                {
                    pOutImage->scalePointX[count] = x;
                    count++;
                }
            }

            // コンテンツ領域
            {
                auto pPixel = reinterpret_cast<Pixel32bpp*>(
                    &srcImage.pPixelData[(yOffset * width + x) * 4]);

                if (isInContents)
                {
                    if (!pPixel->IsBlack())
                    {
                        // コンテンツ領域終了
                        isInContents = false;
                        contentsArea.length = x - contentsArea.offset;
                    }
                }
                else
                {
                    if (pPixel->IsBlack())
                    {
                        // コンテンツ領域開始
                        isInContents = true;
                        contentsArea.offset = x;
                    }
                }
            }
        }
    }

    // Y 方向の探索
    {
        int count = 0;
        int width = static_cast<int>(srcImage.size.width);
        int checkHeight = static_cast<int>(srcImage.size.height) - 2;
        int xOffset = static_cast<int>(srcImage.size.width) - 1;
        bool isInContents = false;
        auto& contentsArea = pOutImage->contentsAreaVertical;
        for (int y = 1; y < checkHeight; y++)
        {
            // スケールポイント
            if (count < ScalePointMax)
            {
                auto pPixel = reinterpret_cast<Pixel32bpp*>(
                    &srcImage.pPixelData[y * width * 4]);

                if (pPixel->IsBlack())
                {
                    pOutImage->scalePointY[count] = y;
                    count++;
                }
            }

            // コンテンツ領域
            {
                auto pPixel = reinterpret_cast<Pixel32bpp*>(
                    &srcImage.pPixelData[(y * width + xOffset) * 4]);

                if (isInContents)
                {
                    if (!pPixel->IsBlack())
                    {
                        // コンテンツ領域終了
                        isInContents = false;
                        contentsArea.length = y - contentsArea.offset;
                    }
                }
                else
                {
                    if (pPixel->IsBlack())
                    {
                        // コンテンツ領域開始
                        isInContents = true;
                        contentsArea.offset = y;
                    }
                }
            }
        }
    }

}  // NOLINT(readability/fn_size)

void Draw9PatchImage(const Image9Patch& image) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(image.pImage);
    NN_ASSERT(image.pImage->IsValid());

    // XXX: ScalePointMax が 1 であることを前提にした仮実装

    Rectangle srcRect;
    srcRect.x      = 1;
    srcRect.y      = 1;
    srcRect.width  = image.pImage->size.width  - 2;
    srcRect.height = image.pImage->size.height - 2;
}

void DrawImageButton(
    const ImageData& image,
    float x,
    float y,
    float width,
    float height,
    ButtonAppearance appearance) NN_NOEXCEPT
{
    NN_ASSERT(image.IsValid());
    NN_ASSERT_MINMAX(
        static_cast<int>(appearance),
        static_cast<int>(ButtonAppearance::Normal),
        static_cast<int>(ButtonAppearance::Disabled)
    );

    const auto& srcOffset = AppearanceOffset[static_cast<int>(appearance)];
    float srcX = srcOffset.x;
    float srcY = srcOffset.y;

    float inWidth  = std::max(width  - 16.0f, 0.0f);
    float inHeight = std::max(height - 16.0f, 0.0f);

    Rectangle srcRect;
    Rectangle dstRect;

    // 左上
    srcRect.Set(srcX, srcY, 8, 8);
    dstRect.Set(x, y, 8, 8);
    DrawImage(image, dstRect, srcRect);

    // 右上
    srcRect.Set(srcX + 56, srcY, 8, 8);
    dstRect.Set(x + inWidth + 8, y, 8, 8);
    DrawImage(image, dstRect, srcRect);

    // 左下
    srcRect.Set(srcX, srcY + 56, 8, 8);
    dstRect.Set(x, y + inHeight + 8, 8, 8);
    DrawImage(image, dstRect, srcRect);

    // 右上
    srcRect.Set(srcX + 56, srcY + 56, 8, 8);
    dstRect.Set(x + inWidth + 8, y + inHeight + 8, 8, 8);
    DrawImage(image, dstRect, srcRect);

    // 上下
    if (inWidth > 0.0f)
    {
        srcRect.Set(srcX + 8, srcY, 48, 8);
        dstRect.Set(x + 8, y, inWidth, 8);
        DrawImage(image, dstRect, srcRect);

        srcRect.y += 56;
        dstRect.y  = y + inHeight + 8;
        DrawImage(image, dstRect, srcRect);
    }

    // 左右
    if (inHeight > 0.0f)
    {
        srcRect.Set(srcX, srcY + 8, 8, 48);
        dstRect.Set(x, y + 8, 8, inHeight);
        DrawImage(image, dstRect, srcRect);

        srcRect.x += 56;
        dstRect.x  = x + inWidth + 8;
        DrawImage(image, dstRect, srcRect);
    }

    // 内部
    if (inWidth > 0.0f && inHeight > 0.0f)
    {
        srcRect.Set(srcX + 8, srcY + 8, 48, 48);
        dstRect.Set(x + 8, y + 8, inWidth, inHeight);
        DrawImage(image, dstRect, srcRect);
    }
}

void PlaySystemSe(SystemSe se) NN_NOEXCEPT
{
#if 0
    switch (se)
    {
    case SystemSe::Cursor:
        SystemSounds::CursorSe.Play();
        break;

    case SystemSe::Press:
        SystemSounds::PressSe.Play();
        break;

    case SystemSe::Decide:
        SystemSounds::DecideSe.Play();
        break;

    case SystemSe::Cancel:
        SystemSounds::CancelSe.Play();
        break;

    case SystemSe::Buzzer:
        SystemSounds::BuzzerSe.Play();
        break;

    case SystemSe::Notice:
        SystemSounds::NoticeSe.Play();
        break;

    case SystemSe::Success:
        SystemSounds::SuccessSe.Play();
        break;

    case SystemSe::Failure:
        SystemSounds::FailureSe.Play();
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
#else
    NN_UNUSED(se);
#endif
}

}}}  // nns::sgx::gui
