﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_Gui.h"
#include "../../input/TouchPanel.h"

namespace nns { namespace sgx { namespace gui {

DisplayObject::DisplayObject() NN_NOEXCEPT
    : m_Mutex(true)
    , m_pOwner(nullptr)
    , m_Position()
    , m_Size()
    , m_OrderZ(0)
    , m_DragOffset()
    , m_IsVisible(true)
    , m_IsDecideEffectEnabled(false)
    , m_Opacity(255)
    , m_PostUpdateGuiEvent()
{
}

float DisplayObject::GetAbsoluteX() const NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    float x = GetX();
    if (m_pOwner == nullptr)
    {
        return x;
    }

    return x + m_pOwner->GetClientAreaPosition().x;
}

float DisplayObject::GetAbsoluteY() const NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    float y = GetY();
    if (m_pOwner == nullptr)
    {
        return y;
    }

    return y + m_pOwner->GetClientAreaPosition().y;
}

void DisplayObject::GetAbsolutePosition(Point2D* pOutPosition) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutPosition);
    NNS_SGX_GUI_SCOPED_LOCK;

    pOutPosition->x = GetAbsoluteX();
    pOutPosition->y = GetAbsoluteY();
}

void DisplayObject::GetRenderPosition(Point2D* pOutPosition) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutPosition);
    NNS_SGX_GUI_SCOPED_LOCK;

    *pOutPosition = m_Position;

#if 0
    if (m_pOwner != nullptr)
    {
        pOutPosition->x += m_pOwner->GetPadding();
        pOutPosition->y += m_pOwner->GetPadding();
    }
#endif
}

Point2D DisplayObject::GetAnchorPoint(Direction dir) NN_NOEXCEPT
{
    switch (dir)
    {
    case Direction::Up:
        return { { GetX() + GetWidth() / 2, GetY() } };

    case Direction::Down:
        return { { GetX() + GetWidth() / 2, GetY() + GetHeight() } };

    case Direction::Left:
        return { { GetX(), GetY() + GetHeight() / 2 } };

    case Direction::Right:
        return { { GetX() + GetWidth(), GetY() + GetHeight() / 2 } };

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void DisplayObject::GetDraggableArea(Rectangle* pOutArea) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutArea);
    NNS_SGX_GUI_SCOPED_LOCK;

    GetAbsolutePosition(&pOutArea->position);
    GetSize(&pOutArea->size);
}

Rectangle DisplayObject::GetHitArea() const NN_NOEXCEPT
{
    Rectangle rect;
    GetAbsolutePosition(&rect.position);
    GetSize(&rect.size);

    if (m_pOwner != nullptr)
    {
        // この時点で座標にスクロール量が反映されているはず

        auto ownerClient = m_pOwner->GetClientAreaSize();
        float x      = GetX();
        float y      = GetY();
        float width  = GetWidth();
        float height = GetHeight();

        if (x < 0.0f)
        {
            // 左にはみ出ている
            rect.position.x -= x;
            rect.size.width  = std::max(rect.size.width + x, 0.0f);
        }
        else if (x + width >= ownerClient.width)
        {
            // 右にはみ出ている
            rect.size.width -= x + width - ownerClient.width;
        }

        if (y < 0.0f)
        {
            // 上にはみ出ている
            rect.position.y -= y;
            rect.size.height = std::max(rect.size.height + y, 0.0f);
        }
        else if (y + height >= ownerClient.height)
        {
            // 下にはみ出ている
            rect.size.height -= y + height - ownerClient.height;
        }
    }

    return rect;
}

bool DisplayObject::IsVisible() const NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    // オーナーが非表示なら共に非表示
    if (m_pOwner != nullptr && !m_pOwner->IsVisible())
    {
        return false;
    }

    return m_IsVisible;
}

uint8_t DisplayObject::GetDisplayOpacity() const NN_NOEXCEPT
{
    int opacity = GetOpacity();
    if (m_pOwner != nullptr)
    {
        opacity = opacity * m_pOwner->GetDisplayOpacity() / 255;
    }

    return static_cast<uint8_t>(opacity);
}

bool DisplayObject::IsEnabled() const NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    // オーナーが無効または非フォーカス状態なら常に無効
    if (m_pOwner != nullptr &&
        (!m_pOwner->IsEnabled() || !m_pOwner->IsFocused()))
    {
        return false;
    }

    return Prohibitable::IsEnabled();
}

void DisplayObject::Update() NN_NOEXCEPT
{
    UpdateDrag();
    m_PostUpdateGuiEvent.InvokeAndClear(this);
}

void DisplayObject::RenderWithOffset(const Point2D& offset) NN_NOEXCEPT
{
    const auto originPosition = m_Position;
    m_Position.x += offset.x;
    m_Position.y += offset.y;

    Render();

    m_Position = originPosition;
}

void DisplayObject::ReservePostUpdateEvent(const GuiEventHandlerType& handler) NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_PostUpdateGuiEvent = handler;
}

void DisplayObject::UpdateDrag() NN_NOEXCEPT
{
    if (!IsDraggable())
    {
        return;
    }

    auto* tp = input::TouchPanel::GetInstance();

    Point2D touchPoint;
    tp->GetTouchPoints(&touchPoint.x, &touchPoint.y, 1);

    Rectangle rect;
    GetDraggableArea(&rect);

    if (IsDragging())
    {
        if (tp->IsTouchReleased())
        {
            EndDrag();
        }
        else if (IsDragMoved(touchPoint))
        {
            MoveDrag(touchPoint);
        }
    }
    else
    {
        if (tp->IsTouchTriggered(rect.ToFloat4()))
        {
            BeginDrag(touchPoint);
        }
    }
}

void DisplayObject::InvokeDragEventHandler(GuiEventHandlerType* pHandler) NN_NOEXCEPT
{
    if (pHandler != nullptr)
    {
        pHandler->Invoke(this);
    }
}

void DisplayObject::BeginDrag(const Point2D& point) NN_NOEXCEPT
{
    Draggable::BeginDrag(point);

    m_DragOffset.x = GetAbsoluteX() - point.x;
    m_DragOffset.y = GetAbsoluteY() - point.y;
}

void DisplayObject::MoveDrag(const Point2D& point) NN_NOEXCEPT
{
    Draggable::MoveDrag(point);

    Point2D destination = { { point.x + m_DragOffset.x, point.y + m_DragOffset.y } };
    if (m_pOwner != nullptr)
    {
        destination.x -= m_pOwner->GetAbsoluteX();
        destination.y -= m_pOwner->GetAbsoluteY();
    }

    SetPosition(destination);
}

void DisplayObject::EndDrag() NN_NOEXCEPT
{
    Draggable::EndDrag();
}

}}}  // nns::sgx::gui
