﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include "SimpleGfx_GuiLayout.h"

namespace nns { namespace sgx { namespace gui {

FlowLayoutContainer::FlowLayoutContainer() NN_NOEXCEPT
    : m_Direction(LayoutDirection::Vertical)
    , m_Alignment(LayoutAlignment::Left)
    , m_ItemGap({ { 8, 8 } })
    , m_IsAutoWrap(true)
{
}

void FlowLayoutContainer::AddChild(DisplayObject* pChild) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pChild);
    NNS_SGX_GUI_SCOPED_LOCK;

    UiContainer::AddChild(pChild);
    UpdateLayout();
}

void FlowLayoutContainer::RemoveChild(const DisplayObject* pChild) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pChild);
    NNS_SGX_GUI_SCOPED_LOCK;

    UiContainer::RemoveChild(pChild);
    UpdateLayout();
}

void FlowLayoutContainer::UpdateLayout() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    float destX = 0;
    float destY = 0;

    if (m_Direction == LayoutDirection::Vertical)
    {
        float maxWidth = 0;
        for (auto current = m_Children.begin(); current != m_Children.end(); current++)
        {
            auto pChild = *current;
            NN_ASSERT_NOT_NULL(pChild);

            // 改列
            if (IsAutoWrap() &&
                destY > 0 &&
                destY + pChild->GetHeight() > GetHeight())
            {
                destX   += maxWidth + m_ItemGap.x;
                destY    = 0;
                maxWidth = 0;
            }

            pChild->SetPosition(destX, destY);
            destY   += pChild->GetHeight() + m_ItemGap.y;
            maxWidth = std::max(maxWidth, pChild->GetWidth());
        }
    }
    else
    {
        float maxHeight = 0;
        for (auto current = m_Children.begin(); current != m_Children.end(); current++)
        {
            auto pChild = *current;
            NN_ASSERT_NOT_NULL(pChild);

            // 改行
            if (IsAutoWrap() &&
                destX > 0 &&
                destX + pChild->GetWidth() > GetWidth())
            {
                destX     = 0;
                destY    += maxHeight + m_ItemGap.y;
                maxHeight = 0;
            }

            pChild->SetPosition(destX, destY);
            destX    += pChild->GetWidth() + m_ItemGap.x;
            maxHeight = std::max(maxHeight, pChild->GetHeight());
        }
    }

    UpdateCursorPosition();
}

}}}  // nns::sgx::gui
