﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommon.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   キーヘルプの Z オーダー初期値です。
 */
const int KeyHelpDefaultZ = 10000;

/**
 * @brief   キーヘルプに表示できるキー文字列の最大長です。終端の NULL を含みます。
 */
const int KeyHelpKeyNameLengthMax = 16;

/**
 * @brief   キーヘルプに表示できる説明文の最大長です。終端の NULL を含みます。
 */
const int KeyHelpDescriptionLengthMax = 32;

/**
 * @brief   キーヘルプを扱うクラスです。
 */
class KeyHelp :
    public DisplayObject
{
    NN_DISALLOW_COPY(KeyHelp);
    NN_DISALLOW_MOVE(KeyHelp);

public:
    KeyHelp() NN_NOEXCEPT
        : m_NeedsUpdateTouchArea(false)
        , m_LineColor()
        , m_TextColor()
        , m_HelpList()
        , m_TouchedIndex(-1)
        , m_TouchState(TouchState::Idle)
    {
        SetZ(KeyHelpDefaultZ);
    }

    /**
     * @brief   キーヘルプを追加します。
     */
    void AddHelp(nn::hid::NpadButtonSet buttons, const char* description) NN_NOEXCEPT;

    template <typename Button>
    void AddHelp(const char* description) NN_NOEXCEPT
    {
        AddHelp(Button::Mask, description);
    }

    /**
     * @brief   キーヘルプを追加します。
     *
     * @details タッチイベントは発生しません。
     */
    void AddHelp(const char* key, const char* description) NN_NOEXCEPT
    {
        AddHelp(key, description, nullptr, 0);
    }

    /**
     * @brief   キーヘルプを追加します。
     *
     * @details タッチイベントのユーザー引数には常に 0 が渡されます。
     */
    void AddHelp(const char* key, const char* description, GuiEventHandler handler) NN_NOEXCEPT
    {
        AddHelp(key, description, handler, 0);
    }

    /**
     * @brief   キーヘルプを追加します。
     */
    void AddHelp(const char* key, const char* description, GuiEventHandler handler, uintptr_t userArg) NN_NOEXCEPT;

    /**
     * @brief   キーヘルプをクリアします。
     */
    void Clear() NN_NOEXCEPT;

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    virtual bool UpdateTouchInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual void CancelTouchInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

private:
    /**
     * @brief   タッチ状態です。
     */
    enum class TouchState
    {
        Idle,   //!< 通常
        Press,  //!< 押下状態
    };

    struct HelpItem
    {
        nn::hid::NpadButtonSet  buttons;
        uint32_t key[KeyHelpKeyNameLengthMax];
        uint32_t description[KeyHelpDescriptionLengthMax];
        GuiEventHandlerType touchHandler;
        Rectangle touchArea;
    };

private:
    /**
     * @brief   タッチ可能な領域を更新します。
     */
    void UpdateTouchArea() NN_NOEXCEPT;

private:
    bool                    m_NeedsUpdateTouchArea;
    Color                   m_LineColor;
    Color                   m_TextColor;
    std::vector<HelpItem>   m_HelpList;
    int                     m_TouchedIndex;
    TouchState              m_TouchState;
};

}}}  // nns::sgx::gui
