﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommonDefs.h"
#include "SimpleGfx_GuiObject.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   カーソルの状態です。
 */
enum CursorState
{
    CursorState_Init,           //!< 初期状態
    CursorState_Idle            //!< 通常
};

/**
 * @brief   カーソルの配色です。
 */
struct CursorColorScheme
{
    Color activeBegin;  //!< アクティブ開始
    Color activeEnd;    //!< アクティブ終了
    Color inactive;     //!< 非アクティブ
};

/**
 * @brief   カーソルを扱うクラスです。
 */
class Cursor :
    public DisplayObject
{
    NN_DISALLOW_COPY(Cursor);
    NN_DISALLOW_MOVE(Cursor);

public:
    /**
     * @brief   デフォルトの配色を設定します。
     */
    static void SetDefaultColor(const CursorColorScheme& scheme) NN_NOEXCEPT
    {
        g_DefaultColor = scheme;
    }

public:
    Cursor() NN_NOEXCEPT;

    /**
     * @brief   配色を設定します。
     */
    void SetColor(const CursorColorScheme& scheme) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_Color = scheme;
    }

    /**
     * @brief   決定エフェクトを開始します。
     */
    void StartDecideEffect(const Point2D& offset) NN_NOEXCEPT;

    /**
     * @brief   指定したオブジェクトの位置で決定エフェクトを開始します。
     */
    void StartDecideEffect(const DisplayObject& obj, const Point2D& offset) NN_NOEXCEPT;

    /**
     * @brief   決定エフェクトを停止します。
     */
    void StopDecideEffect() NN_NOEXCEPT;

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   決定エフェクトを描画します。
     */
    void RenderDecideEffect(const Point2D& originOffset) NN_NOEXCEPT;

private:
    /**
     * @brief   デフォルト配色を初期化します。
     */
    static void InitializeDefaultColorScheme() NN_NOEXCEPT;

    /**
     * @brief   必要に応じて初期化処理を行います。
     *
     * @note    静的初期化時に配色が初期化されていない可能性があるため、
     *          最初に更新または描画されるまで初期化を遅延します。
     */
    void InitializeIfNeeded() NN_NOEXCEPT
    {
        if (m_State == CursorState_Init)
        {
            SetupColorScheme();
            m_State = CursorState_Idle;
        }
    }

    /**
     * @brief   配色を初期化します。
     */
    void SetupColorScheme() NN_NOEXCEPT;

private:
    static bool                 g_IsDefaultColorInitialized;
    static CursorColorScheme    g_DefaultColor;       //!< デフォルトの配色

private:
    CursorState         m_State;
    CursorColorScheme   m_Color;                //!< 配色
    float               m_LineWeight;           //!< 線の太さ
    int                 m_Duration;             //!< アニメーションの更新用カウンタ
    int                 m_DecideDuration;       //!< 決定アニメーションの更新用カウンタ
    Rectangle           m_DecideEffectRect;     //!< 決定アニメーションの表示位置・サイズ
};

}}}  // nns::sgx::gui
