﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/util/util_FormatString.h>

#include "../SimpleGfx_Api.h"
#include "SimpleGfx_GuiButtonSkin.h"
#include "../detail/SimpleGfx_Json.h"

namespace nns { namespace sgx { namespace gui {

ButtonSkin::ButtonSkin() NN_NOEXCEPT
    : m_BaseImage()
    , m_BaseOffset()
    , m_BaseSizeOffset()
    , m_ContentsPadding()
    , m_NormalTextColor()
{
}

ButtonSkin::~ButtonSkin() NN_NOEXCEPT
{
    Destroy();
}

void ButtonSkin::SetDefault() NN_NOEXCEPT
{
    m_BaseImage = ImageData();
    m_BaseOffset.Set(-3, -3);
    m_BaseSizeOffset.Set(6, 6);
    m_ContentsPadding.Set(24, 8, 24, 8);
    m_NormalTextColor = Colors::Shadow();
}

ResultCode ButtonSkin::Load(const char* filename) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(filename);

    std::unique_ptr<detail::JsonDocument> json(new detail::JsonDocument());

    // スキンファイルを開く
    {
        auto result = json->Open(filename);
        if (result != detail::JsonResult::Success)
        {
            return ResultCode::ReadFailed;
        }
    }

    SetDefault();

    nne::rapidjson::Document& document = json->GetDocument();

    // スキン画像の読み込み
    if (document.HasMember("baseImage"))
    {
        auto& baseImage = document["baseImage"];
        char imagePath[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(imagePath, sizeof(imagePath), "%s:/%s", "rom", baseImage.GetString());  // XXX: マウント名固定

        auto result = LoadImage(&m_BaseImage, imagePath);
        if (result != ResultCode::Success)
        {
            return result;
        }
    }
    else
    {
        return ResultCode::ReadFailed;
    }

    // パラメータの取得
    if (document.HasMember("baseOffset"))
    {
        auto& baseOffset = document["baseOffset"];
        m_BaseOffset.x = baseOffset["x"].GetInt();
        m_BaseOffset.y = baseOffset["y"].GetInt();
    }

    if (document.HasMember("baseSizeOffset"))
    {
        auto& baseSizeOffset = document["baseSizeOffset"];
        m_BaseSizeOffset.width  = baseSizeOffset["width"].GetInt();
        m_BaseSizeOffset.height = baseSizeOffset["height"].GetInt();
    }

    if (document.HasMember("contentsPadding"))
    {
        auto& contentsPadding = document["contentsPadding"];
        m_ContentsPadding.left   = contentsPadding["left"].GetInt();
        m_ContentsPadding.top    = contentsPadding["top"].GetInt();
        m_ContentsPadding.right  = contentsPadding["right"].GetInt();
        m_ContentsPadding.bottom = contentsPadding["bottom"].GetInt();
    }

    return ResultCode::Success;
}

void ButtonSkin::Destroy() NN_NOEXCEPT
{
    if (m_BaseImage.IsValid())
    {
        DestroyImage(&m_BaseImage);
    }
}

}}}  // nns::sgx::gui
