﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommonDefs.h"
#include "SimpleGfx_GuiButtonBase.h"
#include "SimpleGfx_GuiButtonSkin.h"
#include "SimpleGfx_GuiUtil.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   標準のボタンを扱うクラスです。
 */
class Button : public ButtonBase
{
    NN_DISALLOW_COPY(Button);
    NN_DISALLOW_MOVE(Button);

public:
    static void LoadDefaultSkin() NN_NOEXCEPT;

public:
    Button() NN_NOEXCEPT;

    virtual Rectangle GetHitArea() const NN_NOEXCEPT NN_OVERRIDE;

    virtual Size GetClientAreaSize() const NN_NOEXCEPT;

    virtual Rectangle GetClientArea() const NN_NOEXCEPT;

    /**
     * @brief   スキンを設定します。
     */
    void SetSkin(const ButtonSkin& skin) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_pSkin = &skin;
        SetState(ButtonState_Idle);
    }

    /**
     * @brief   テキストの整列位置を設定します。
     */
    void SetTextAlignment(nns::sgx::TextAlignment alignment) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_TextAlignment = alignment;
    }

    /**
     * @brief   ボタンのテキストを UTF-8 で設定します。
     */
    void SetText(const char* text) NN_NOEXCEPT;

    /**
     * @brief   ボタンのテキストを UTF-32 で設定します。
     */
    void SetText(const uint32_t* text) NN_NOEXCEPT;

    /**
     * @brief   テキストの描画サイズを設定します。
     */
    void SetTextSize(float size) NN_NOEXCEPT
    {
        m_TextSize = size;
    }

    /**
     * @brief   コンテンツ領域の描画タイミングで呼び出されるハンドラを登録します。
     *
     * @details ユーザー引数には常に 0 が渡されます。
     */
    void SetRenderContentsHandler(const GuiEventHandler& handler) NN_NOEXCEPT
    {
        SetRenderContentsHandler(handler, 0);
    }

    /**
     * @brief   コンテンツ領域の描画タイミングで呼び出されるハンドラを登録します。
     *
     * @details 登録したハンドラは、RenderContents の後に呼び出されます。
     *          新たなクラスを定義せずに、一部の描画のみ変更したい場合に使用できます。
     */
    void SetRenderContentsHandler(const GuiEventHandler& handler, uintptr_t argument) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_HandlerForRenderContents.handler  = handler;
        m_HandlerForRenderContents.argument = argument;
    }

    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   コンテンツ領域を描画します。継承先で上書きして使用します。
     */
    virtual void RenderContents() NN_NOEXCEPT {}

protected:
    Color GetTextColor() const NN_NOEXCEPT
    {
        return m_pSkin->m_NormalTextColor;
    }

private:
    static ButtonSkin       g_DefaultSkin;                  //!< デフォルトのスキン

private:
    const ButtonSkin*       m_pSkin;                        //!< スキン
    nns::sgx::TextAlignment m_TextAlignment;                //!< テキストの位置揃え
    uint32_t                m_Text[ButtonTextLengthMax];    //!< 表示する文字列
    float                   m_TextSize;                     //!< 文字の描画サイズ
    GuiEventHandlerType     m_HandlerForRenderContents;     //!< コンテンツ領域の描画ハンドラ
};

}}}  // nns::sgx::gui
