﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>

#include "../SimpleGfx_Types.h"
#include "../SimpleGfx_TypesInternal.h"

namespace nns { namespace sgx { namespace detail {

/**
 * @brief   Shader Attribute を取得する。失敗時は Assert
 */
NN_FORCEINLINE
GLuint GetAttribLocationWithAssert(const ShaderPack& shader, const char* name) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(name);

    GLint index = glGetAttribLocation(shader.programId, name);
    NN_ASSERT_GREATER_EQUAL(index, 0);
    return static_cast<GLuint>(index);
}

/**
 * @brief   シェーダ―の生成
 */
void CreateShader(
    ShaderPack* pOutShader,
    const GLchar* const* vertexProgram,
    const GLchar* const* fragmentProgram) NN_NOEXCEPT;

/**
 * @brief   シェーダ―の破棄
 */
void DeleteShader(const ShaderPack& shader) NN_NOEXCEPT;

/**
 * @brief   シェーダ―の登録
 */
void RegisterShader(
    ShaderPack* pSimpleShader,
    ShaderPack* pTextureShader) NN_NOEXCEPT;

/**
 * @brief   使用するシェーダの切り替え
 *
 * @retval  true    シェーダが変更された
 * @retval  false   既に指定されたシェーダに切り替わっている
 */
bool ChangeShader(ShaderType type) NN_NOEXCEPT;

/**
 * @brief   現在のキャンバスサイズを取得
 */
Size GetCanvasSize() NN_NOEXCEPT;

/**
 * @brief   キャンバスサイズを変更
 */
void SetCanvasSize(const Size& size) NN_NOEXCEPT;

/**
 * @brief   現在適用されている描画範囲を取得
 */
void GetCurrentRenderArea(Rectangle* pOutArea, Point2D* pOutOffset) NN_NOEXCEPT;
void GetCurrentRenderArea(Rectangle* pOutArea) NN_NOEXCEPT;

/**
 * @brief   描画領域の制限
 */
void SetRenderArea(const Rectangle& rect, const Point2D& offset) NN_NOEXCEPT;
void SetRenderArea(float x, float y, float width, float height) NN_NOEXCEPT;

/**
 * @brief   描画領域の初期化
 */
void ResetRenderArea() NN_NOEXCEPT;

/**
 * @brief   描画範囲を指定した矩形内に制限する
 */
void ApplyRenderArea(const Rectangle& rect) NN_NOEXCEPT;

/**
 * @brief   制限した描画範囲を元に戻す
 */
void RestoreRenderArea() NN_NOEXCEPT;

}}}  // nns::sgx::detail
