﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_Color.h>
#include <nn/util/util_MathTypes.h>
#include <nn/util/util_Vector.h>

#include "detail/GraphicsHelper.h"

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    #if defined(USE_OEPGL_ES)
        #include <GLES3/gl32.h>
        #define DECL_SPEC GL_APIENTRY
    #else
        #define GL_GLEXT_PROTOTYPES
        #include <GL/gl.h>
        #define DECL_SPEC APIENTRY
    #endif  // if defined(USE_OEPGL_ES)
#else
    #include <GL/glew.h>
    #define DECL_SPEC GLAPIENTRY
#endif


namespace nns { namespace sgx {

/**
 * @brief   リソースデータを扱うファイルシステムのマウント名
 */
const char* const RomMountName = "rom";

/**
 * @brief   リザルトコード
 */
enum class ResultCode
{
    Success,            //!< 成功

    NotSupported,       //!< サポート外の操作
    OutOfMemory,        //!< メモリ (バッファ) 不足
    ReadFailed,         //!< ファイルの読み込みに失敗
    DecodeFailed        //!< データのデコードに失敗
};

/**
 * @brief   描画時の原点位置
 */
enum class OriginPosition
{
    Default,    //!< 中央原点、Y 座標上向き
    TopLeft     //!< 左上原点、Y 座標下向き
};

/**
 * @brief   文字列描画時の整列位置
 */
enum class TextAlignment
{
    Left,       //!< 左寄せ
    Center,     //!< 中央寄せ
    Right       //!< 右寄せ
};
/**
 * @brief   ブレンドモード
 */
enum class BlendMode
{
    Normal,     //!< 通常
    Add,        //!< 加算
    Subtract,   //!< 減算
    Multiple,   //!< 乗算
    Screen,     //!< スクリーン
    Xor,        //!< XOR
    Inverse     //!< 反転
};

/**
 * @brief   ステンシルテストの方法
 */
enum class StencilFunc
{
    Never,
    Less,
    LessEqual,
    Greater,
    GreaterEqual,
    Equal,
    NotEqual,
    Always
};

/**
 * @brief   画像フォーマット
 */
enum class ImageFormat
{
    Raw,        //!< 生ピクセル値
    Bitmap,     //!< ビットマップ
    Jpeg,       //!< JPEG
    Png         //!< PNG
};

/**
 * @brief   ピクセルフォーマット
 */
enum class PixelFormat
{
    R8G8B8A8,   //!< RGBA 各 8 bit
    U8,         //!< 8 bit グレースケール
};

/**
 * @brief   2D 座標
 */
union Point2D
{
    struct
    {
        float x;    //!< X 座標
        float y;    //!< Y 座標
    };
    float v[2];

    void Set(float x, float y) NN_NOEXCEPT
    {
        this->x = x;
        this->y = y;
    }

    ::nn::util::Float2 ToFloat2() const NN_NOEXCEPT
    {
        return { { { x, y } } };
    }

    /**
     * @brief   2D 座標をベクトルと見なしたときの大きさの 2 乗を計算
     */
    float CalcSqrMagnitude() const NN_NOEXCEPT
    {
        return x * x + y * y;
    }

    /**
     * @brief   2D 座標をベクトルと見なしたときの大きさを計算
     */
    float CalcMaginitude() const NN_NOEXCEPT
    {
        return std::sqrtf(CalcSqrMagnitude());
    }

    Point2D operator-() NN_NOEXCEPT
    {
        return { { -x, -y } };
    }

    const Point2D operator+(const Point2D& rhs) const NN_NOEXCEPT
    {
        return Point2D { { x + rhs.x, y + rhs.y } };
    }

    const Point2D operator-(const Point2D& rhs) const NN_NOEXCEPT
    {
        return Point2D { { x - rhs.x, y - rhs.y } };
    }

    Point2D& operator+=(const Point2D& rhs) NN_NOEXCEPT
    {
        x += rhs.x;
        y += rhs.y;
        return *this;
    }

    Point2D& operator-=(const Point2D& rhs) NN_NOEXCEPT
    {
        x -= rhs.x;
        y -= rhs.y;
        return *this;
    }
};

/**
 * @brief   3D 座標
 */
union Point3D
{
    struct
    {
        float x;    //!< X 座標
        float y;    //!< Y 座標
        float z;    //!< Z 座標
    };
    float v[3];

    static Point3D FromVector3f(const ::nn::util::Vector3f& vector) NN_NOEXCEPT
    {
        return { { vector.GetX(), vector.GetY(), vector.GetZ() } };
    }

    ::nn::util::Float3 ToFloat3() const NN_NOEXCEPT
    {
        return { { { x, y, z } } };
    }

    ::nn::util::Vector3f ToVector3f() const NN_NOEXCEPT
    {
        return ::nn::util::Vector3f(x, y, z);
    }

    /**
     * @brief   3D 座標をベクトルと見なしたときの大きさの 2 乗を計算
     */
    float CalcSqrMagnitude() const NN_NOEXCEPT
    {
        return x * x + y * y + z * z;
    }

    /**
     * @brief   3D 座標をベクトルと見なしたときの大きさを計算
     */
    float CalcMaginitude() const NN_NOEXCEPT
    {
        return std::sqrtf(CalcSqrMagnitude());
    }

    /**
     * @brief   符号を反転させた座標を取得します。
     */
    Point3D Invert() const NN_NOEXCEPT
    {
        return { { -x, -y, -z } };
    }
};

/**
 * @brief   サイズ
 */
union Size
{
    struct
    {
        float width;    //!< 幅
        float height;   //!< 高さ
    };
    float v[2];

    void Set(float width, float height) NN_NOEXCEPT
    {
        this->width  = width;
        this->height = height;
    }

    bool IsEmpty() const NN_NOEXCEPT
    {
        return width <= 0.0f || height <= 0.0f;
    }

    ::nn::util::Float2 ToFloat2() const NN_NOEXCEPT
    {
        return { { { width, height } } };
    }
};

/**
 * @brief   矩形
 */
union Rectangle
{
    struct
    {
        float x;            //!< X 座標
        float y;            //!< Y 座標
        float width;        //!< 幅
        float height;       //!< 高さ
    };
    struct
    {
        Point2D position;   //!< 位置
        Size    size;       //!< サイズ
    };
    float v[4];

    void Set(float x, float y, float width, float height) NN_NOEXCEPT
    {
        position.Set(x, y);
        size.Set(width, height);
    }

    ::nn::util::Float4 ToFloat4() const NN_NOEXCEPT
    {
        return { { { x, y, width, height } } };
    }
};

/**
 * @brief   パディング
 */
struct Padding
{
    float left;
    float top;
    float right;
    float bottom;

    void Set(float left, float top, float right, float bottom) NN_NOEXCEPT
    {
        this->left   = left;
        this->top    = top;
        this->right  = right;
        this->bottom = bottom;
    }

    ::nn::util::Float4 ToFloat4() const NN_NOEXCEPT
    {
        return { { { left, top, right, bottom } } };
    }
};

/**
 * @brief   四角形
 */
struct Quadrangle
{
    Point2D points[4];
};

/**
 * @brief   色
 */
class Color : public ::nn::util::Color4u8
{
public:
    /**
     * @brief   コンストラクタです。
     *
     * @details 各成分の値は初期化されません。
     */
    Color() NN_NOEXCEPT
    {}

    /**
     * @brief   引数付きコンストラクタです。
     *
     * @param[in]   red     カラーの R 要素
     * @param[in]   green   カラーの G 要素
     * @param[in]   blue    カラーの B 要素
     * @param[in]   alpha   カラーの A 要素
     */
    Color(uint8_t red, uint8_t green, uint8_t blue, uint8_t alpha = nn::util::Color4u8::ElementMax) NN_NOEXCEPT
        : ::nn::util::Color4u8(red, green, blue, alpha)
    {}

    NN_IMPLICIT Color(const nn::util::Color4u8& color) NN_NOEXCEPT
        : ::nn::util::Color4u8(color)
    {}

    /**
     * @brief   アルファ値を変更した色を取得
     */
    Color BlendAlpha(uint8_t alpha) const NN_NOEXCEPT
    {
        return Color(GetR(), GetG(), GetB(), static_cast<uint8_t>(GetA() * alpha / 255));
    }
};

/**
 * @brief   フォントの描画に関する状態
 */
struct FontContext
{
    Color color;
    float offsetX;
    float scaleX;
    float scaleY;

    /**
     * @brief   状態の初期化
     */
    void Initialize() NN_NOEXCEPT;
};

/**
 * @brief   スコープを抜ける際に、生成時のフォントコンテキストを復元するクラス
 */
class ScopedFontContextSaver final
{
    NN_DISALLOW_COPY(ScopedFontContextSaver);
    NN_DISALLOW_MOVE(ScopedFontContextSaver);

public:
    ScopedFontContextSaver() NN_NOEXCEPT;

    ~ScopedFontContextSaver() NN_NOEXCEPT;

private:
    FontContext m_Context;
};

/**
 * @brief   画像データ
 */
struct ImageData
{
    GLuint      textureId;      //!< テクスチャ ID
    Size        size;           //!< 画像のサイズ
    ImageFormat imageFormat;    //!< 画像フォーマット
    PixelFormat pixelFormat;    //!< ピクセルフォーマット
    char*       pPixelData;     //!< ピクセルデータ

    mutable bool isDirty;       //!< ピクセルデータの変更済みフラグ
    bool         isFlippedH;    //!< 左右反転されているか
    bool         isFlippedV;    //!< 上下反転されているか

    bool IsValid() const NN_NOEXCEPT
    {
        return pPixelData != nullptr;
    }
};

/**
 * @brief   初期化パラメータ
 */
struct InitializationParameters
{
    bool            isInitializeFs;     //!< ファイルシステムを初期化するか
    int             vertexCountMax;     //!< 最大頂点数
    OriginPosition  origin;             //!< 原点位置
    Size            canvasSize;         //!< 描画領域のサイズ
    int             baseFontSize;       //!< 基準となるフォントサイズ

    /**
     * @brief   初期化パラメータのデフォルト値を生成
     *
     * @details 各パラメータにデフォルト値を格納した状態で生成します。
     *          下記の値が格納されます。
     *
     *          - isInitializeFs = true
     *          - vertexCountMax = 32768
     *          - origin         = OriginPosition::TopLeft
     *          - canvasSize     = { 1280, 720 }
     *          - baseFontSize   = 40
     */
    static InitializationParameters CreateDefault() NN_NOEXCEPT;
};

/**
 * @brief   色のプリセット
 */
class Colors final
{
    NN_DISALLOW_COPY(Colors);
    NN_DISALLOW_MOVE(Colors);

public:
    static const Color& Transparent() NN_NOEXCEPT
    {
        return ConstTransparent;
    }

    static const Color& Black() NN_NOEXCEPT
    {
        return ConstBlack;
    }

    static const Color& Shadow() NN_NOEXCEPT
    {
        return ConstShadow;
    }

    static const Color& Smoke() NN_NOEXCEPT
    {
        return ConstSmoke;
    }

    static const Color& DimGray() NN_NOEXCEPT
    {
        return ConstDimGray;
    }

    static const Color& Gray() NN_NOEXCEPT
    {
        return ConstGray;
    }

    static const Color& DarkGray() NN_NOEXCEPT
    {
        return ConstDarkGray;
    }

    static const Color& Silver() NN_NOEXCEPT
    {
        return ConstSilver;
    }

    static const Color& LightGray() NN_NOEXCEPT
    {
        return ConstLightGray;
    }

    static const Color& WhiteSmoke() NN_NOEXCEPT
    {
        return ConstWhiteSmoke;
    }

    static const Color& White() NN_NOEXCEPT
    {
        return ConstWhite;
    }

    static const Color& Red() NN_NOEXCEPT
    {
        return ConstRed;
    }

    static const Color& DarkRed() NN_NOEXCEPT
    {
        return ConstDarkRed;
    }

    static const Color& Crymson() NN_NOEXCEPT
    {
        return ConstCrymson;
    }

    static const Color& NeonRed() NN_NOEXCEPT
    {
        return ConstNeonRed;
    }

    static const Color& Lime() NN_NOEXCEPT
    {
        return ConstLime;
    }

    static const Color& Green() NN_NOEXCEPT
    {
        return ConstGreen;
    }

    static const Color& Blue() NN_NOEXCEPT
    {
        return ConstBlue;
    }

    static const Color& Yellow() NN_NOEXCEPT
    {
        return ConstYellow;
    }

    static const Color& Magenta() NN_NOEXCEPT
    {
        return ConstMagenta;
    }

    static const Color& Cyan() NN_NOEXCEPT
    {
        return ConstCyan;
    }

    static const Color& Orange() NN_NOEXCEPT
    {
        return ConstOrange;
    }

    static const Color& Gold() NN_NOEXCEPT
    {
        return ConstGold;
    }

    static const Color& GoldenRod() NN_NOEXCEPT
    {
        return ConstGoldenRod;
    }

    static const Color& DarkGoldenRod() NN_NOEXCEPT
    {
        return ConstDarkGoldenRod;
    }

    static const Color& Violet() NN_NOEXCEPT
    {
        return ConstViolet;
    }

    static const Color& LimeGreen() NN_NOEXCEPT
    {
        return ConstLimeGreen;
    }

    static const Color& PaleGreen() NN_NOEXCEPT
    {
        return ConstPaleGreen;
    }

    static const Color& MidNightBlue() NN_NOEXCEPT
    {
        return ConstMidNightBlue;
    }

    static const Color& Navy() NN_NOEXCEPT
    {
        return ConstNavy;
    }

    static const Color& MediumBlue() NN_NOEXCEPT
    {
        return ConstMediumBlue;
    }

    static const Color& RoyalBlue() NN_NOEXCEPT
    {
        return ConstRoyalBlue;
    }

    static const Color& NeonBlue() NN_NOEXCEPT
    {
        return ConstNeonBlue;
    }

private:
    Colors() NN_NOEXCEPT {}
    ~Colors() NN_NOEXCEPT {}

private:
    static const Color ConstTransparent;
    static const Color ConstBlack;
    static const Color ConstShadow;
    static const Color ConstSmoke;
    static const Color ConstDimGray;
    static const Color ConstGray;
    static const Color ConstDarkGray;
    static const Color ConstSilver;
    static const Color ConstLightGray;
    static const Color ConstWhiteSmoke;
    static const Color ConstWhite;
    static const Color ConstRed;
    static const Color ConstDarkRed;
    static const Color ConstCrymson;
    static const Color ConstNeonRed;
    static const Color ConstLime;
    static const Color ConstGreen;
    static const Color ConstBlue;
    static const Color ConstYellow;
    static const Color ConstMagenta;
    static const Color ConstCyan;
    static const Color ConstOrange;
    static const Color ConstGold;
    static const Color ConstGoldenRod;
    static const Color ConstDarkGoldenRod;
    static const Color ConstViolet;
    static const Color ConstLimeGreen;
    static const Color ConstPaleGreen;
    static const Color ConstMidNightBlue;
    static const Color ConstNavy;
    static const Color ConstMediumBlue;
    static const Color ConstRoyalBlue;
    static const Color ConstNeonBlue;
};

/**
 * @brief   デフォルトの描画領域サイズ
 */
const Size DefaultCanvasSize = { { 1280, 720 } };

/**
 * @brief   外字コード
 */
enum ExtensionCharacter : uint16_t
{
    // ボタン
    ExtensionCharacter_ButtonA = 0xE0A0u,
    ExtensionCharacter_ButtonB,
    ExtensionCharacter_ButtonX,
    ExtensionCharacter_ButtonY,
    ExtensionCharacter_ButtonL,
    ExtensionCharacter_ButtonR,
    ExtensionCharacter_ButtonZl,
    ExtensionCharacter_ButtonZr,
    ExtensionCharacter_ButtonSl,
    ExtensionCharacter_ButtonSr,
    ExtensionCharacter_ButtonArrow,
    ExtensionCharacter_ButtonRight,
    ExtensionCharacter_ButtonDown,
    ExtensionCharacter_ButtonUp,
    ExtensionCharacter_ButtonLeft,
    ExtensionCharacter_ButtonArrowUp,
    ExtensionCharacter_ButtonArrowDown,
    ExtensionCharacter_ButtonArrowLeft,
    ExtensionCharacter_ButtonArrowRight,
    ExtensionCharacter_ButtonPlus,
    ExtensionCharacter_ButtonMinus,
    ExtensionCharacter_ButtonJoyPlus,
    ExtensionCharacter_ButtonJoyMinus,
    ExtensionCharacter_ButtonPower,
    ExtensionCharacter_ButtonPower2,
    ExtensionCharacter_ButtonHome,
    ExtensionCharacter_ButtonCapture,

    ExtensionCharacter_ButtonDownRight = 0xE0CAu,
    ExtensionCharacter_ButtonLeftRight,
    ExtensionCharacter_ButtonUpRight,
    ExtensionCharacter_ButtonDownLeft,
    ExtensionCharacter_ButtonUpDown,
    ExtensionCharacter_ButtonUpLeft,
    ExtensionCharacter_ButtonCross,
    ExtensionCharacter_ButtonCrossUp,
    ExtensionCharacter_ButtonCrossDown,
    ExtensionCharacter_ButtonCrossLeft,
    ExtensionCharacter_ButtonCrossRight,
    ExtensionCharacter_ButtonCrossUpDown,
    ExtensionCharacter_ButtonCrossLeftRight,

    // ボタン (色反転)
    ExtensionCharacter_ButtonAInvert = 0xE0E0u,
    ExtensionCharacter_ButtonBInvert,
    ExtensionCharacter_ButtonXInvert,
    ExtensionCharacter_ButtonYInvert,
    ExtensionCharacter_ButtonLInvert,
    ExtensionCharacter_ButtonRInvert,
    ExtensionCharacter_ButtonZlInvert,
    ExtensionCharacter_ButtonZrInvert,
    ExtensionCharacter_ButtonSlInvert,
    ExtensionCharacter_ButtonSrInvert,
    ExtensionCharacter_ButtonArrowInvert,
    ExtensionCharacter_ButtonArrowUpInvert,
    ExtensionCharacter_ButtonArrowDownInvert,
    ExtensionCharacter_ButtonArrowLeftInvert,
    ExtensionCharacter_ButtonArrowRightInvert,
    ExtensionCharacter_ButtonPlusInvert,
    ExtensionCharacter_ButtonMinusInvert,
    ExtensionCharacter_ButtonJoyPlusInvert,
    ExtensionCharacter_ButtonJoyMinusInvert,
    ExtensionCharacter_ButtonPowerInvert,
    ExtensionCharacter_ButtonHomeInvert,
    ExtensionCharacter_ButtonCaptureInvert,

    ExtensionCharacter_ButtonCrossInvert = 0xE110u,
    ExtensionCharacter_ButtonCrossUpInvert,
    ExtensionCharacter_ButtonCrossDownInvert,
    ExtensionCharacter_ButtonCrossLeftInvert,
    ExtensionCharacter_ButtonCrossRightInvert,
    ExtensionCharacter_ButtonCrossUpDownInvert,
    ExtensionCharacter_ButtonCrossLeftRightInvert,

    // スティック
    ExtensionCharacter_Stick = 0xE0C0u,
    ExtensionCharacter_StickL,
    ExtensionCharacter_StickR,
    ExtensionCharacter_StickPush,
    ExtensionCharacter_StickPushL,
    ExtensionCharacter_StickPushR,
    ExtensionCharacter_StickRollLeftL,
    ExtensionCharacter_StickRollLeftR,
    ExtensionCharacter_StickRollRightL,
    ExtensionCharacter_StickRollRightR,

    ExtensionCharacter_StickInvert = 0xE100u,
    ExtensionCharacter_StickLInvert,
    ExtensionCharacter_StickRInvert,
    ExtensionCharacter_StickPushInvert,
    ExtensionCharacter_StickPushLInvert,
    ExtensionCharacter_StickPushRInvert,

    // デバイス
    ExtensionCharacter_Handheld = 0xE121u,
    ExtensionCharacter_JoyDual,
    ExtensionCharacter_JoyLeft,
    ExtensionCharacter_JoyRight,
    ExtensionCharacter_JoyLeftWithButton,
    ExtensionCharacter_JoyRightWithButton,
    ExtensionCharacter_JoyHorizontal,
    ExtensionCharacter_JoyLeftHorizontal,
    ExtensionCharacter_JoyRightHorizontal,
    ExtensionCharacter_JoyConGrip,
    ExtensionCharacter_JoyConGripEmpty,
    ExtensionCharacter_SwitchProController,

    // その他アイコン
    ExtensionCharacter_OceanSettings = 0xE130u,
    ExtensionCharacter_OceanFriend,
    ExtensionCharacter_OceanGameNews,
    ExtensionCharacter_OceanShop,
    ExtensionCharacter_OceanAlbum,
    ExtensionCharacter_OceanFullLauncher,
    ExtensionCharacter_OceanController,
    ExtensionCharacter_Mail,
    ExtensionCharacter_SmartDevice,
    ExtensionCharacter_Pc,
    ExtensionCharacter_Tv,
    ExtensionCharacter_Headphone,
    ExtensionCharacter_Speaker,

    ExtensionCharacter_ExclamationCircle = 0xE140u,
    ExtensionCharacter_ExclamationSquare,
    ExtensionCharacter_QuestionCircle,
    ExtensionCharacter_LineArrowUp,
    ExtensionCharacter_LineArrowDown,
    ExtensionCharacter_LineArrowLeft,
    ExtensionCharacter_LineArrowRight,
    ExtensionCharacter_ArrowUp,
    ExtensionCharacter_ArrowDown,
    ExtensionCharacter_ArrowLeft,
    ExtensionCharacter_ArrowRight,
    ExtensionCharacter_Check,
    ExtensionCharacter_Cross,
    ExtensionCharacter_InformationCircle,
    ExtensionCharacter_ExclamationCircleInvert,
    ExtensionCharacter_ExclamationSquareInvert,
    ExtensionCharacter_QuestionCircleInvert,
    ExtensionCharacter_InformationCircleInvert
};

}}  // nns::sgx
