﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <vector>

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/ae.h>

#include "SceneBase.h"
#include "SceneManager.h"
#include "../sgx/SimpleGfx_PostEffect.h"
#include "../util/util.h"

// LibraryApplet として動作した場合を想定するモード
//#define ENABLE_LA_MODE

#define VERBOSE

#ifdef VERBOSE
#define NNS_SGX_SCENE_LOG(...)  NN_LOG("[SgxScene] " __VA_ARGS__);
#else
#define NNS_SGX_SCENE_LOG(...)  static_cast<void>(0)
#endif

namespace
{

}  // anonymous

namespace nns { namespace hid { namespace scene {

const int DefaultSceneStacksize = 64;

const int DefaultTransitionFrameCount = 10;

nn::os::Mutex       SceneManager::g_Mutex(true);
bool                SceneManager::g_IsInitialized = false;
bool                SceneManager::g_IsRenderFrozen = false;
SceneListType       SceneManager::g_SceneList({});
SceneListType       SceneManager::g_TerminatedSceneList({});
TerminateCallback   SceneManager::g_TerminateCallback = nullptr;

nns::sgx::ImageData SceneManager::g_BackgroundImage;
nns::sgx::ImageData SceneManager::g_TransitionImage;
nns::sgx::Color     SceneManager::g_TransitionColor;
int                 SceneManager::g_TransitionFrameCount = DefaultTransitionFrameCount;
int                 SceneManager::g_TransitionDuration = 0;

void SceneManager::Initialize() NN_NOEXCEPT
{
    NN_ASSERT(!g_IsInitialized);

    g_IsInitialized = true;

    nn::oe::Initialize();

    g_IsRenderFrozen = false;
    g_SceneList.clear();
    g_SceneList.reserve(DefaultSceneStacksize);
    g_TerminatedSceneList.clear();
    g_TerminatedSceneList.reserve(DefaultSceneStacksize);
    g_TerminateCallback = nullptr;

    nns::sgx::CreateImage(&g_BackgroundImage, nns::sgx::PixelFormat::R8G8B8A8, 1280, 720);
    g_TransitionImage      = nns::sgx::ImageData();
    g_TransitionColor      = nns::sgx::Colors::Black();
    g_TransitionFrameCount = DefaultTransitionFrameCount;
    g_TransitionDuration   = 0;

#if defined(ENABLE_LA_MODE)
    // 背景画像を作成
    const size_t imageSize = 1280 * 720 * 4;
    bool isScreenShotEnabled;
    char* pixelData = new char[imageSize];
    nn::ae::GetLastForegroundCaptureBuffer(&isScreenShotEnabled, pixelData, imageSize);
    if (isScreenShotEnabled)
    {
        nns::sgx::UpdateImageByMemory(&g_BackgroundImage, pixelData, imageSize, nns::sgx::ImageFormat::Raw);

        // 初期トランジションにも流用
        g_TransitionDuration = 1;
        nns::sgx::CreateImage(&g_TransitionImage, nns::sgx::PixelFormat::R8G8B8A8, 1280, 720);
        nns::sgx::UpdateImageByMemory(&g_TransitionImage, pixelData, imageSize, nns::sgx::ImageFormat::Raw);
    }
    delete[] pixelData;
#endif  // if defined(ENABLE_LA_MODE)
}

void SceneManager::Finalize() NN_NOEXCEPT
{
    NN_ASSERT(g_IsInitialized);

    g_IsInitialized = false;

    // シーンの全解放
    TerminateTo(0);

    nns::sgx::DestroyImage(&g_BackgroundImage);
}

bool SceneManager::IsExitRequested() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);

    NN_ASSERT(g_IsInitialized);

    if (IsEmpty())
    {
        return true;
    }

    return false;
}

SceneBase* SceneManager::GetActiveScene() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);

    NN_ASSERT(g_IsInitialized);

    if (IsEmpty())
    {
        return nullptr;
    }

    return g_SceneList.back();
}

void SceneManager::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);

    NN_ASSERT(g_IsInitialized);

    UpdateTransitionEffect();

    if (IsOutingScene())
    {
        return;
    }

    CheckSceneTerminated();

    ProcessActiveScene([](SceneBase* pScene)
    {
        pScene->Update();
    });
}

void SceneManager::Render() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);

    NN_ASSERT(g_IsInitialized);

    if (g_IsRenderFrozen)
    {
        return;
    }

    ProcessActiveScene([](SceneBase* pScene)
    {
        nns::sgx::SetClearColor(pScene->GetBackgroundColor());
        nns::sgx::BeginRender();
        RenderBackground();
        pScene->Render();
        //nns::sgx::EndRenderNoSwap();
    });

    if (IsTransiting())
    {
        // 遷移エフェクト中はキャプチャ画像を上書き
        auto prevOpacity = nns::sgx::GetImageOpacity();
        int diff = std::max(g_TransitionFrameCount - g_TransitionDuration, 0);
        nns::sgx::SetImageOpacity(diff * 1.0f / g_TransitionFrameCount);
        nns::sgx::DrawImage(g_TransitionImage, 0, 0);
        nns::sgx::SetImageOpacity(prevOpacity);
    }

    nns::sgx::EndRender();
    //nns::sgx::SwapBuffer();
}

void SceneManager::RenderBackground() NN_NOEXCEPT
{
#if defined(ENABLE_LA_MODE)
#if 0
    // キャプチャ画像をぼかして描画
    auto prevOpacity = nns::sgx::GetImageOpacity();
    nns::sgx::SetImageOpacity(0.12f);
    nns::sgx::DrawImage(g_BackgroundImage, 0, 0);
    nns::sgx::SetImageOpacity(prevOpacity);

    //nns::sgx::ApplyBlurEffect(2, 2);
#else
    // 画面全体をグレーアウト
    {
        nns::sgx::Rectangle rect;
        rect.x = rect.y = 0;
        nns::sgx::GetCanvasSize(&rect.size);

        nns::sgx::FillRectangle(rect, nns::sgx::Colors::Smoke().BlendAlpha(192));
    }
#endif
#endif  // if defined(ENABLE_LA_MODE)
}

void SceneManager::Push(SceneBase* pNewScene) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pNewScene);

    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);

    NN_ASSERT(g_IsInitialized);

    // 現在のシーンを非アクティブ化
    ProcessActiveScene([](SceneBase* pScene)
    {
        pScene->Deactivate();
    });

    // 遷移先のシーンをアクティブ化してスタックに積む
    if (!pNewScene->IsInitialized())
    {
        pNewScene->Initialize();
    }
    pNewScene->Activate();
    g_SceneList.push_back(pNewScene);

    if (g_SceneList.size() > 1)
    {
        // 遷移エフェクトを開始
        StartTransitionEffect();
    }
}

void SceneManager::Goto(SceneBase* pNewScene) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pNewScene);

    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);

    NN_ASSERT(g_IsInitialized);

    TerminateTo(0);
    Push(pNewScene);
}

SceneBase* SceneManager::Pop() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);

    NN_ASSERT(g_IsInitialized);

    // 現在のシーンを終了して前のシーンに戻す
    auto pActiveScene = ProcessActiveScene([](SceneBase* pScene)
    {
        pScene->Deactivate();
        TerminateScene(pScene);
    });

    g_TerminatedSceneList.push_back(pActiveScene);
    g_SceneList.pop_back();

    // ノイズ防止のために描画を一時停止
    FreezeRender();

    return pActiveScene;
}

void SceneManager::BackTo(SceneBase* pTargetScene) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTargetScene);

    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);

    NN_ASSERT(g_IsInitialized);
    NN_ABORT_UNLESS(!g_SceneList.empty());

    int index = util::FindIndex(g_SceneList, pTargetScene);
    NN_ABORT_UNLESS_GREATER_EQUAL(index, 0);

    TerminateTo(index + 1);
}

void SceneManager::BackTo(const char* sceneName) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(sceneName);

    std::lock_guard<decltype(g_Mutex)> lock(g_Mutex);

    NN_ASSERT(g_IsInitialized);
    NN_ABORT_UNLESS(!g_SceneList.empty());

    // シーン名が一致するシーンを探して戻る
    for (auto iter = g_SceneList.rbegin(); iter != g_SceneList.rend(); iter++)
    {
        if ((*iter)->IsNameMatched(sceneName))
        {
            BackTo(*iter);
            return;
        }
    }
}

bool SceneManager::IsEmpty() NN_NOEXCEPT
{
    return g_SceneList.empty();
}

SceneBase* SceneManager::ProcessActiveScene(SceneProcessFunc func) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(func);

    auto pScene = GetActiveScene();
    if (pScene != nullptr)
    {
        func(pScene);
    }

    return pScene;
}

void SceneManager::TerminateScene(SceneBase* pScene) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pScene);

    pScene->Terminate();

    if (g_TerminateCallback != nullptr)
    {
        g_TerminateCallback(pScene);
    }
}

void SceneManager::TerminateTo(int count) NN_NOEXCEPT
{
    NN_ASSERT_GREATER_EQUAL(count, 0);

    if (count < g_SceneList.size())
    {
        for (int i = g_SceneList.size() - 1; i >= count; i--)
        {
            auto pScene = g_SceneList[i];
            TerminateScene(pScene);
            g_TerminatedSceneList.push_back(pScene);
        }

        // 終了したシーンをスタックから除去
        g_SceneList.erase(g_SceneList.begin() + count, g_SceneList.end());

        // ノイズ防止のために描画を一時停止
        FreezeRender();
    }
}

void SceneManager::CheckSceneTerminated() NN_NOEXCEPT
{
    if (IsEmpty())
    {
        return;
    }

    // 現在のシーンが終了していればスタックから取り除く
    ProcessActiveScene([](SceneBase* pScene)
    {
        if (pScene->IsTerminated())
        {
            g_TerminatedSceneList.push_back(pScene);
            g_SceneList.pop_back();
        }
    });

    // シーン終了時の後処理
    if (!g_TerminatedSceneList.empty())
    {
        for (auto pScene : g_TerminatedSceneList)
        {
            if (pScene->IsAutoDeleteEnabled())
            {
                delete pScene;
            }
        }

        g_TerminatedSceneList.clear();

        // 前のシーンをアクティブ化
        ProcessActiveScene([](SceneBase* pScene)
        {
            if (!pScene->IsActive())
            {
                pScene->Activate();
            }
        });

        // 遷移エフェクトを開始
        StartTransitionEffect();
    }
}

void SceneManager::FreezeRender() NN_NOEXCEPT
{
    g_IsRenderFrozen = true;

    NNS_SGX_SCENE_LOG("%s\n", NN_CURRENT_FUNCTION_NAME);
}

void SceneManager::ResumeRender() NN_NOEXCEPT
{
    g_IsRenderFrozen = false;

    NNS_SGX_SCENE_LOG("%s\n", NN_CURRENT_FUNCTION_NAME);
}

void SceneManager::StartTransitionEffect() NN_NOEXCEPT
{
    EndTransitionEffect();
    ResumeRender();

    g_TransitionDuration = 1;
    nns::sgx::CaptureToImage(&g_TransitionImage);
}

void SceneManager::EndTransitionEffect() NN_NOEXCEPT
{
    g_TransitionDuration = 0;
    if (g_TransitionImage.IsValid())
    {
        nns::sgx::DestroyImage(&g_TransitionImage);
    }
}

bool SceneManager::IsTransiting() NN_NOEXCEPT
{
    return g_TransitionDuration > 0;
}

bool SceneManager::IsOutingScene() NN_NOEXCEPT
{
#if 0
    return g_TransitionDuration > 0 &&
        g_TransitionDuration < g_TransitionFrameCount;
#else
    return false;
#endif
}

void SceneManager::UpdateTransitionEffect() NN_NOEXCEPT
{
    if (!IsTransiting())
    {
        return;
    }

    g_TransitionDuration++;
    if (g_TransitionDuration > g_TransitionFrameCount)
    {
        EndTransitionEffect();
    }
}

}}}  // nns::hid::scene
