﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/util/util_MathTypes.h>

namespace input {

/**
 * @brief   タッチパネルのタッチ情報です。
 */
struct TouchState
{
    bool isTouched;     //!< タッチされているか
    NN_PADDING7;
    int  frames;        //!< タッチが継続しているフレーム数
    int  id;            //!< 指 ID
    int  x;             //!< タッチ位置の X 座標
    int  y;             //!< タッチ位置の Y 座標

    /**
     * @brief   メンバを初期化
     */
    void Clear() NN_NOEXCEPT
    {
        isTouched = false;
        frames    = 0;
        id        = -1;
        x         = 0;
        y         = 0;
    }
};

/**
 * @brief   タッチパネル情報を管理するクラスです。
 */
class TouchPanel final
{
    NN_DISALLOW_COPY(TouchPanel);
    NN_DISALLOW_MOVE(TouchPanel);

public:
    /**
     * @brief   タッチパネルのインスタンスを取得します。
     */
    static TouchPanel* GetInstance() NN_NOEXCEPT
    {
        static TouchPanel s_Instance;
        return &s_Instance;
    }

    /**
     * @brief   タッチパネル情報を更新します。
     */
    void Update() NN_NOEXCEPT;

    /**
     * @brief   タッチ位置を取得します。
     */
    int GetTouchPoints(float* pOutListX, float* pOutListY, int maxCount) NN_NOEXCEPT;

    /**
     * @brief   指定した矩形にタッチ位置が含まれているか判定します。
     */
    bool IsTouched(float x, float y, float width, float height) NN_NOEXCEPT;

    bool IsTouched(const nn::util::Float4& rect) NN_NOEXCEPT
    {
        return IsTouched(rect.x, rect.y, rect.z, rect.w);
    }

    /**
     * @brief   任意の位置でタッチトリガが発生したか判定します。
     */
    bool IsTouchTriggered() NN_NOEXCEPT;

    /**
     * @brief   指定した矩形内でタッチトリガが発生したか判定します。
     */
    bool IsTouchTriggered(float x, float y, float width, float height) NN_NOEXCEPT;

    bool IsTouchTriggered(const nn::util::Float4& rect) NN_NOEXCEPT
    {
        return IsTouchTriggered(rect.x, rect.y, rect.z, rect.w);
    }

    /**
     * @brief   任意の位置でリリーストリガが発生したか判定します。
     */
    bool IsTouchReleased() NN_NOEXCEPT;

    /**
     * @brief   指定した矩形内でリリーストリガが発生したか判定します。
     */
    bool IsTouchReleased(float x, float y, float width, float height) NN_NOEXCEPT;

    bool IsTouchReleased(const nn::util::Float4& rect) NN_NOEXCEPT
    {
        return IsTouchReleased(rect.x, rect.y, rect.z, rect.w);
    }

private:
    TouchPanel() NN_NOEXCEPT :
        m_Mutex(true),
        m_CurrentTouchStates(),
        m_PreviousTouchStates()
    {
        Initialize();
    }

    ~TouchPanel() NN_NOEXCEPT
    {
        Finalize();
    }

    /**
     * @brief   タッチパネルの初期化処理を行います。
     */
    void Initialize() NN_NOEXCEPT;

    /**
     * @brief   タッチパネルの終了処理を行います。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   指定したステート中に同一 finger ID のタッチが含まれているか判定します。
     */
    bool IsTouchedImpl(int fingerId, const TouchState* states, int stateCount) const NN_NOEXCEPT;

private:
    mutable nn::os::Mutex   m_Mutex;

    TouchState              m_CurrentTouchStates[nn::hid::TouchScreenStateCountMax];
    TouchState              m_PreviousTouchStates[nn::hid::TouchScreenStateCountMax];
};

}  // input
