﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/result/result_HandlingUtility.h>

#include "TouchPanel.h"

namespace input {

namespace
{

// タッチ位置が指定した矩形範囲にあるか
bool IsRectTouched(const TouchState& touch, float x, float y, float width, float height) NN_NOEXCEPT
{
    if (touch.isTouched &&
        touch.x >= x && touch.x <= x + width &&
        touch.y >= y && touch.y <= y + height)
    {
        return true;
    }

    return false;
}

// タッチしている最初の finger ID を取得
int FindFirstTouchId(const TouchState* touches, int touchCount) NN_NOEXCEPT
{
    for (int i = 0; i < touchCount; i++)
    {
        const auto& touch = touches[i];
        if (touch.isTouched)
        {
            return touch.id;
        }
    }

    return -1;
}

// 指定した矩形範囲をタッチしている最初の finger ID を取得
int FindFirstTouchId(const TouchState* touches, int touchCount, float x, float y, float width, float height) NN_NOEXCEPT
{
    for (int i = 0; i < touchCount; i++)
    {
        const auto& touch = touches[i];
        if (touch.isTouched &&
            touch.x >= x && touch.x <= x + width &&
            touch.y >= y && touch.y <= y + height)
        {
            return touch.id;
        }
    }

    return -1;
}

}

void TouchPanel::Update() NN_NOEXCEPT
{
    std::memcpy(m_PreviousTouchStates, m_CurrentTouchStates, sizeof(m_PreviousTouchStates));
    for (auto& touch : m_CurrentTouchStates)
    {
        touch.Clear();
    }

    nn::hid::TouchScreenState<nn::hid::TouchScreenStateCountMax> touches;
    nn::hid::GetTouchScreenState(&touches);

    auto findPreviousTouch = [this](TouchState* pOutState, int fingerId) NN_NOEXCEPT
    {
        for (const auto& touch : m_PreviousTouchStates)
        {
            if (touch.id == fingerId)
            {
                *pOutState = touch;
                return true;
            }
        }

        return false;
    };

    for (int i = 0; i < touches.count; i++)
    {
        const auto& touch = touches.touches[i];

        TouchState prevTouch;
        if (findPreviousTouch(&prevTouch, touch.fingerId))
        {
            // タッチ継続
            m_CurrentTouchStates[i] = prevTouch;
        }
        else
        {
            // 新規タッチ
            auto& newTouch = m_CurrentTouchStates[i];
            newTouch.isTouched = true;
            newTouch.frames    = 0;
            newTouch.id        = touch.fingerId;
        }

        auto& newTouch = m_CurrentTouchStates[i];
        newTouch.frames++;
        newTouch.x = touch.x;
        newTouch.y = touch.y;
    }
}

int TouchPanel::GetTouchPoints(float* pOutListX, float* pOutListY, int maxCount) NN_NOEXCEPT
{
    int touchCount = 0;
    for (const auto& touch : m_CurrentTouchStates)
    {
        if (touchCount >= maxCount)
        {
            break;
        }

        if (touch.isTouched)
        {
            pOutListX[touchCount] = static_cast<float>(touch.x);
            pOutListY[touchCount] = static_cast<float>(touch.y);
            touchCount++;
        }
    }

    return touchCount;
}

bool TouchPanel::IsTouched(float x, float y, float width, float height) NN_NOEXCEPT
{
    for (const auto& touch : m_CurrentTouchStates)
    {
        if (IsRectTouched(touch, x, y, width, height))
        {
            return true;
        }
    }

    return false;
}

bool TouchPanel::IsTouchTriggered() NN_NOEXCEPT
{
    auto id = FindFirstTouchId(m_CurrentTouchStates, NN_ARRAY_SIZE(m_CurrentTouchStates));
    if (id < 0)
    {
        return false;
    }

    // 同じ ID で前回タッチしていなければタッチトリガ成立
    return !IsTouchedImpl(id, m_PreviousTouchStates, NN_ARRAY_SIZE(m_PreviousTouchStates));
}

bool TouchPanel::IsTouchTriggered(float x, float y, float width, float height) NN_NOEXCEPT
{
    auto id = FindFirstTouchId(m_CurrentTouchStates, NN_ARRAY_SIZE(m_CurrentTouchStates), x, y, width, height);
    if (id < 0)
    {
        return false;
    }

    // 同じ ID で前回タッチしていなければタッチトリガ成立
    return !IsTouchedImpl(id, m_PreviousTouchStates, NN_ARRAY_SIZE(m_PreviousTouchStates));
}

bool TouchPanel::IsTouchReleased() NN_NOEXCEPT
{
    auto id = FindFirstTouchId(m_PreviousTouchStates, NN_ARRAY_SIZE(m_PreviousTouchStates));
    if (id < 0)
    {
        // 直前にタッチされていない
        return false;
    }

    // 同じ finger ID でタッチが継続していなければリリーストリガ成立
    return !IsTouchedImpl(id, m_CurrentTouchStates, NN_ARRAY_SIZE(m_CurrentTouchStates));
}

bool TouchPanel::IsTouchReleased(float x, float y, float width, float height) NN_NOEXCEPT
{
    auto id = FindFirstTouchId(m_PreviousTouchStates, NN_ARRAY_SIZE(m_PreviousTouchStates), x, y, width, height);
    if (id < 0)
    {
        // 直前にタッチされていない
        return false;
    }

    // 同じ finger ID でタッチが継続していなければリリーストリガ成立
    return !IsTouchedImpl(id, m_CurrentTouchStates, NN_ARRAY_SIZE(m_CurrentTouchStates));
}

void TouchPanel::Initialize() NN_NOEXCEPT
{
    for (auto& touch : m_CurrentTouchStates)
    {
        touch.Clear();
    }
    for (auto& touch : m_PreviousTouchStates)
    {
        touch.Clear();
    }
}

void TouchPanel::Finalize() NN_NOEXCEPT
{
}

bool TouchPanel::IsTouchedImpl(int fingerId, const TouchState* states, int stateCount) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(states);

    for (int i = 0; i < stateCount; i++)
    {
        const auto& state = states[i];
        if (state.isTouched && state.id == fingerId)
        {
            return true;
        }
    }

    return false;
}

}  // input
