﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <memory>
#include <new>
#include <vector>

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "NpadManager.h"

namespace input {

namespace
{

// 指定した NpadId のインデックスを取得
int GetControllerIndex(const nn::hid::NpadIdType& id) NN_NOEXCEPT
{
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        if (id == SupportedNpadIds[i])
        {
            return i;
        }
    }

    return -1;
}

// 指定した NpadId に対して適切な NpadController を生成
INpadController* CreateNpadController(void* pStorage, const nn::hid::NpadIdType& id) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pStorage);

    if (GetControllerIndex(id) < 0)
    {
        return nullptr;
    }

    // 接続されたデバイスに応じたコントローラクラスを作成
    auto style = nn::hid::GetNpadStyleSet(id);
    if (style.Test<nn::hid::NpadStyleJoyDual>())
    {
        return new(pStorage) NpadController<NpadJoyDual>(id);
    }
    else if (style.Test<nn::hid::NpadStyleJoyLeft>())
    {
        return new(pStorage) NpadController<NpadLeftJoy>(id);
    }
    else if (style.Test<nn::hid::NpadStyleJoyRight>())
    {
        return new(pStorage) NpadController<NpadRightJoy>(id);
    }
    else if (style.Test<nn::hid::NpadStyleFullKey>())
    {
        return new(pStorage) NpadController<NpadFullKey>(id);
    }
    else if (style.Test<nn::hid::NpadStyleHandheld>())
    {
        return new(pStorage) NpadController<NpadHandheld>(id);
    }

    return nullptr;
}

}  // anonymous

bool INpadController::g_IsSilent = false;

void NpadManager::Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();

    // Npad の設定
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleFullKey::Mask |
        nn::hid::NpadStyleJoyDual::Mask |
        nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(SupportedNpadIds, NpadIdCountMax);
    nn::hid::SetNpadHandheldActivationMode(nn::hid::NpadHandheldActivationMode_Single);

    m_LastControlledNpadId = nn::hid::NpadIdType();
}

void NpadManager::Finalize() NN_NOEXCEPT
{
    INpadController::SetSilent(false);
}

INpadController* NpadManager::GetNpadController(const nn::hid::NpadIdType& id) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    int index = GetControllerIndex(id);
    if (index < 0)
    {
        return nullptr;
    }

    auto& controller = m_NpadControllersStorage[index];
    if (controller.IsAvailable())
    {
        // 有効状態の再検証
        controller.style = nn::hid::GetNpadStyleSet(id);
    }

    if (controller.IsAvailable())
    {
        return reinterpret_cast<INpadController*>(controller.storage);
    }

    auto* pController = CreateNpadController(controller.storage, id);
    if (pController == nullptr)
    {
        return nullptr;
    }

    pController->Initialize();

    controller.style = nn::hid::GetNpadStyleSet(id);
    return pController;
}

INpadController* NpadManager::GetDummyController() NN_NOEXCEPT
{
    auto& controller = m_NpadControllersStorage[NpadIdCountMax];
    if (!controller.IsAvailable())
    {
        // Handheld 扱いにする
        auto* pController = new(controller.storage) NpadController<NpadHandheld>(nn::hid::NpadId::Handheld);
        pController->Initialize();
        controller.style = nn::hid::NpadStyleHandheld::Mask;
    }

    return reinterpret_cast<INpadController*>(controller.storage);
}

std::vector<INpadController*> NpadManager::GetNpadControllers() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    std::vector<INpadController*> controllers;

    for (const auto& id : SupportedNpadIds)
    {
        auto pController = GetNpadController(id);
        if (pController != nullptr)
        {
            controllers.push_back(pController);
        }
    }

    return controllers;
}

bool NpadManager::IsConnected(const nn::hid::NpadIdType& id) NN_NOEXCEPT
{
    return GetNpadController(id) != nullptr;
}

void NpadManager::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    auto controllers = GetNpadControllers();
    for (auto pController : controllers)
    {
        pController->Update();
    }

    auto pDummyController = GetDummyController();
    if (pDummyController != nullptr)
    {
        pDummyController->Update();
    }

    UpdateLastControlledId();
}

void NpadManager::UpdateLastControlledId() NN_NOEXCEPT
{
    auto controllers = GetNpadControllers();
    for (auto pController : controllers)
    {
        if (pController->IsControlled())
        {
            m_LastControlledNpadId = pController->GetNpadId();
            return;
        }
    }

    // 最終操作コントローラーがなければ、最初に見つかったコントローラーにする
    for (auto pController : controllers)
    {
        if (pController != nullptr)
        {
            m_LastControlledNpadId = pController->GetNpadId();
            break;
        }
    }
}

void PerformPressLastController(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
{
    auto pManager   = NpadManager::GetInstance();
    auto lastNpadId = pManager->GetLastController();

    // 最後に操作されたコントローラーが顕在なら、そのコントローラーを操作
    if (pManager->IsConnected(lastNpadId))
    {
        pManager->GetNpadController(lastNpadId)->PerformPress(buttons);
        return;
    }

    // 最後に操作されたコントローラーが切断されていれば、最初に見つかったコントローラーを操作
    auto controllers = pManager->GetNpadControllers();
    if (!controllers.empty())
    {
        controllers[0]->PerformPress(buttons);
    }
}

void PerformPressDummyController(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
{
    auto pController = NpadManager::GetInstance()->GetDummyController();
    pController->PerformPress(buttons);
}

}  // input
