﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ベース部分となるレイアウト作成ための API の宣言
 */

#pragma once

#include <string>
#include <cstdlib>
#include <map>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_Color.h>

#include "../gfx.h"
#include "../hid.h"

#define NNF_LAYOUT_LAZY_DESC_FUNCTION(funcName, varName, varIndex)\
    virtual bool Is ## funcName () const NN_NOEXCEPT { return (varName).Test(varIndex); }\
    virtual void funcName(bool is ## funcName) NN_NOEXCEPT {  (varName).Set(varIndex,is ## funcName); }

namespace nns { namespace hidfw { namespace layout {

    /**
     * @brief       画面上に表示されるボタンのオプションを設定するための構造体です
     */
    struct LayoutOption
    {
        typedef ::nn::util::BitFlagSet<16, LayoutOption>::Flag<0> Visible;                  //!< false の場合非表示にします (選択対象からも除外したい場合Enableをfakseにして下さい)
        typedef ::nn::util::BitFlagSet<16, LayoutOption>::Flag<1> MultiMode;                //!< true の場合ButtonSetで同一グループに設定したボタンの同時選択が可能に
        typedef ::nn::util::BitFlagSet<16, LayoutOption>::Flag<2> OnOffMode;                //!< ボタンを選択するたびに ON / OFF を切り替えます
        typedef ::nn::util::BitFlagSet<16, LayoutOption>::Flag<3> ThroughSelect;            //!< (コントローラのみ) 選択 を一時的に無効化します
        typedef ::nn::util::BitFlagSet<16, LayoutOption>::Flag<4> ThroughCancel;            //!< (コントローラのみ) キャンセル を一時的に無効化します
        typedef ::nn::util::BitFlagSet<16, LayoutOption>::Flag<5> ThroughChoose;            //!< (コントローラのみ) 項目の移動を一時的に無効化します
        typedef ::nn::util::BitFlagSet<16, LayoutOption>::Flag<6> ForciblyFocus;            //!< フォーカスがあっていなくともフォーカスがあっているものとして強制的に操作を実行する
        typedef ::nn::util::BitFlagSet<16, LayoutOption>::Flag<7> CallFunction;             //!< false の場合、選択された際に関数呼び出しを実行しません
        typedef ::nn::util::BitFlagSet<16, LayoutOption>::Flag<8> ImmediateExecute;         //!< true の場合、アニメーションや待機時間を無視して設定された処理を即時実行します。
    };
    typedef ::nn::util::BitFlagSet<16, LayoutOption> LayoutOptionSet;

    /**
     * @brief       画面上に表示されるボタンの状態を管理するための構造体です
     */
    struct LayoutState
    {
        typedef ::nn::util::BitFlagSet<16, LayoutState>::Flag<0>  Enable;                   //!< ボタンが有効な常態か (false の場合グレーアウト)
        typedef ::nn::util::BitFlagSet<16, LayoutState>::Flag<1>  OnFocus;                  //!< フォーカスがあっている場合 true
        typedef ::nn::util::BitFlagSet<16, LayoutState>::Flag<2>  Active;                   //!< 押下げている最中 / タッチしている最中
        typedef ::nn::util::BitFlagSet<16, LayoutState>::Flag<3>  Hover;                    //!< カーソルがあっている
        typedef ::nn::util::BitFlagSet<16, LayoutState>::Flag<4>  Selected;                 //!< 選択済み
        typedef ::nn::util::BitFlagSet<16, LayoutState>::Flag<5>  Canceled;                 //!< キャンセルした瞬間
        typedef ::nn::util::BitFlagSet<16, LayoutState>::Flag<6>  Animation;                //!< アニメーション中にtrue
        typedef ::nn::util::BitFlagSet<16, LayoutState>::Flag<7>  UpdateValue;              //!< 値が更新された際にtrue
        typedef ::nn::util::BitFlagSet<16, LayoutState>::Flag<8>  LimitingValue;            //!< 値が限界値に達した瞬間
    };
    typedef ::nn::util::BitFlagSet<16, LayoutState> LayoutStateSet;

    typedef void(*LayoutFunction)(void* executant, void* param);                            //!< ボタンを押下げた際の動作

    /**
     * @brief       ベースとなる画面フレームを描画する為のクラスです
     */
    class BaseFrame
    {
    protected:
        bool        m_ExitFlag;
        uint64_t    m_FrameCount;
    public:
        virtual void Initialize() NN_NOEXCEPT;
        virtual void Finalize() NN_NOEXCEPT;
        virtual void WaitExit() NN_NOEXCEPT;
        virtual void Update() NN_NOEXCEPT;
        virtual void Draw() NN_NOEXCEPT;
        virtual void SetButtons(const LayoutStateSet& _button) NN_NOEXCEPT;
    };

    /**
     * @brief       描画するボタン等の、レイアウト上に表示するアイテムの管理を行う為のクラスです
     */
    class BaseItem
    {
    public:
        /**
         * @brief       描画するボタンの操作方法を選択、設定する為の構造体です
         */
        struct ChoiceMode
        {
            typedef ::nn::util::BitFlagSet<8, ChoiceMode>::Flag<0> None;                //!< 他のフラグに関わらずボタン操作の選択を無効化
            typedef ::nn::util::BitFlagSet<8, ChoiceMode>::Flag<1> Touch;               //!< タッチでの選択を可能とする
            typedef ::nn::util::BitFlagSet<8, ChoiceMode>::Flag<2> Controller;          //!< コントローラー操作を可能とする
        };
        typedef ::nn::util::BitFlagSet<8, ChoiceMode> ChoiceModeSet;

    public:
        BaseItem() NN_NOEXCEPT {}
        virtual ~BaseItem() NN_NOEXCEPT {}

        //----------------------------------------------------------------
        //! @brief      デフォルト値を設定します
        //----------------------------------------------------------------
        void SetDefault() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      関数の登録
        //! @param[in]  func Select時に呼び出される関数
        //----------------------------------------------------------------
        void SetFunc(LayoutFunction func) NN_NOEXCEPT { m_Func = func; }

        //----------------------------------------------------------------
        //! @brief      登録された関数を、登録した引数で実行します
        //! @details    関数が登録されていない場合は何もせず即時抜け出します
        //----------------------------------------------------------------
        void CallFunc() NN_NOEXCEPT { if (m_Func != nullptr) { m_Func(this, m_Param); } }

        void Setup(float x, float y, float w, float h, LayoutFunction func, const char* text, ...) NN_NOEXCEPT;

        // Setter
        virtual void SetX(float x) NN_NOEXCEPT { m_Pos.x = x; }
        virtual void SetY(float y) NN_NOEXCEPT { m_Pos.y = y; }
        virtual void SetPos(float x, float y) NN_NOEXCEPT { m_Pos = NN_UTIL_FLOAT_2_INITIALIZER(x, y); }
        virtual void SetWidth(float w) NN_NOEXCEPT { m_Size.x = w; }
        virtual void SetHeight(float h) NN_NOEXCEPT { m_Size.x = h; }
        virtual void SetSize(float w, float h) NN_NOEXCEPT { m_Size = NN_UTIL_FLOAT_2_INITIALIZER(w, h); }
        void SetParam(void* param) NN_NOEXCEPT { m_Param = param; }
        void SetText(const char* _str, ...) NN_NOEXCEPT;

        // Getter
        float GetX() const NN_NOEXCEPT { return m_Pos.x; }
        float GetY() const NN_NOEXCEPT { return m_Pos.y; }
        nn::util::Float2 GetPos() const NN_NOEXCEPT { return m_Pos; }
        float GetWidth() const NN_NOEXCEPT { return m_Size.x; }
        float GetHeight() const NN_NOEXCEPT { return m_Size.y; }
        nn::util::Float2 GetSize() const NN_NOEXCEPT { return m_Size; }
        std::string GetText() const NN_NOEXCEPT { return m_Text; }

        void SetChooseMode(const ChoiceModeSet& mode) NN_NOEXCEPT { m_ChooseMode = mode; }
        ChoiceModeSet GetChooseMode() const NN_NOEXCEPT { return m_ChooseMode; }

        //----------------------------------------------------------------
        //! @brief      オブジェクトのスケールを取得します
        //! @details    このオブジェクトはアニメーション何度によりスケールが変化することがあります
        //!             本関数では現時点でのオブジェクトのスケールを取得することが出来ます
        //!             スケールは内部処理でのみ変化しユーザーが直接変更することは出来ません
        //----------------------------------------------------------------
        nn::util::Float2 GetScale() const NN_NOEXCEPT { return m_Scale; }

        const nn::util::Color4u8& GetMainColor() const NN_NOEXCEPT { return m_MainColor; }
        const nn::util::Color4u8& GetSubColor() const NN_NOEXCEPT { return m_SubColor; }
        const nn::util::Color4u8& GetTextColor() const NN_NOEXCEPT { return m_TextColor; }
        const nn::util::Color4u8& GetBorderColor() const NN_NOEXCEPT { return m_BorderColor; }
        const nn::util::Color4u8& GetEffectColor() const NN_NOEXCEPT { return m_EffectColor; }
        const nn::util::Color4u8& GetFocusColor() const NN_NOEXCEPT { return m_FocusColor; }

        void SetMainColor(const nn::util::Color4u8& color) NN_NOEXCEPT { m_MainColor = color; }
        void SetSubColor(const nn::util::Color4u8& color) NN_NOEXCEPT { m_SubColor = color; }
        void SetTextColor(const nn::util::Color4u8& color) NN_NOEXCEPT { m_TextColor = color; }
        void SetBorderColor(const nn::util::Color4u8& color) NN_NOEXCEPT { m_BorderColor = color; }
        void SetEffectColor(const nn::util::Color4u8& color) NN_NOEXCEPT { m_EffectColor = color; }
        void SetFocusColor(const nn::util::Color4u8& color) NN_NOEXCEPT { m_FocusColor = color; }

        NNF_LAYOUT_LAZY_DESC_FUNCTION(Visible, m_Option, LayoutOption::Visible::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(MultiMode, m_Option, LayoutOption::MultiMode::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(OnOffMode, m_Option, LayoutOption::OnOffMode::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(ThroughSelect, m_Option, LayoutOption::ThroughSelect::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(ThroughCancel, m_Option, LayoutOption::ThroughCancel::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(ThroughChoose, m_Option, LayoutOption::ThroughChoose::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(ForciblyFocus, m_Option, LayoutOption::ForciblyFocus::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(ImmediateExecute, m_Option, LayoutOption::ImmediateExecute::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(Enable, m_State, LayoutState::Enable::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(OnFocus, m_State, LayoutState::OnFocus::Index)
        NNF_LAYOUT_LAZY_DESC_FUNCTION(Active, m_State, LayoutState::Active::Index)
        //NNF_LAYOUT_LAZY_DESC_FUNCTION(Hover, m_State, LayoutState::Hover::Index)
        //NNF_LAYOUT_LAZY_DESC_FUNCTION(Selected, m_State, LayoutState::Selected::Index)
        //NNF_LAYOUT_LAZY_DESC_FUNCTION(Canceled, m_State, LayoutState::Canceled::Index)
        bool IsSelected() const NN_NOEXCEPT { return m_State.Test<LayoutState::Selected>(); }

        //----------------------------------------------------------------
        //! @brief      ステータスを代入します
        //! @details    矛盾するステータスを同時に有効化した場合
        //!             優先順位に応じてパースされます。
        //----------------------------------------------------------------
        void SetLayoutState(const LayoutStateSet& state) NN_NOEXCEPT { m_State = state; }

        //----------------------------------------------------------------
        //! @brief      オプションをセットします
        //----------------------------------------------------------------
        void SetLayoutOption(const LayoutOptionSet& options) NN_NOEXCEPT { m_Option = options; }

        //----------------------------------------------------------------
        //! @brief      ステータスを取得します
        //! @return     アイテムのステータス
        //----------------------------------------------------------------
        LayoutStateSet GetLayoutState() const NN_NOEXCEPT { return m_State; }

        //----------------------------------------------------------------
        //! @brief      ステータスを取得します
        //! @return     アイテムのステータス
        //----------------------------------------------------------------
        LayoutOptionSet GetLayoutOption() const NN_NOEXCEPT { return m_Option; }

        //----------------------------------------------------------------
        //! @brief      選択のキャンセル
        //! @details    次回のUpdate時にキャンセル処理を実行します
        //----------------------------------------------------------------
        virtual void Cancel() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      選択実行
        //! @details    次回のUpdate時に項目の選択を実行します
        //! @param[in]  callFunc 関数呼び出しを実行します
        //----------------------------------------------------------------
        virtual void Select(bool callFunc = true) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      ホバー状態
        //! @details    オブジェクトに照準が合っており、かつカーソルが直情にあるか
        //! @param[in]  enable trueの場合、カーソルがこのオブジェクトへ移動します
        //----------------------------------------------------------------
        virtual void Hover(bool enable) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      更新処理
        //! @details    直前のフレームで行ったオブジェクトへの操作が適応されます
        //----------------------------------------------------------------
        virtual void Update() NN_NOEXCEPT = 0;

        //----------------------------------------------------------------
        //! @brief      文字列の描画処理
        //----------------------------------------------------------------
        virtual void PrintText() NN_NOEXCEPT {}

        //----------------------------------------------------------------
        //! @brief      描画処理
        //----------------------------------------------------------------
        virtual void Draw() NN_NOEXCEPT = 0;

    protected:
        uint64_t                        m_FrameCount;                                   //!< ボタンの内部カウンタ
        uint64_t                        m_AnimationCount;                               //!< アニメーション用のカウンタ

        nn::util::Color4u8              m_MainColor;                                    //!< レイアウトアイテムのメインカラー
        nn::util::Color4u8              m_SubColor;                                     //!< レイアウトアイテムのサブカラー
        nn::util::Color4u8              m_TextColor;                                    //!< テキストカラー
        nn::util::Color4u8              m_BorderColor;                                  //!< 枠線の色
        nn::util::Color4u8              m_EffectColor;                                  //!< Hover 状態で表示されるエフェクトの色
        nn::util::Color4u8              m_FocusColor;                                   //!< フォーカスがあっている際に上乗せされる色

        nn::util::Float2                m_Pos;                                          //!< レイアウトアイテムの表示座標
        nn::util::Float2                m_Size;                                         //!< レイアウトアイテムのサイズ
        nn::util::Float2                m_Scale;                                        //!< (内部用) ボタンのスケール
        std::string                     m_Text;                                         //!< ボタンに表示されるテキスト
        LayoutFunction                  m_Func;                                         //!< ボタンが Selected 時に呼び出される関数
        void*                           m_Param;                                        //!< m_Func を呼び出す際に引数に設定される値

        ChoiceModeSet                   m_ChooseMode;                                   //!< どのような入力を受け付けるか
        LayoutOptionSet                 m_Option;                                       //!< ボタンのオプション
        LayoutStateSet                  m_State;                                        //!< ボタンの現在の状態
        LayoutStateSet                  m_OldState;                                     //!< 直前のフレームのステータス

    };

    /**
     * @brief       レイアウト上のアイテムをセットで管理する為のクラスです
     */
    class BaseItemSet
    {
    public:
        /**
         * @brief       ボタンの配置方向
         * @details     ボタンの配置方向です。ボタンの移動に使用される方向キーに反映されます。
         */
        enum eOrientation
        {
            eOrientation_Vertical = 0,       // 縦配置
            eOrientation_Horizontal          // 横配置
        };
        /**
         * @brief       描画するボタンの動作方式を指定する為の構造体です
         * @details     詳細は各メンバのコメントを参照してください
         */
        struct Action
        {
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<0> Select;             //!< 選択
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<1> Cancel;             //!< キャンセル
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<2> Next;               //!< 次のアイテムに移動
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<3> Back;               //!< 前野アイテムに移動
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<4> Priority;           //!< (最優先) 関数呼び出しによる操作など最優先で処理すべき操作
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<5> Touch;              //!< (優先) タッチ入力
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<6> Gamepad;            //!< コントローラ入力
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<7> CallFunc;           //!< (未実装) 選択・キャンセル時に関数呼び出しを実行するか否か
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<8> ChangeItemSet;      //!< (最低優先度) アイテムセットの照準を切り替えます
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<9> ChangeFocusItem;    //!< 照準が合っているアイテムを変更します
            typedef ::nn::util::BitFlagSet<16, Action>::Flag<15> NoneAction;        //!< 一切合財のアクションを無効化します
        };
        typedef ::nn::util::BitFlagSet<16, Action> ActionSet;

    public:
        BaseItemSet() NN_NOEXCEPT;
        virtual ~BaseItemSet() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      更新処理
        //----------------------------------------------------------------
        virtual void Update() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      入力の更新処理
        //! @details    Update 内で呼び出されます
        //----------------------------------------------------------------
        virtual void UpdateInput() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      タッチ入力の更新
        //! @details    UpdateInput 内で呼び出されます
        //----------------------------------------------------------------
        virtual void UpdateTouch() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      コントローラ入力の更新
        //! @details    UpdateInput 内で呼び出されます
        //----------------------------------------------------------------
        virtual void UpdateController() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      ActionCommand の処理
        //! @details    Update 内で呼び出されます
        //----------------------------------------------------------------
        virtual void ProcessActionCommand() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      描画処理
        //----------------------------------------------------------------
        virtual void Draw() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      アイテムリストへの追加
        //! @pre        new されたアイテムを追加する
        //! @param[inout]  button アイテムリストのサイズ
        //----------------------------------------------------------------
        int Add(BaseItem* button) NN_NOEXCEPT { m_ItemList.push_back(button); return GetListSize(); }

        //----------------------------------------------------------------
        //! @brief      アイテムリストの取得
        //! @return     アイテムリストへの参照を返します
        //----------------------------------------------------------------
        std::vector<BaseItem*>& GetItemSet() NN_NOEXCEPT { return m_ItemList; }

        //----------------------------------------------------------------
        //! @brief      リストサイズの取得
        //! @return     保持しているアイテムの個数を返します
        //----------------------------------------------------------------
        int GetListSize() const NN_NOEXCEPT { return static_cast<int>(m_ItemList.size()); }

        //----------------------------------------------------------------
        //! @brief      ボタンの配置方向を設定します
        //----------------------------------------------------------------
        void SetOrientation(eOrientation orientation) NN_NOEXCEPT { m_Orientation = orientation; }

        //----------------------------------------------------------------
        //! @brief      ボタンの配置方向を取得します
        //----------------------------------------------------------------
        eOrientation GetOrientation() const NN_NOEXCEPT { return m_Orientation; }

        //----------------------------------------------------------------
        //! @brief      リストを選択状態にします
        //! @param[in]  index 初期状態で照準が合っているアイテムのインデックス
        //! @param[in]  callFunc 選択状態切り替え時にindexのアイテムに登録された関数を呼び出すか
        //----------------------------------------------------------------
        bool Select(int index, bool callFunc) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      リストを選択状態にします
        //! @param[in]  index 初期状態で照準が合っているアイテムのインデックス
        //----------------------------------------------------------------
        bool Select(int index) NN_NOEXCEPT { return Select(index, true);}

        //----------------------------------------------------------------
        //! @brief      アイテムの選択状態を変更します
        //! @pre        index < GetListSize()
        //! @param[in]  index 選択状態にするアイテムのインデックス
        //! @param[in]  index 選択状態にした際、関数呼び出しを行うか
        //----------------------------------------------------------------
        void SelectItem(int index, bool callFunc) NN_NOEXCEPT { Select(index, callFunc); }

        //----------------------------------------------------------------
        //! @brief      照準があってるアイテムのインデックスを返します
        //! @return     アイテムのインデックス
        //----------------------------------------------------------------
        int GetFocusItemIndex() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      選択されているアイテムのインデックスを返します
        //! @details    廃止予定です。
        //!             引数付きの GetSelectedItemIndex を利用するか
        //!             GetSelectedItemIndices をご利用下さい
        //! @return     アイテムのインデックス
        //----------------------------------------------------------------
        int GetSelectedItemIndex() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      選択されているアイテムのインデックスを返します
        //! @details    廃止予定です。GetSelectedItemIndices をご利用下さい。
        //! @pre        index != nullptr
        //! @param[out] アイテムのインデックス
        //! @return     選択されているアイテムがある場合 true
        //----------------------------------------------------------------
        bool GetSelectedItemIndex(int* pOutIndex) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      選択されているアイテムのインデックスを返します
        //! @details    複数選択を有効にしており、複数の選択組みアイテムの
        //!             インデックスを取得したいときにご利用下さい
        //! @return     アイテムのインデックス
        //----------------------------------------------------------------
        const std::vector<int>& GetSelectedItemIndices() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      アイテムのフォーカス状態を変更します
        //! @pre        index < GetListSize()
        //! @param[in]  index フォーカスをあわせる状態にするアイテムのインデックス
        //----------------------------------------------------------------
        void FocusItem(int index) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      フォーカスを合わせます
        //! @details    フォーカスが合っているリストのみ操作することが出来ます
        //!             複数のリストにフォーカスを持たせることも可能です
        //! @param[in]  enable 有効か否か
        //----------------------------------------------------------------
        void Focus(bool enable) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      フォーカスがあっているか確認します
        //! @return     フォーカスがあっている場合trueを返します
        //----------------------------------------------------------------
        bool IsFocus() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      リストに登録されたアイテムの表示状態を変更します
        //! @details    アイテムとリストの表示は別管理となっているため
        //!             本関数を呼び出してもアイテムが持つ表示状態は上書きされません
        //! @param[in]  enable 有効か否か
        //----------------------------------------------------------------
        void Visible(bool enable) NN_NOEXCEPT { m_State.Set(LayoutOption::Visible::Index, enable); }

        //----------------------------------------------------------------
        //! @brief      親のリストを登録
        //! @details    登録されている場合、キャンセル時に移動します
        //! @param[in]  items 親のアイテムリスト
        //----------------------------------------------------------------
        void SetParentItemSet(BaseItemSet* items) NN_NOEXCEPT { m_pParentButtonSet = items; }

        //----------------------------------------------------------------
        //! @brief      次のリスト
        //! @details    次のリストが登録されており
        //!             かつ、現在のeOrientationの設定とは合致しない方向ボタンが押下げられた場合
        //!             登録されたリストに進みます
        //! @param[in]  enable 移動先のリスト
        //----------------------------------------------------------------
        void SetNextItemSet(BaseItemSet* items) NN_NOEXCEPT { m_pNextButtonSet = items; }

        //----------------------------------------------------------------
        //! @brief      前のリスト
        //! @details    前のリストが登録されており
        //!             かつ、現在のeOrientationの設定とは合致しない方向ボタンが押下げられた場合
        //!             登録されたリストに戻ります
        //! @param[in]  enable 移動先のリスト
        //----------------------------------------------------------------
        void SetPrevItemSet(BaseItemSet* items) NN_NOEXCEPT { m_pPrevButtonSet = items; }

        //----------------------------------------------------------------
        //! @brief      アイテムリストを一括登録
        //! @param[in]  parentItems 親のリストを登録
        //! @param[in]  parentItems 親のリストを登録
        //! @param[in]  parentItems 親のリストを登録
        //----------------------------------------------------------------
        void SetItemSet(BaseItemSet* parentItems, BaseItemSet* nextItems, BaseItemSet* prevItems) NN_NOEXCEPT
        {
            SetParentItemSet(parentItems), SetNextItemSet(nextItems), SetPrevItemSet(prevItems);
        }

    protected:
        static int                          m_MoveForciblyFocusButtonIndex;         //!< 強制的なフォーカスの移動先ボタンインデックス
        static bool                         m_IsForciblyFocusChanced;               //!< タッチ操作などにより強制的にフォーカスの移動が発生した
        static BaseItemSet*                 m_MoveForciblyFocusButtonSet;           //!< 強制的なフォーカスの移動先

        std::vector<BaseItem*>              m_ItemList;                             //!< このアイテムセットが管理するアイテム郡
        ActionSet                           m_Action;                               //!< 次回のUpdate時に実行される操作
        int                                 m_SelectedId;                           //!< 選択中のアイテムのインデックス
        std::vector<int>                    m_SelectedIds;                          //!< 選択中のアイテムのインデックス
        int                                 m_FutureSelectId;                       //!< カーソルを合っているアイテムのインデックス
        int                                 m_FutureFocusId;                        //!< カーソルを合わせるアイテムのインデックス
        eOrientation                        m_Orientation;                          //!< 現在のボタン配置モード
        LayoutOptionSet                     m_Option;                               //!< ボタンセットのオプション
        LayoutStateSet                      m_OldState;                             //!< 一つ前のステータス
        LayoutStateSet                      m_State;                                //!< 現在のステータス

        BaseItemSet*                        m_pNextButtonSet;                       //!< 項目の移動方向と異なる向きの方向ボタンが入力された場合に移動するボタンセット
        BaseItemSet*                        m_pPrevButtonSet;                       //!< 項目の移動方向と異なる向きの方向ボタンが入力された場合に移動するボタンセット
        BaseItemSet*                        m_pParentButtonSet;                     //!< 全ボタンがキャンセル状態(マルチ選択が有効の場合は常時)の時にキャンセルボタンを押下げた際に移動するボタンセット

        BaseItemSet*                        m_pChangeItemSet;                       //!< (内部用) 何らかの要因でボタンセットを変更される場合に利用します

        nn::hid::NpadButtonSet              m_SelectButton;                         //!< 項目の選択として扱うボタン
        nn::hid::NpadButtonSet              m_CancelButton;                         //!< 項目のキャンセルとして扱うボタン
        nn::hid::NpadButtonSet              m_UpButton;                             //!< 上入力として扱うボタン
        nn::hid::NpadButtonSet              m_DownButton;                           //!< 下入力として扱うボタン
        nn::hid::NpadButtonSet              m_LeftButton;                           //!< 左入力として扱うボタン
        nn::hid::NpadButtonSet              m_RightButton;                          //!< 右入力として扱うボタン

    };

}}}
