﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラ操作のための API の宣言
 */

#pragma once

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/nfc.h>
#include <nn/settings/settings_DebugPad.h>

#include <nn/irsensor.h>

#include "hid_Vibration.h"

namespace nns { namespace hidfw { namespace hid {

    /**
     * @brief       コントローラ関連の設定、状態取得等を行うためのクラスです
     * @details     複数台のコントローラを同時に使用するアプリを作成する場合は、
                    ControllerManagerを利用します
     */
    class Controller
    {
    public:
        /**
         * @brief       コントローラの初期化完了状況を表す構造体です
         */
        struct InitializeInfo
        {
            bool isNpad;
            nn::hid::NpadIdType npadId;
        };
        /**
         * @brief       コントローラの状態を管理するControllerの子クラスです
         */
        class ControllerState
        {
        public:
            //----------------------------------------------------------------
            //! @brief              ステータスを別のステータスへ変換します
            //! @pre                T = nn::hid::NpadFullKeyState
            //!                     T = nn::hid::NpadHandheldState
            //!                     T = nn::hid::NpadJoyDualState
            //!                     T = nn::hid::NpadJoyLeftState
            //!                     T = nn::hid::NpadJoyRightState
            //! 6return             変換後のステータス
            //----------------------------------------------------------------
            template<typename T> T Convert() NN_NOEXCEPT { return *(T*)this; }

        public:
            int64_t samplingNumber;                     //!< Npad の入力状態が更新される度に増加する値です。
            nn::hid::NpadButtonSet buttons;             //!< Npad のデジタルボタンの状態です。
            nn::hid::AnalogStickState analogStickL;     //!< Npad の左アナログスティックの状態です。
            nn::hid::AnalogStickState analogStickR;     //!< Npad の右アナログスティックの状態です。
            nn::hid::NpadAttributesSet attributes;      //!< Npad の状態を表す属性値です。
            ControllerState& operator= (const nn::hid::NpadFullKeyState& state);
            ControllerState& operator= (const nn::hid::NpadHandheldState& state);
            ControllerState& operator= (const nn::hid::NpadJoyDualState& state);
            ControllerState& operator= (const nn::hid::NpadJoyLeftState& state);
            ControllerState& operator= (const nn::hid::NpadJoyRightState& state);
        };

        /**
         * @brief       ボタンの繰り返し入力を判定する際に利用されるカウンタです
         */
        struct RepeatButtonCounter
        {
            static const uint64_t ButtonCountMax = 32;  //!< ボタンの数
            static const uint64_t DefaultInterval = 8;  //!< インターバルの初期値
            static const uint64_t DefaultDelay = 12;    //!< ディレイの初期値
            uint64_t interval;                          //!< リピート間隔
            uint64_t delay;                             //!< 最初のリピートまでの間隔
            uint64_t counter[ButtonCountMax];           //!< カウンタ
            RepeatButtonCounter() NN_NOEXCEPT
            {
                memset(this, NULL, sizeof(RepeatButtonCounter));
                interval = DefaultInterval;
                delay = DefaultDelay;
            }
        };
        /**
         * @brief       (-1.f .. 1.f) の範囲に調整された AnalogStickStateです
         */
        struct NormalizedAnalogStickState
        {
            float x;
            float y;
        };
    public:
        //----------------------------------------------------------------
        //! @brief              初期化を行います
        //! @param[in] info     このコントローラに割り当てるnpadIdの設定
        //----------------------------------------------------------------
        void Initialize(const InitializeInfo& info) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラ情報の更新
        //----------------------------------------------------------------
        void Update() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              NFC デバイス情報の更新を行います
        //----------------------------------------------------------------
        void UpdateNfcDevice() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              振動の操作
        //! @return             自身が持つ振動管理クラスを返します
        //----------------------------------------------------------------
        nns::hidfw::hid::Vibration& GetVibration() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラのステータスの取得
        //! @return             コントローラのステータス
        //----------------------------------------------------------------
        nns::hidfw::hid::Controller::ControllerState GetState() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラの割り当てを変更します
        //! @param[in] mode      設定するコントローラの割り当てモード
        //----------------------------------------------------------------
        void SetJoyAssignmentMode( const nn::hid::NpadJoyAssignmentMode& mode) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              コントローラの割り当てを取得します
        //! @return             コントローラの割り当てモード
        //----------------------------------------------------------------
        nn::hid::NpadJoyAssignmentMode GetJoyAssignmentMode() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              NpadIdをスワップする
        //! @param[in] id        スワップされるコントローラのNpadId
        //----------------------------------------------------------------
        void SwapNpadAssignment(nn::hid::NpadIdType id) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  NpadIdをスワップする
        //! @param[in] controller    スワップされるコントローラ
        //----------------------------------------------------------------
        void SwapNpadAssignment(const Controller& controller) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              SingleのコントローラをマージしDualにする
        //! @param[in] style     このコントローラにマージされるコントローラのNpadId
        //----------------------------------------------------------------
        void MergeSingleJoyAsDualJoy(nn::hid::NpadIdType id) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  SingleのコントローラをマージしDualにする
        //! @param[in] controller    このコントローラにマージされるコントローラ
        //----------------------------------------------------------------
        void MergeSingleJoyAsDualJoy(const Controller& controller) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              六軸センサを開始する
        //----------------------------------------------------------------
        void StartSixAxisSensor() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              六軸センサを開始する
        //! @details            内部でハンドルを再取得し指定されたスタイルのハンドルに対して処理を実行します
        //!                     style は複数指定可能です
        //! @param[in] style    開始するスタイル
        //----------------------------------------------------------------
        void StartSixAxisSensor(nn::hid::NpadStyleSet style) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              六軸センサを停止する
        //----------------------------------------------------------------
        void StopSixAxisSensor() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              IRセンサー動作を開始する
        //! @param[in] id       npadId
        //----------------------------------------------------------------
        void StartIrSensor(nn::hid::NpadIdType id) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              IRセンサー動作を停止する
        //! @param[in] id       npadId
        //----------------------------------------------------------------
        void StopIrSensor(nn::hid::NpadIdType id) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              NFC の動作を開始する
        //! @return             このコントローラに関連図けられた最低一つ以上の NFC デバイスが
        //!                     タグの検知を開始した場合 true を返します。
        //----------------------------------------------------------------
        bool StartNfc() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              NFC の動作を停止する
        //! @return             このコントローラに関連図けられた全ての NFC デバイスが
        //!                     タグの検知を停止した場合 true を返します
        //----------------------------------------------------------------
        bool StopNfc() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              六軸センサを停止する
        //! @details            内部でハンドルを再取得し指定されたスタイルのハンドルに対して処理を実行します
        //!                     style は複数指定可能です
        //! @param[in] style    停止するスタイル
        //----------------------------------------------------------------
        void StopSixAxisSensor(nn::hid::NpadStyleSet style) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      NpadID の取得
        //! @details    コントローラの NpadId を返します
        //!             コントローラが接続されていない場合や、
        //!             Npad でないコントローラの場合、0を返します
        //! @return     このコントローラの NpadId
        //----------------------------------------------------------------
        nn::hid::NpadIdType GetNpadId() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      スタイルの取得
        //! @details    コントローラが接続されていなかったり、
        //!             スタイルの取得に失敗したりした時は AllOff を返します
        //! @return     このコントローラの NpadStyleSet
        //----------------------------------------------------------------
        nn::hid::NpadStyleSet GetStyleSet() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      コントローラの接続確認を行います
        //! @details    JoyDualで片側だけが接続されている場合はfalseを返します
        //! @return     コントローラが接続されている場合 true を返します
        //----------------------------------------------------------------
        bool isConnected() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      コントローラの状態
        //! @return     コントローラの状態値を返します
        //----------------------------------------------------------------
        nn::hid::NpadAttributesSet GetAttributesSet() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      コントローラを切断します
        //----------------------------------------------------------------
        void Disconnect() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      ボタンの取得
        //! @return     押下げられているボタン
        //----------------------------------------------------------------
        nn::hid::NpadButtonSet GetButtonSet() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      指定したボタンが押下げられた瞬間に true を返します
        //! @pre        button.CountPopulation = 1
        //! @param[in]  button 確認するボタン
        //! @return     押下げられた瞬間に true
        //----------------------------------------------------------------
        bool IsTrigger(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      押下げられた瞬間のボタンを取得します
        //! @return     押下げられた瞬間のボタン
        //----------------------------------------------------------------
        nn::hid::NpadButtonSet GetTrgButton() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      指定したボタンが押下げられているか返します
        //! @details    button.CountPopulation = 1
        //! @param[in]  button 確認するボタン
        //! @return     押下げられている時に true
        //----------------------------------------------------------------
        bool IsHold(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      押下げられているボタンを取得します
        //! @details    this->GetButtonSet と同等です
        //! @return     押下げられているボタン
        //----------------------------------------------------------------
        nn::hid::NpadButtonSet GetHldButton() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      最も長くホールドしているボタンのホールド時間を返します
        //! @param[in]  inButtons 対象のボタンセット
        //! @param[out] outButton 最も長く押されているボタン (CountPopulation = 1)
        //! @return     ホールド時間 (フレーム)
        //----------------------------------------------------------------
        uint64_t GetMaxHoldCount(const nn::hid::NpadButtonSet& inButtons, nn::hid::NpadButtonSet* outButton) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      指定したボタンが押下げられている場合一定のタイミングでtrueを返します
        //! @details    button.CountPopulation = 1
        //! @param[in]  button 確認するボタン
        //! @return     押下げられている時に一定のタイミングで true
        //----------------------------------------------------------------
        bool IsRepeat(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      リピート入力されているボタンを返します
        //! @return     リピート入力されているボタン
        //----------------------------------------------------------------
        nn::hid::NpadButtonSet GetRepButton() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      指定したボタンが離された瞬間か返します
        //! @details    button.CountPopulation = 1
        //!             他のボタンが一切押下げられていない場合 true を返します
        //! @param[in]  button 確認するボタン
        //! @return     確認するボタンが離された瞬間に true
        //----------------------------------------------------------------
        bool IsRelease(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      リピート入力されているボタンを取得します
        //! @return     リピート入力されているボタン
        //----------------------------------------------------------------
        nn::hid::NpadButtonSet GetRelButton() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      プレイヤーランプの点灯パターンの取得
        //! @return      点灯パターン
        //----------------------------------------------------------------
        uint8_t GetPlayerLedPattern() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      左側アナログスティックの値を取得
        //! @return     アナログスティックの値
        //----------------------------------------------------------------
        nn::hid::AnalogStickState GetLeftAnalogStickState() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      左側アナログスティックの値を取得
        //! @return     アナログスティックの値
        //----------------------------------------------------------------
        NormalizedAnalogStickState GetNormalizedLeftAnalogStickState() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      右側アナログスティックの値を取得
        //! @return     アナログスティックの値
        //----------------------------------------------------------------
        nn::hid::AnalogStickState GetRightAnalogStickState() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief      右側アナログスティックの値を取得
        //! @return     アナログスティックの値
        //----------------------------------------------------------------
        NormalizedAnalogStickState GetNormalizedRightAnalogStickState() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  六軸センサの取得
        //! @param[in] num          第二引数 sensorState の要素数
        //! @param[out] sensorState 六軸センサのステータスの格納先
        //! @return                 取得できたステータスの個数
        //----------------------------------------------------------------
        int GetSixAxisSensorState(int num, nn::hid::SixAxisSensorState* sensorState) const NN_NOEXCEPT;

        int GetSixAxisSensorStates(nn::hid::NpadJoyDeviceType deviceType, nn::hid::SixAxisSensorState* outStates, int count) const NN_NOEXCEPT;

        bool IsSixAxisSensorAtRest(nn::hid::NpadJoyDeviceType deviceType) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  NFC のデバイスハンドルの取得
        //! @param[out] pOutState   取得したハンドルの格納先
        //! @param[out] pOutCount   実際に取得したハンドルの数の格納先
        //! @param[in] stateCount   取得するハンドルの最大数 (要素数)
        //! @return                 一つ以上のハンドルが取得できた場合 true
        //----------------------------------------------------------------
        bool GetNfcDeviceHandles(nn::nfc::DeviceHandle* pOutHandle, int* pOutCount, const int handleCount) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  NFC のデバイスステータスの取得
        //! @param[out] pOutState   取得したステータスの格納先
        //! @param[out] pOutCount   実際に取得したステータスの数の格納先
        //! @param[in] stateCount   取得するステータスの最大数 (要素数)
        //! @return                 一つ以上のステータスが取得できた場合 true
        //----------------------------------------------------------------
        bool GetNfcDeviceState(nn::nfc::DeviceState* pOutState, int* pOutCount, const int stateCount) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  ボタン配置を横持ち用に変換します
        //! @param[in] style        スタイル
        //! @param[in] button       変換対象のボタンセット
        //! @return                 変換されたボタンセット
        //----------------------------------------------------------------
        nn::hid::NpadButtonSet ConvertHorizontalButtonLayout(const nn::hid::NpadStyleSet& style, const nn::hid::NpadButtonSet& button) NN_NOEXCEPT;

        nn::irsensor::IrCameraHandle GetIrCameraHandle() const { return m_IrCameraHandle; }

        void RunIrProcessor(const nn::irsensor::MomentProcessorConfig& config) NN_NOEXCEPT { nn::irsensor::RunMomentProcessor(m_IrCameraHandle, config); }
        void RunIrProcessor(const nn::irsensor::ClusteringProcessorConfig& config) NN_NOEXCEPT { nn::irsensor::RunClusteringProcessor(m_IrCameraHandle, config); }
        void RunIrProcessor(const nn::irsensor::ImageTransferProcessorConfig& config, void* buffer, size_t bufferSize) NN_NOEXCEPT { nn::irsensor::RunImageTransferProcessor(m_IrCameraHandle, config, buffer, bufferSize); }
        void RunIrProcessor(const nn::irsensor::ImageTransferProcessorExConfig& config, void* buffer, size_t bufferSize) NN_NOEXCEPT { nn::irsensor::RunImageTransferProcessor(m_IrCameraHandle, config, buffer, bufferSize); }
        void RunIrProcessor(const nn::irsensor::HandAnalysisConfig& config) NN_NOEXCEPT { nn::irsensor::RunHandAnalysis(m_IrCameraHandle, config); }

        int GetIrProcessorStates(nn::irsensor::MomentProcessorState* states, int stateCount) NN_NOEXCEPT
        {
            NN_UNUSED(states);
            NN_UNUSED(stateCount);
            return 0;
        }

        int GetIrProcessorStates(nn::irsensor::ClusteringProcessorState* states, int stateCount) NN_NOEXCEPT
        {
            int result = 0;
            nn::irsensor::GetClusteringProcessorStates(states, &result, stateCount, m_IrCameraHandle);
            return result;
        }

        int GetIrProcessorStates(nn::irsensor::ImageTransferProcessorState* states, int stateCount) NN_NOEXCEPT
        {
            NN_UNUSED(states);
            NN_UNUSED(stateCount);
            return 0;
        }

        int GetIrProcessorStates(nn::irsensor::HandAnalysisSilhouetteState* states, int stateCount) NN_NOEXCEPT
        {
            int result = 0;
            nn::irsensor::GetHandAnalysisSilhouetteState(states, &result, stateCount, 0, m_IrCameraHandle);
            return result;
        }

        int GetIrProcessorStates(nn::irsensor::HandAnalysisImageState* states, int stateCount) NN_NOEXCEPT
        {
            int result = 0;
            nn::irsensor::GetHandAnalysisImageState(states, &result, stateCount, 0, m_IrCameraHandle);
            return result;
        }

        void StopImageProcessor() NN_NOEXCEPT { nn::irsensor::StopImageProcessor(m_IrCameraHandle); }
        void StopImageProcessorAsync() NN_NOEXCEPT { nn::irsensor::StopImageProcessorAsync(m_IrCameraHandle); }

        nn::irsensor::ImageProcessorStatus GetImageProcessorStatus() const NN_NOEXCEPT { return nn::irsensor::GetImageProcessorStatus(m_IrCameraHandle); }

    private:
        bool                    m_IsNpad;               //!< Npadか否か
        bool                    m_IsConnected;          //!< コントローラの接続状態
        nn::hid::NpadIdType     m_NpadId;               //!< コントローラの NpadId

        nn::hid::NpadButtonSet  m_ButtonUp;             //!< 話された瞬間のボタン
        nn::hid::NpadButtonSet  m_ButtonDown;           //!< 押下げられた瞬間のボタン
        nn::hid::NpadButtonSet  m_Button;               //!< 押下げられているボタン
        nn::hid::NpadButtonSet  m_ButtonRepeat;         //!< リピート入力するボタン
        RepeatButtonCounter     m_RepeatCounter;        //!< リピート入力で使用するカウンタ

        nn::hid::NpadStyleSet   m_StyleSet;             //!< コントローラの有効なスタイルです。(DebugPadではAllOffを返します)
        ControllerState         m_State;                //!< コントローラのステータスです。

        int                             m_SixAxisSensorHandleCount;
        nn::hid::SixAxisSensorHandle    m_SixAxisSensorHandles[nn::hid::NpadSixAxisSensorHandleCountMax];   //!< ハンドル
        nn::hid::SixAxisSensorState     m_SixAxisSensorState[nn::hid::NpadSixAxisSensorHandleCountMax];     //!< ステータス

        int                             m_SixAxisSensorStateCounts[nn::hid::NpadSixAxisSensorHandleCountMax];
        nn::hid::SixAxisSensorState     m_SixAxisSensorStates[nn::hid::NpadSixAxisSensorHandleCountMax][nn::hid::SixAxisSensorStateCountMax];     //!< ステータス

        nn::irsensor::IrCameraHandle    m_IrCameraHandle;

        int                             m_NfcDeviceHandleCount;
        nn::nfc::DeviceHandle           m_NfcDeviceHandles[nn::nfc::DeviceCountMax];

        Vibration               m_Vibration;
    };

}}}
