﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "hid_Controller.h"

namespace nns { namespace hidfw { namespace hid {

    Controller::ControllerState& Controller::ControllerState::operator= (const nn::hid::NpadFullKeyState& state) NN_NOEXCEPT
    {
        samplingNumber = state.samplingNumber;
        buttons = state.buttons;
        analogStickL = state.analogStickL;
        analogStickR = state.analogStickR;
        attributes = state.attributes;
        return (*this);
    }

    Controller::ControllerState& Controller::ControllerState::operator= (const nn::hid::NpadHandheldState& state) NN_NOEXCEPT
    {
        samplingNumber = state.samplingNumber;
        buttons = state.buttons;
        analogStickL = state.analogStickL;
        analogStickR = state.analogStickR;
        attributes = state.attributes;
        return (*this);
    }

    Controller::ControllerState& Controller::ControllerState::operator= (const nn::hid::NpadJoyDualState& state) NN_NOEXCEPT
    {
        samplingNumber = state.samplingNumber;
        buttons = state.buttons;
        analogStickL = state.analogStickL;
        analogStickR = state.analogStickR;
        attributes = state.attributes;
        return (*this);
    }

    Controller::ControllerState& Controller::ControllerState::operator= (const nn::hid::NpadJoyLeftState& state) NN_NOEXCEPT
    {
        samplingNumber = state.samplingNumber;
        buttons = state.buttons;
        analogStickL = state.analogStickL;
        analogStickR = state.analogStickR;
        attributes = state.attributes;
        return (*this);
    }

    Controller::ControllerState& Controller::ControllerState::operator= (const nn::hid::NpadJoyRightState& state) NN_NOEXCEPT
    {
        samplingNumber = state.samplingNumber;
        buttons = state.buttons;
        analogStickL = state.analogStickL;
        analogStickR = state.analogStickR;
        attributes = state.attributes;
        return (*this);
    }

    void Controller::Initialize(const InitializeInfo& info) NN_NOEXCEPT
    {
        m_IsNpad = info.isNpad;
        m_IsConnected = false;
        m_NpadId = info.npadId;

        m_ButtonUp.Reset();
        m_ButtonDown.Reset();
        m_Button.Reset();
        m_ButtonRepeat.Reset();

        m_StyleSet.Reset();
        m_Vibration.Initialize(m_NpadId);

        m_IrCameraHandle = nn::irsensor::GetIrCameraHandle(m_NpadId);
        nn::irsensor::Initialize(m_IrCameraHandle);

        for (auto& count : m_SixAxisSensorStateCounts) { count = 0; }

        // 初期化時に NFC デバイス情報を更新する
        UpdateNfcDevice();
    }

    void Controller::Update() NN_NOEXCEPT
    {
        // 振動処理
        m_Vibration.Update();

        // 現在のコントローラの有効なスタイルを取得
        m_StyleSet = nn::hid::GetNpadStyleSet(m_NpadId);

        for (auto& count : m_SixAxisSensorStateCounts) { count = 0; }

        if (m_StyleSet.CountPopulation() == 1)
        {
            // 6軸センサのハンドルを取得します
            m_SixAxisSensorHandleCount = nn::hid::GetSixAxisSensorHandles(m_SixAxisSensorHandles, nn::hid::NpadSixAxisSensorHandleCountMax, m_NpadId, m_StyleSet);

            // 6軸センサのステータスを取得します
            for (int i = 0; i < m_SixAxisSensorHandleCount; ++i)
            {
                nn::hid::GetSixAxisSensorState(&m_SixAxisSensorState[i], m_SixAxisSensorHandles[i]);

                m_SixAxisSensorStateCounts[i] = nn::hid::GetSixAxisSensorStates(m_SixAxisSensorStates[i], nn::hid::SixAxisSensorStateCountMax, m_SixAxisSensorHandles[i]);
            }
        }
        else
        {
            for (auto& state : m_SixAxisSensorState)
            {
                state = nn::hid::SixAxisSensorState();
            }
        }

        // ボタンや接続状態の更新
        if (m_StyleSet.Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::GetNpadState((nn::hid::NpadFullKeyState*)&m_State, m_NpadId);
        }
        else if (m_StyleSet.Test<nn::hid::NpadStyleHandheld>())
        {
            nn::hid::GetNpadState((nn::hid::NpadHandheldState*)&m_State, m_NpadId);
        }
        else if (m_StyleSet.Test<nn::hid::NpadStyleJoyDual>())
        {
            nn::hid::GetNpadState((nn::hid::NpadJoyDualState*)&m_State, m_NpadId);
        }
        else if (m_StyleSet.Test<nn::hid::NpadStyleJoyLeft>())
        {
            nn::hid::GetNpadState((nn::hid::NpadJoyLeftState*)&m_State, m_NpadId);
        }
        else if (m_StyleSet.Test<nn::hid::NpadStyleJoyRight>())
        {
            nn::hid::GetNpadState((nn::hid::NpadJoyRightState*)&m_State, m_NpadId);
        }
        else
        {
            m_State = nns::hidfw::hid::Controller::ControllerState();
        }

        nn::hid::NpadButtonSet button = m_State.buttons;

        // 横もちのボタン配置に変換します

        if (nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Horizontal)
        {
            button = ConvertHorizontalButtonLayout(m_StyleSet, m_State.buttons);

            if (m_StyleSet.Test<nn::hid::NpadStyleJoyLeft>())
            {
                std::swap(m_State.analogStickL.x, m_State.analogStickL.y);
                std::swap(m_State.analogStickR.x, m_State.analogStickR.y);
                m_State.analogStickL.x = -m_State.analogStickL.x;
                m_State.analogStickR.y = -m_State.analogStickR.y;
            }
            else if (m_StyleSet.Test<nn::hid::NpadStyleJoyRight>())
            {
                std::swap(m_State.analogStickL.x, m_State.analogStickL.y);
                std::swap(m_State.analogStickR.x, m_State.analogStickR.y);
                m_State.analogStickL.x = -m_State.analogStickL.x;
                m_State.analogStickR.y = -m_State.analogStickR.y;
            }
        }

        // 接続状態をクリアします
        m_IsConnected = m_State.attributes.Test<nn::hid::NpadJoyAttribute::IsConnected>();

        // 新たに押下されたデジタルボタンを最新の値に更新します。
        m_ButtonDown = ~m_Button & button;

        // 新たに開放されたデジタルボタンを最新の値に更新します。
        m_ButtonUp = m_Button & ~button;

        // 押下中のデジタルボタンを最新の値に更新します。
        m_Button = button;

        // デジタルボタンのリピート状態をリセットします。
        m_ButtonRepeat.Reset();

        for (size_t i = 0; i < RepeatButtonCounter::ButtonCountMax; ++i)
        {
            if (button.Test(i))
            {
                if (
                    (m_ButtonDown.Test(i)) ||
                    ((m_RepeatCounter.counter[i] > m_RepeatCounter.delay) &&
                    ((m_RepeatCounter.counter[i] - m_RepeatCounter.delay) % m_RepeatCounter.interval == 0))
                    )
                {
                    m_ButtonRepeat.Set(i, true);
                }
                // デジタルボタンが押下状態であれば経過フレーム数を更新します。
                ++m_RepeatCounter.counter[i];
            }
            else
            {
                // デジタルボタンが押下状態でなければ経過フレーム数をリセットします。
                m_RepeatCounter.counter[i] = 0;
            }
        }
    }

    void Controller::UpdateNfcDevice() NN_NOEXCEPT
    {
        // NFC デバイスハンドルの更新を行います
        int handleCount = 0;
        nn::nfc::DeviceHandle handles[nn::nfc::DeviceCountMax];
        nn::nfc::ListDevices(handles, &handleCount, nn::nfc::DeviceCountMax);

        nn::hid::NpadIdType id;
        m_NfcDeviceHandleCount = 0;
        for (int i = 0; i < handleCount; ++i)
        {
            if (
                nn::nfc::GetNpadId(&id, handles[i]).IsSuccess() &&
                id == m_NpadId
                )
            {
                m_NfcDeviceHandles[m_NfcDeviceHandleCount++] = handles[i];
            }
        }
    }

    nns::hidfw::hid::Vibration& Controller::GetVibration() NN_NOEXCEPT
    {
        return m_Vibration;
    }

    nns::hidfw::hid::Controller::ControllerState Controller::GetState() NN_NOEXCEPT
    {
        return m_State;
    }

    void Controller::SetJoyAssignmentMode(const nn::hid::NpadJoyAssignmentMode& mode) const NN_NOEXCEPT
    {
        (mode == nn::hid::NpadJoyAssignmentMode_Single) ? nn::hid::SetNpadJoyAssignmentModeSingle(m_NpadId) : nn::hid::SetNpadJoyAssignmentModeDual(m_NpadId);
    }

    nn::hid::NpadJoyAssignmentMode Controller::GetJoyAssignmentMode() const NN_NOEXCEPT
    {
        return nn::hid::GetNpadJoyAssignment(m_NpadId);
    }

    void Controller::SwapNpadAssignment(nn::hid::NpadIdType id) const NN_NOEXCEPT
    {
        nn::hid::SwapNpadAssignment(m_NpadId, id);
    }

    void Controller::SwapNpadAssignment(const Controller& controller) const NN_NOEXCEPT
    {
        SwapNpadAssignment(controller.GetNpadId());
    }

    void Controller::MergeSingleJoyAsDualJoy(nn::hid::NpadIdType id) const NN_NOEXCEPT
    {
        nn::hid::MergeSingleJoyAsDualJoy(m_NpadId, id);
    }

    void Controller::MergeSingleJoyAsDualJoy(const Controller& controller) const NN_NOEXCEPT
    {
        MergeSingleJoyAsDualJoy(controller.GetNpadId());
    }

    void Controller::StartSixAxisSensor() const NN_NOEXCEPT
    {
        StartSixAxisSensor(
            nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStyleFullKey::Mask |
            nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleJoyLeft::Mask |
            nn::hid::NpadStyleJoyRight::Mask
        );
    }

    void Controller::StartSixAxisSensor(nn::hid::NpadStyleSet style) const NN_NOEXCEPT
    {
        // 6軸センサのハンドルを取得します
        nn::hid::SixAxisSensorHandle handle[nn::hid::NpadSixAxisSensorHandleCountMax];
        nn::hid::NpadStyleSet targetStyleSet = style;
        nn::hid::NpadStyleSet styleSet;

        size_t enableStyleCount = style.CountPopulation();

        if (m_NpadId == nn::hid::NpadId::Handheld)
        {
            targetStyleSet.Reset();
            targetStyleSet.Set<nn::hid::NpadStyleHandheld>(style.Test<nn::hid::NpadStyleHandheld>());
        }
        else
        {
            targetStyleSet.Reset<nn::hid::NpadStyleHandheld>();
        }
        for (auto styleCount = 0; styleCount < style.GetCount() && enableStyleCount > 0; ++styleCount)
        {
            styleSet.Reset();
            styleSet.Set(styleCount, true);
            if ((styleSet & targetStyleSet).IsAnyOn())
            {
                auto handleCount = nn::hid::GetSixAxisSensorHandles(handle, nn::hid::NpadSixAxisSensorHandleCountMax, m_NpadId, styleSet);
                for (int i = 0; i < handleCount; ++i)
                {
                    nn::hid::StartSixAxisSensor(handle[i]);
                }
                --enableStyleCount;
            }
        }
    }

    void Controller::StopSixAxisSensor() const NN_NOEXCEPT
    {
        StopSixAxisSensor(
            nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStyleFullKey::Mask |
            nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleJoyLeft::Mask |
            nn::hid::NpadStyleJoyRight::Mask
        );
    }

    void Controller::StartIrSensor(nn::hid::NpadIdType id) const NN_NOEXCEPT
    {
        nn::irsensor::IrCameraHandle irCameraHandle;
        irCameraHandle = nn::irsensor::GetIrCameraHandle(id);

        nn::irsensor::MomentProcessorConfig config;
        //モーメントプロセッサのデフォルト設定の取得
        nn::irsensor::GetMomentProcessorDefaultConfig(&config);
        //モーメントプロセッサの開始
        nn::irsensor::RunMomentProcessor(irCameraHandle, config);
    }

    void Controller::StopIrSensor(nn::hid::NpadIdType id) const NN_NOEXCEPT
    {
        nn::irsensor::IrCameraHandle irCameraHandle;
        irCameraHandle = nn::irsensor::GetIrCameraHandle(id);

        nn::irsensor::StopImageProcessor(irCameraHandle);
    }

    bool Controller::StartNfc() const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(nn::nfc::GetState(), nn::nfc::State_Init);

        bool isStartNfcDevice = false;

        for (int i = 0; i < m_NfcDeviceHandleCount; ++i)
        {
            const auto deviceState = nn::nfc::GetDeviceState(m_NfcDeviceHandles[i]);
            if (
                deviceState == nn::nfc::DeviceState_Init ||
                deviceState == nn::nfc::DeviceState_Deactive
                )
            {
                if (nn::nfc::StartDetection(m_NfcDeviceHandles[i]).IsSuccess())
                {
                    isStartNfcDevice = true;
                }
            }
        }
        return isStartNfcDevice;
    }

    bool Controller::StopNfc() const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(nn::nfc::GetState(), nn::nfc::State_Init);

        bool isStopNfcDevice = true;

        for (int i = 0; i < m_NfcDeviceHandleCount; ++i)
        {
            if (nn::nfc::StopDetection(m_NfcDeviceHandles[i]).IsFailure())
            {
                isStopNfcDevice = false;
            }
        }
        return isStopNfcDevice;
    }

    void Controller::StopSixAxisSensor(nn::hid::NpadStyleSet style) const NN_NOEXCEPT
    {
        nn::hid::SixAxisSensorHandle handle[nn::hid::NpadSixAxisSensorHandleCountMax];
        nn::hid::NpadStyleSet targetStyleSet = style;
        nn::hid::NpadStyleSet styleSet;

        size_t enableStyleCount = style.CountPopulation();

        if (m_NpadId == nn::hid::NpadId::Handheld)
        {
            targetStyleSet.Reset();
            targetStyleSet.Set<nn::hid::NpadStyleHandheld>(style.Test<nn::hid::NpadStyleHandheld>());
        }
        else
        {
            targetStyleSet.Reset<nn::hid::NpadStyleHandheld>();
        }
        for (auto styleCount = 0; styleCount < style.GetCount() && enableStyleCount > 0; ++styleCount)
        {
            styleSet.Reset();
            styleSet.Set(styleCount, true);
            if ((styleSet & targetStyleSet).IsAnyOn())
            {
                auto handleCount = nn::hid::GetSixAxisSensorHandles(handle, nn::hid::NpadSixAxisSensorHandleCountMax, m_NpadId, styleSet);
                for (int i = 0; i < handleCount; ++i)
                {
                    nn::hid::StopSixAxisSensor(handle[i]);
                }
                --enableStyleCount;
            }
        }
    }

    nn::hid::NpadIdType Controller::GetNpadId() const NN_NOEXCEPT
    {
        return m_NpadId;
    }

    nn::hid::NpadStyleSet Controller::GetStyleSet() const NN_NOEXCEPT
    {
        return m_StyleSet;
    }

    bool Controller::isConnected() const NN_NOEXCEPT
    {
        return m_IsConnected;
    }


    nn::hid::NpadAttributesSet Controller::GetAttributesSet() const NN_NOEXCEPT
    {
        return m_State.attributes;
    }

    void Controller::Disconnect() const NN_NOEXCEPT
    {
        nn::hid::DisconnectNpad(m_IsNpad);
    }

    nn::hid::NpadButtonSet Controller::GetButtonSet() const NN_NOEXCEPT
    {
        return m_Button;
    }

    bool Controller::IsTrigger(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT
    {
        return (m_ButtonDown & button).IsAnyOn();
    }

    nn::hid::NpadButtonSet Controller::GetTrgButton() const NN_NOEXCEPT
    {
        return m_ButtonDown;;
    }

    bool Controller::IsHold(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT
    {
        return (m_Button & button).IsAnyOn();
    }

    nn::hid::NpadButtonSet Controller::GetHldButton() const NN_NOEXCEPT
    {
        return GetButtonSet();
    }

    uint64_t Controller::GetMaxHoldCount(const nn::hid::NpadButtonSet& inButtons, nn::hid::NpadButtonSet* outButton) const NN_NOEXCEPT
    {
        uint64_t resultCount = 0;
        int count = 0;
        nn::hid::NpadButtonSet targetButtonSet = inButtons;
        nn::hid::NpadButtonSet resultButtonSet = inButtons;

        while (targetButtonSet.CountPopulation() > 0 && count < nn::hid::NpadButtonSet::StorageBitCount)
        {
            if (targetButtonSet.Test(count))
            {
                if (m_RepeatCounter.counter[count] > resultCount)
                {
                    resultCount = m_RepeatCounter.counter[count];
                    resultButtonSet.Reset();
                    resultButtonSet.Set(count, true);
                }
            }
            targetButtonSet.Set(count++, false);
        }
        if (outButton != nullptr)
        {
            *outButton = resultButtonSet;
        }
        return resultCount;
    }

    bool Controller::IsRepeat(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT
    {
        if (button.CountPopulation() <= 1)
        {
            return (m_ButtonRepeat & button).IsAnyOn();
        }
        else
        {
            nn::hid::NpadButtonSet buttonSet;
            buttonSet.Reset();
            if (GetMaxHoldCount(button, &buttonSet) > 0)
            {
                return IsRepeat(buttonSet);
            }
        }
        return false;
    }

    nn::hid::NpadButtonSet Controller::GetRepButton() const NN_NOEXCEPT
    {
        return m_ButtonRepeat;
    }

    bool Controller::IsRelease(const nn::hid::NpadButtonSet& button) const NN_NOEXCEPT
    {
        return (m_ButtonUp & button).IsAnyOn();
    }

    nn::hid::NpadButtonSet Controller::GetRelButton() const NN_NOEXCEPT
    {
        return m_ButtonUp;
    }

    uint8_t Controller::GetPlayerLedPattern() const NN_NOEXCEPT
    {
        return nn::hid::GetPlayerLedPattern(m_NpadId);
    }

    nn::hid::AnalogStickState Controller::GetLeftAnalogStickState() const NN_NOEXCEPT
    {
        return m_State.analogStickL;
    }

    Controller::NormalizedAnalogStickState Controller::GetNormalizedLeftAnalogStickState() const NN_NOEXCEPT
    {
        NormalizedAnalogStickState result;
        result.x = static_cast<float>(m_State.analogStickL.x) / static_cast<float>(nn::hid::AnalogStickMax);
        result.y = static_cast<float>(m_State.analogStickL.y) / static_cast<float>(nn::hid::AnalogStickMax);
        return result;
    }

    nn::hid::AnalogStickState Controller::GetRightAnalogStickState() const NN_NOEXCEPT
    {
        return m_State.analogStickR;
    }

    Controller::NormalizedAnalogStickState Controller::GetNormalizedRightAnalogStickState() const NN_NOEXCEPT
    {
        NormalizedAnalogStickState result;
        result.x = static_cast<float>(m_State.analogStickR.x) / static_cast<float>(nn::hid::AnalogStickMax);
        result.y = static_cast<float>(m_State.analogStickR.y) / static_cast<float>(nn::hid::AnalogStickMax);
        return result;
    }

    int Controller::GetSixAxisSensorState(int num, nn::hid::SixAxisSensorState* sensorState) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(sensorState);
        int cnt;
        for (cnt = 0; (cnt < nn::hid::NpadSixAxisSensorHandleCountMax) && (cnt < num); ++cnt)
        {
            if (cnt < m_SixAxisSensorHandleCount)
            {
                sensorState[cnt] = m_SixAxisSensorState[cnt];
            }
            else
            {
                //sensorState[cnt] = nn::hid::SixAxisSensorState();
                break;
            }
        }
        return (cnt < m_SixAxisSensorHandleCount) ? cnt : m_SixAxisSensorHandleCount;
    }

    bool Controller::GetNfcDeviceHandles(nn::nfc::DeviceHandle* pOutHandle, int* pOutCount, const int handleCount) const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(nn::nfc::GetState(), nn::nfc::State_Init);
        NN_ASSERT_LESS_EQUAL(handleCount, nn::nfc::DeviceCountMax);
        NN_ASSERT_NOT_NULL(pOutHandle);
        NN_ASSERT_NOT_NULL(pOutCount);

        bool result = false;
        int& outCount = *pOutCount;

        outCount = 0;
        for (int i = 0; i < m_NfcDeviceHandleCount; ++i)
        {
            pOutHandle[outCount++] = m_NfcDeviceHandles[i];
            result = true;
        }
        return result;
    }

    bool Controller::GetNfcDeviceState(nn::nfc::DeviceState* pOutState, int* pOutCount, const int stateCount) const NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(nn::nfc::GetState(), nn::nfc::State_Init);
        NN_ASSERT_LESS_EQUAL(stateCount, nn::nfc::DeviceCountMax);
        NN_ASSERT_NOT_NULL(pOutState);
        NN_ASSERT_NOT_NULL(pOutCount);

        bool result = false;
        int& outCount = *pOutCount;

        outCount = 0;
        for (int i = 0; i < m_NfcDeviceHandleCount; ++i)
        {
            pOutState[outCount++] = nn::nfc::GetDeviceState(m_NfcDeviceHandles[i]);
            result = true;
        }
        return result;
    }

    int Controller::GetSixAxisSensorStates(nn::hid::NpadJoyDeviceType deviceType, nn::hid::SixAxisSensorState* outStates, int count) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(outStates);
        int cnt = 0;

        if (
            (m_StyleSet.Test<nn::hid::NpadStyleJoyLeft>() && deviceType == nn::hid::NpadJoyDeviceType_Right) ||
            (m_StyleSet.Test<nn::hid::NpadStyleJoyRight>() && deviceType == nn::hid::NpadJoyDeviceType_Left) ||
            (m_SixAxisSensorHandleCount == 0)
            )
        {
            return 0;
        }

        auto handleNumber = (m_SixAxisSensorHandleCount == 1 || deviceType == nn::hid::NpadJoyDeviceType_Left) ? 0 : 1;

        for (; (cnt < m_SixAxisSensorStateCounts[handleNumber]) && (cnt < count); ++cnt)
        {
            outStates[cnt] = m_SixAxisSensorStates[handleNumber][cnt];
        }
        return cnt;
    }

    bool Controller::IsSixAxisSensorAtRest(nn::hid::NpadJoyDeviceType deviceType) const NN_NOEXCEPT
    {
        if (
            (m_StyleSet.Test<nn::hid::NpadStyleJoyLeft>() && deviceType == nn::hid::NpadJoyDeviceType_Right) ||
            (m_StyleSet.Test<nn::hid::NpadStyleJoyRight>() && deviceType == nn::hid::NpadJoyDeviceType_Left) ||
            (m_SixAxisSensorHandleCount == 0)
            )
        {
            return true;
        }

        auto handleNumber = (m_SixAxisSensorHandleCount == 1 || deviceType == nn::hid::NpadJoyDeviceType_Left) ? 0 : 1;

        if (handleNumber < m_SixAxisSensorHandleCount)
        {
            return  nn::hid::IsSixAxisSensorAtRest(m_SixAxisSensorHandles[handleNumber]);
        }
        return true;
    }

    nn::hid::NpadButtonSet Controller::ConvertHorizontalButtonLayout(const nn::hid::NpadStyleSet& style, const nn::hid::NpadButtonSet& button) NN_NOEXCEPT
    {
        nn::hid::NpadButtonSet convButton = button;
        if (style.Test<nn::hid::NpadStyleJoyLeft>())
        {
            convButton.Set<nn::hid::NpadButton::A>(button.Test<nn::hid::NpadButton::Down>());
            convButton.Set<nn::hid::NpadButton::B>(button.Test<nn::hid::NpadButton::Left>());
            convButton.Set<nn::hid::NpadButton::X>(button.Test<nn::hid::NpadButton::Right>());
            convButton.Set<nn::hid::NpadButton::Y>(button.Test<nn::hid::NpadButton::Up>());

            convButton.Set<nn::hid::NpadButton::Right>(button.Test<nn::hid::NpadButton::StickLDown>());
            convButton.Set<nn::hid::NpadButton::Down>(button.Test<nn::hid::NpadButton::StickLLeft>());
            convButton.Set<nn::hid::NpadButton::Up>(button.Test<nn::hid::NpadButton::StickLRight>());
            convButton.Set<nn::hid::NpadButton::Left>(button.Test<nn::hid::NpadButton::StickLUp>());

            convButton.Set<nn::hid::NpadButton::L>(button.Test<nn::hid::NpadJoyButton::L>() | button.Test<nn::hid::NpadJoyButton::LeftSL>());
            convButton.Set<nn::hid::NpadButton::R>(button.Test<nn::hid::NpadJoyButton::LeftSR>());

            convButton.Set<nn::hid::NpadButton::ZL>(button.Test<nn::hid::NpadJoyButton::ZL>());

            convButton.Set<nn::hid::NpadButton::StickLRight>(button.Test<nn::hid::NpadButton::StickLDown>());
            convButton.Set<nn::hid::NpadButton::StickLDown>(button.Test<nn::hid::NpadButton::StickLLeft>());
            convButton.Set<nn::hid::NpadButton::StickLUp>(button.Test<nn::hid::NpadButton::StickLRight>());
            convButton.Set<nn::hid::NpadButton::StickLLeft>(button.Test<nn::hid::NpadButton::StickLUp>());

            convButton.Set<nn::hid::NpadJoyButton::LeftSL>(false);
            convButton.Set<nn::hid::NpadJoyButton::LeftSR>(false);
        }
        else if (style.Test<nn::hid::NpadStyleJoyRight>())
        {
            convButton.Set<nn::hid::NpadButton::A>(button.Test<nn::hid::NpadButton::X>());
            convButton.Set<nn::hid::NpadButton::B>(button.Test<nn::hid::NpadButton::A>());
            convButton.Set<nn::hid::NpadButton::X>(button.Test<nn::hid::NpadButton::Y>());
            convButton.Set<nn::hid::NpadButton::Y>(button.Test<nn::hid::NpadButton::B>());

            convButton.Set<nn::hid::NpadButton::Right>(button.Test<nn::hid::NpadButton::StickRUp>());
            convButton.Set<nn::hid::NpadButton::Down>(button.Test<nn::hid::NpadButton::StickRRight>());
            convButton.Set<nn::hid::NpadButton::Up>(button.Test<nn::hid::NpadButton::StickRLeft>());
            convButton.Set<nn::hid::NpadButton::Left>(button.Test<nn::hid::NpadButton::StickRDown>());

            convButton.Set<nn::hid::NpadButton::L>(button.Test<nn::hid::NpadJoyButton::RightSL>());
            convButton.Set<nn::hid::NpadButton::R>(button.Test<nn::hid::NpadJoyButton::R>() | button.Test<nn::hid::NpadJoyButton::RightSR>());

            convButton.Set<nn::hid::NpadButton::ZR>(button.Test<nn::hid::NpadJoyButton::ZR>());

            convButton.Set<nn::hid::NpadButton::StickRRight>(button.Test<nn::hid::NpadButton::StickRUp>());
            convButton.Set<nn::hid::NpadButton::StickRDown>(button.Test<nn::hid::NpadButton::StickRRight>());
            convButton.Set<nn::hid::NpadButton::StickRUp>(button.Test<nn::hid::NpadButton::StickRLeft>());
            convButton.Set<nn::hid::NpadButton::StickRLeft>(button.Test<nn::hid::NpadButton::StickRDown>());

            convButton.Set<nn::hid::NpadJoyButton::RightSL>(false);
            convButton.Set<nn::hid::NpadJoyButton::RightSR>(false);
        }
        return convButton;
    }

}}}
