﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../hid.h"
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/hid/detail/hid_Log.h>

namespace nns { namespace hidfw { namespace hid {

    Input& Input::GetInstance() NN_NOEXCEPT
    {
        static Input instance;
        return instance;
    }

    void Input::Initialize() NN_NOEXCEPT
    {
        nn::hid::InitializeNpad();
        nn::nfc::Initialize();

        nn::hid::NpadIdType npadIds[] =
        {
            nn::hid::NpadId::No1,
            nn::hid::NpadId::No2,
            nn::hid::NpadId::No3,
            nn::hid::NpadId::No4,
            nn::hid::NpadId::No5,
            nn::hid::NpadId::No6,
            nn::hid::NpadId::No7,
            nn::hid::NpadId::No8,
            nn::hid::NpadId::Handheld
        };

        nn::hid::SetSupportedNpadIdType(
            npadIds,
            NN_ARRAY_SIZE(npadIds)
        );

        nn::hid::SetSupportedNpadStyleSet(
            nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStyleFullKey::Mask |
            nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleJoyLeft::Mask |
            nn::hid::NpadStyleJoyRight::Mask
        );

        gController.Initialize();
        gTouch.Initialize();
    }

    void Input::Update() NN_NOEXCEPT
    {
        gController.Update();
        gTouch.Update();
        WriteLog();
    }

    void Input::Finalize() NN_NOEXCEPT
    {
        gController.Finalize();
        StopKeyLogger();
    }

    void Input::SetLoggingTarget(const KeyLoggerTargetet& loggingTarget) NN_NOEXCEPT
    {
        m_OutputTarget = loggingTarget;
    }

    bool Input::StartKeyLogger(KeyLoggerOutputMode outputMode) NN_NOEXCEPT
    {
        if (!m_IsStartedKeyLogger)
        {
            switch (outputMode)
            {
            case nns::hidfw::hid::Input::KeyLoggerOutputMode_Sd:
            {
                const char* fileName = "sd:/Nintendo/Key.log";

                nn::Result result;

                // SD カードをマウント名 "sd" でマウントします。
                result = nn::fs::MountSdCardForDebug("sd");
                if (nn::fs::ResultSdCardAccessFailed::Includes(result))
                {
                    // SD カードが挿入されていません。
                    // SD カードの挿入を促すメッセージを表示してください。
                    NN_DETAIL_HID_INFO("SD card not found.");
                    return false;
                }

                // 上記以外の失敗は実装ミスのため、必ずアボートしてください。
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                // ファイルが存在しない場合は作成します。
                {
                    nn::fs::DirectoryEntryType directoryEntryType;
                    result = nn::fs::GetEntryType(&directoryEntryType, fileName);
                    if (nn::fs::ResultPathNotFound().Includes(result))
                    {
                        // 対象ファイルが存在しません。
                        // 対象ファイルを作成します。
                        NN_DETAIL_HID_INFO("Create \"%s\"\n", fileName);
                        result = nn::fs::CreateFile(fileName, 0);
                        if (nn::fs::ResultPathNotFound::Includes(result))
                        {
                            // 対象ファイルが存在しません。
                            // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
                        }
                        else if (nn::fs::ResultPathAlreadyExists::Includes(result)
                            || nn::fs::ResultTargetLocked::Includes(result))
                        {
                            // 対象ファイルが既に存在しています。
                            // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                            // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                            // 必要であれば対象ファイルを削除してから再度作成してください。
                        }
                        else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
                        {
                            // 空き領域が不足しています。
                            // SD カードの空き領域を確認してください。
                            NN_ABORT("Usable space not enough.\n");
                            return false;
                        }
                        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                        // これ以上のエラーハンドリングは不要です。
                    }
                    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                    // これ以上のエラーハンドリングは不要です。
                }

                result = nn::fs::OpenFile(&m_KeyLoggerFileHandle, fileName, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);

                if (nn::fs::ResultPathNotFound::Includes(result))
                {
                    // 対象ファイルが存在しません。
                    // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
                    return false;
                }
                else if (nn::fs::ResultTargetLocked::Includes(result))
                {
                    // 対象ファイルが既にオープンされています。
                    // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
                }
                break;
            }
            case nns::hidfw::hid::Input::KeyLoggerOutputMode_Host:
            {
                return false;
                break;
            }
            case nns::hidfw::hid::Input::KeyLoggerOutputMode_TargetManager:
                break;
            default : NN_UNEXPECTED_DEFAULT;
            }

            m_IsStartedKeyLogger = true;
            m_OutputMode = outputMode;
            return true;
        }
        return (outputMode == m_OutputMode);
    }

    bool Input::StartKeyLoggerHost(const char* outFilePath) NN_NOEXCEPT
    {
        if (!m_IsStartedKeyLogger)
        {
            nn::Result result;

            result = nn::fs::MountHost("host", outFilePath);
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                // ホスト PC 上に対象のパスが見つかりません。
                // ホスト PC 上に "C:/Windows/Temp" があるか確かめてください。
                NN_ASSERT(false, "Target directory not found.\n");
                return false;
            }
            else if (nn::fs::ResultTargetNotFound::Includes(result))
            {
                // ホスト PC が認識されませんでした。
                // ホスト PC に正しく接続してください。
                NN_ASSERT(false, "Host PC not found.\n");
                return false;
            }
            // 上記以外の失敗は実装ミスのため、必ずアボートしてください。
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            // ディレクトリ・ファイルを作成します。
            {
                NN_DETAIL_HID_INFO("Create\thost:/Log\n");
                result = nn::fs::CreateDirectory("host:/Log");
                if (nn::fs::ResultPathNotFound::Includes(result))
                {
                    // パスに含まれるディレクトリが存在しません。
                    // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
                }
                else if (nn::fs::ResultPathAlreadyExists::Includes(result)
                    || nn::fs::ResultTargetLocked::Includes(result))
                {
                    // 対象ディレクトリが既に存在しています。
                    // 対象ディレクトリを削除し再度作成する、無視して続行する等の処理を必要に応じて追加してください。
                }
                else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
                {
                    // ストレージに空き領域が不足しています。
                    // ホスト側ストレージの空き領域を増やしてください。
                    NN_ASSERT("Usable space not enough.\n");
                    return false;
                }

                // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                // これ以上のエラーハンドリングは不要です。

                NN_DETAIL_HID_INFO("Create\thost:/Log/Key.log\n");
                result = nn::fs::CreateFile("host:/Log/Key.log", 0);
                if (nn::fs::ResultPathNotFound::Includes(result))
                {
                    // パスに含まれるディレクトリが存在しません。
                    // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
                }
                else if (nn::fs::ResultPathAlreadyExists::Includes(result)
                    || nn::fs::ResultTargetLocked::Includes(result))
                {
                    // 対象ファイルが既に存在しています。
                    // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                    // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                    // 必要であれば対象ファイルを削除してから再度作成してください。
                }
                else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
                {
                    // ストレージに空き領域が不足しています。
                    // ホスト側ストレージの空き領域を増やしてください。
                    NN_ASSERT("Usable space not enough.\n");
                    return false;
                }
                // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                // これ以上のエラーハンドリングは不要です。
            }


            result = nn::fs::OpenFile(&m_KeyLoggerFileHandle, "host:/Log/Key.log", nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);

            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                // 対象ファイルが存在しません。
                // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
                return false;
            }
            else if (nn::fs::ResultTargetLocked::Includes(result))
            {
                // 対象ファイルが既にオープンされています。
                // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
            }

            m_IsStartedKeyLogger = true;
            m_OutputMode = KeyLoggerOutputMode::KeyLoggerOutputMode_Host;
            return true;
        }

        return (m_OutputMode == KeyLoggerOutputMode::KeyLoggerOutputMode_Host);
    }

    void Input::StopKeyLogger() NN_NOEXCEPT
    {
        if (m_IsStartedKeyLogger)
        {
            switch (m_OutputMode)
            {
            case nns::hidfw::hid::Input::KeyLoggerOutputMode_Host:
            {
                nn::fs::CloseFile(m_KeyLoggerFileHandle);
                nn::fs::Unmount("host");
                m_IsStartedKeyLogger = false;
                break;
            }
            case nns::hidfw::hid::Input::KeyLoggerOutputMode_Sd:
            {
                nn::fs::CloseFile(m_KeyLoggerFileHandle);
                nn::fs::Unmount("sd");
                m_IsStartedKeyLogger = false;
                break;
            }
            case nns::hidfw::hid::Input::KeyLoggerOutputMode_TargetManager:
            {
                m_IsStartedKeyLogger = false;
                break;
            }
            default : NN_UNEXPECTED_DEFAULT;
            }
        }
    }

    Input::Input() NN_NOEXCEPT :
        m_IsStartedKeyLogger(false),
        m_OutputMode(KeyLoggerOutputMode::KeyLoggerOutputMode_TargetManager)
    {
        m_OutputTarget.Reset();
    }

    void Input::WriteLog() NN_NOEXCEPT
    {
        if (m_IsStartedKeyLogger)
        {
            static uint64_t     writeCount = 0;
            static const int    bufferSize = 2048;

            static const char* ButtonText[] =
            {
                "A","B","X","Y","LS","RS","L","R","ZL","ZR",
                "+","-","<","^",">","V","<","^",">","V",
                "<","^",">","V","SL","SR","SL","SL","?","?",
                "?","?","?","?","?","?","?","?","?","?",
                "?","?","?","?","?","?","?","?","?","?",
                "?","?","?","?","?","?","?","?","?","?",
                "?","?","?","?"
            };

            static const char* StyleText[] =
            {
                "NpadStyleFullKey",
                "NpadStyleHandheld",
                "NpadStyleJoyDual",
                "NpadStyleJoyLeft",
                "NpadStyleJoyRight"
            };

            static const char* HoldTypeText[] =
            {
                "Vertical",
                "Horizontal"
            };

            static const char* NpadHandheldActivationModeText[] =
            {
                "Dual",
                "Single",
                "None",
            };

            nn::Result result;

            char* pBuffer = new char[bufferSize];

            int writePos = 0;

            writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "[%d]\n", writeCount);

            // 無線1コン
            {
                nn::hid::NpadButtonSet wirelessButtonSet = gController.GetController(nn::hid::NpadId::No1)->GetButtonSet();
                nn::hid::NpadStyleSet wirelessStyle = gController.GetController(nn::hid::NpadId::No1)->GetStyleSet();
                nn::hid::SixAxisSensorState wirelessSixAxisState[nn::hid::NpadSixAxisSensorHandleCountMax];
                size_t SixAxisStateCount = gController.GetController(nn::hid::NpadId::No1)->GetSixAxisSensorState(nn::hid::NpadSixAxisSensorHandleCountMax, wirelessSixAxisState);

                writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "No1,\n");
                // スタイルの表示
                if (m_OutputTarget.Test<KeyLoggerTarget::Style>())
                {
                    size_t enableStyleCount = wirelessStyle.CountPopulation();
                    for (size_t i = 0; (i < static_cast<size_t>(wirelessStyle.GetCount())) && (enableStyleCount > 0); ++i)
                    {
                        if (wirelessStyle.Test(i))
                        {
                            writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "%s,", (i < NN_ARRAY_SIZE(StyleText)) ? StyleText[i] : "?");
                            --enableStyleCount;
                        }
                    }
                    writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "\n");
                }
                // ボタンの入力状態の出力
                if (m_OutputTarget.Test<KeyLoggerTarget::ButtonSet>())
                {
                    for (auto i = 0; i < wirelessButtonSet.GetCount(); ++i)
                    {
                        writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "%s,", wirelessButtonSet.Test(i) ? ButtonText[i] : "_");
                    }
                    writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "\n");
                }
                // スティック
                if (m_OutputTarget.Test<KeyLoggerTarget::AnalogStick>())
                {
                    writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "(%05d:%05d),(%05d:%05d)\n",
                        gController.GetController(nn::hid::NpadId::No1)->GetLeftAnalogStickState().x,
                        gController.GetController(nn::hid::NpadId::No1)->GetLeftAnalogStickState().y,
                        gController.GetController(nn::hid::NpadId::No1)->GetRightAnalogStickState().x,
                        gController.GetController(nn::hid::NpadId::No1)->GetRightAnalogStickState().y
                    );
                }
                // 6軸センサ
                if (m_OutputTarget.Test<KeyLoggerTarget::SixAxisSensor>())
                {
                    for (size_t i = 0; i < SixAxisStateCount; ++i)
                    {
                        writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "%llu,%s,%s,(%.5f:%.5f:%.5f),(%.5f:%.5f:%.5f),\n",
                            wirelessSixAxisState[i].samplingNumber,
                            wirelessSixAxisState[i].attributes.Test<nn::hid::SixAxisSensorAttribute::IsConnected>() ? "Connected" : "_",
                            wirelessSixAxisState[i].attributes.Test<nn::hid::SixAxisSensorAttribute::IsInterpolated>() ? "Interpolated" : "_",
                            wirelessSixAxisState[i].acceleration.x,
                            wirelessSixAxisState[i].acceleration.y,
                            wirelessSixAxisState[i].acceleration.z,
                            wirelessSixAxisState[i].angularVelocity.x,
                            wirelessSixAxisState[i].angularVelocity.y,
                            wirelessSixAxisState[i].angularVelocity.z
                        );
                    }
                }
                // 持ち方
                if (m_OutputTarget.Test<KeyLoggerTarget::HoldType>())
                {
                    writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "%s\n", HoldTypeText[nn::hid::GetNpadJoyHoldType()]);
                }
            }

            // 本体装着コン
            {
                nn::hid::NpadButtonSet handheldButtonSet = gController.GetController(nn::hid::NpadId::Handheld)->GetButtonSet();
                nn::hid::NpadStyleSet handheldStyle = gController.GetController(nn::hid::NpadId::Handheld)->GetStyleSet();
                nn::hid::SixAxisSensorState handheldSixAxisState[nn::hid::NpadSixAxisSensorHandleCountMax];
                auto SixAxisStateCount = gController.GetController(nn::hid::NpadId::Handheld)->GetSixAxisSensorState(nn::hid::NpadSixAxisSensorHandleCountMax, handheldSixAxisState);

                writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "Handheld,\n");
                // スタイルの表示
                if (m_OutputTarget.Test < KeyLoggerTarget::Style> ())
                {
                    auto enableStyleCount = handheldStyle.CountPopulation();
                    for (size_t i = 0; (i < static_cast<size_t>(handheldStyle.GetCount())) && (enableStyleCount > 0); ++i)
                    {
                        if (handheldStyle.Test(i))
                        {
                            writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "%s,", (i < NN_ARRAY_SIZE(StyleText)) ? StyleText[i] : "?");
                            --enableStyleCount;
                        }
                    }
                    writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "\n");
                }
                // ボタンの入力状態の出力
                if (m_OutputTarget.Test<KeyLoggerTarget::ButtonSet>())
                {
                    for (auto i = 0; i < handheldButtonSet.GetCount(); ++i)
                    {
                        writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "%s,", handheldButtonSet.Test(i) ? ButtonText[i] : "_");
                    }
                    writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "\n");
                }
                // スティック
                if (m_OutputTarget.Test<KeyLoggerTarget::AnalogStick>())
                {
                    writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "(%05d:%05d),(%05d:%05d)\n",
                        gController.GetController(nn::hid::NpadId::No1)->GetLeftAnalogStickState().x,
                        gController.GetController(nn::hid::NpadId::No1)->GetLeftAnalogStickState().y,
                        gController.GetController(nn::hid::NpadId::No1)->GetRightAnalogStickState().x,
                        gController.GetController(nn::hid::NpadId::No1)->GetRightAnalogStickState().y
                    );
                }
                // 6軸センサ
                if (m_OutputTarget.Test<KeyLoggerTarget::SixAxisSensor>())
                {
                    for (int i = 0; i < SixAxisStateCount; ++i)
                    {
                        writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "%llu,%s,%s,(%.5f:%.5f:%.5f),(%.5f:%.5f:%.5f),\n",
                            handheldSixAxisState[i].samplingNumber,
                            handheldSixAxisState[i].attributes.Test<nn::hid::SixAxisSensorAttribute::IsConnected>() ? "Connected" : "_",
                            handheldSixAxisState[i].attributes.Test<nn::hid::SixAxisSensorAttribute::IsInterpolated>() ? "Interpolated" : "_",
                            handheldSixAxisState[i].acceleration.x,
                            handheldSixAxisState[i].acceleration.y,
                            handheldSixAxisState[i].acceleration.z,
                            handheldSixAxisState[i].angularVelocity.x,
                            handheldSixAxisState[i].angularVelocity.y,
                            handheldSixAxisState[i].angularVelocity.z
                        );
                    }
                }
                // 有効になる本数
                if (m_OutputTarget.Test<KeyLoggerTarget::HandheldActivationMode>())
                {
                    writePos += nn::util::SNPrintf(&pBuffer[writePos], bufferSize - writePos, "%s\n", NpadHandheldActivationModeText[nn::hid::GetNpadHandheldActivationMode()]);
                }
            }

            switch (m_OutputMode)
            {
            case nns::hidfw::hid::Input::KeyLoggerOutputMode_Host:
            case nns::hidfw::hid::Input::KeyLoggerOutputMode_Sd:
            {
                int64_t fileSize = 0;

                // ファイルサイズの取得に失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
                (void)nn::fs::GetFileSize(&fileSize, m_KeyLoggerFileHandle);

                result = nn::fs::WriteFile(m_KeyLoggerFileHandle, fileSize, pBuffer, writePos, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
                {
                    if (pBuffer)
                    {
                        delete[] pBuffer;
                    }
                    // 空き領域が不足しています。
                    // 書き込みサイズが想定範囲内の値となっているか、SD カードの空き領域は十分かを確認してください。
                    NN_ABORT("Usable space not enough.\n");
                    return;
                }
                break;
            }
            case nns::hidfw::hid::Input::KeyLoggerOutputMode_TargetManager:
            {
                NN_DETAIL_HID_INFO("%s", pBuffer);
                break;
            }
            default : NN_UNEXPECTED_DEFAULT;
            }

            if (pBuffer)
            {
                delete[] pBuffer;
                pBuffer = nullptr;
            }

            ++writeCount;
        }
    } // NOLINT(readability/fn_size)
}}}
