﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       線や図形の描画を行うための API の宣言
 */

#pragma once

#include <nn/nn_Assert.h>
#include "gfx_GraphicsSystem.h"

namespace nns { namespace hidfw { namespace gfx {

    /**
     * @brief       線や図形の描画を行う処理群を纏めたクラスです
     * @details     線や図形、キャプチャ画面等を簡単に描画することが出来ます
     */
    class GraphicsDrawer
    {
        NN_DISALLOW_COPY(GraphicsDrawer);
        NN_DISALLOW_MOVE(GraphicsDrawer);
    public:
        enum GradationDirection
        {
            GradationDirection_None    = 0,                         //! グラデーションをかけません
            GradationDirection_Left    = 1,                         //! 右手側から左手側にグラデーションをかけます
            GradationDirection_Up,                                  //! 下から上に向かってグラデーションをかけます
            GradationDirection_Right,                               //! 左手側から右手側にグラデーションをかけます
            GradationDirection_Down,                                //! 上から下に向かってグラデーションをかけます
            GradationDirection_In,                                  //! [TBD] 内側に向かってグラデーションをかけます
            GradationDirection_Out                                  //! [TBD] 外側に向かってグラデーションをかけます
        };
        enum State
        {
            State_Initialized   = 0,                                //! 初期化は完了したが描画を行っていない状態
            State_Started       = 1,                                //! 描画が開始しています
            State_End           = 2,                                //! 描画が終了しています
        };
        enum ScreenShotSlot
        {
            ScreenShotSlot_Slot_1 = 0,
            ScreenShotSlot_Slot_2 = 1,
            ScreenShotSlot_Slot_3,
            ScreenShotSlot_Slot_4,
            ScreenShotSlot_Slot_5,
            ScreenShotSlot_Slot_6,
            ScreenShotSlot_Slot_7,
            ScreenShotSlot_Slot_8,
            ScreenShotSlot_Slot_Num,
            ScreenShotSlot_Slot_Auto = ScreenShotSlot_Slot_1
        };
        enum AntiAliasingType
        {
            AntiAliasingType_None = 0,                              //! ジャギけしを行いません
            AntiAliasingType_Pseudo = 1,                            //! 描画対象を外側方向に1px拡張し擬似的なジャギ消しを行う (高付加)
            AntiAliasingType_Num
        };

    public:
        //! @brief リソースを開放します
        ~GraphicsDrawer() NN_NOEXCEPT;

        //! @brief GraphicsDrawer のインスタンスを取得します
        static GraphicsDrawer& GetInstance() NN_NOEXCEPT;

        //! @brief PrimitiveRenderer を取得します
        nns::gfx::PrimitiveRenderer::Renderer& GetPrimitiveRenderer() NN_NOEXCEPT;

        //! @brief GraphicsSystem を取得します
        nns::hidfw::gfx::GraphicsSystem& GetGraphicsSystem() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief  GraphicsDrawer を初期化します
        //! @details    nns::hidfw::gfx::GraphicFramework を使用する場合は
        //!             本関数を呼び出す必要はありません
        //! @param[in] system   GraphicsSystem へのポインタ
        //----------------------------------------------------------------
        void Initialize(nns::hidfw::gfx::GraphicsSystem* system) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief  GraphicsDrawer の終了処理を行います
        //----------------------------------------------------------------
        void Finalize() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief  アンチエイリアスの種類を設定します
        //! @param[in] type アンチエイリアスの種類
        //----------------------------------------------------------------
        void SetAntiAliasingType(AntiAliasingType type) NN_NOEXCEPT { m_AntiAliasingType = type; }

        //----------------------------------------------------------------
        //! @brief  現在適応中のアンチエイリアスの種類を返します
        //! @return 現在適応中のアンチエイリアスの種類を返します
        //----------------------------------------------------------------
        AntiAliasingType GetAntiAliasingType() NN_NOEXCEPT { return m_AntiAliasingType; }

        //----------------------------------------------------------------
        //! @brief 描画対象を標準のものに戻します
        //! @details レンダーターゲットを標準のカラーターゲットに戻します
        //----------------------------------------------------------------
        void ResetRenderTarget() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 描画対象の指定するコマンドを発行します
        //! @details レンダーターゲットを指定されたカラーターゲットに変更します
        //!          深度バッファの変更は行いません
        //! @param[in] pColorTargetView カラーターゲット
        //----------------------------------------------------------------
        void SetRenderTarget(nn::gfx::ColorTargetView* pColorTargetView) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 描画対象の指定するコマンドを発行します
        //! @details レンダーターゲットを指定されたカラーターゲットに変更します
        //! @param[in] pColorTargetView カラーターゲット
        //! @param[in] pColorTargetView 深度バッファ
        //----------------------------------------------------------------
        void SetRenderTarget(nn::gfx::ColorTargetView* pColorTargetView, nn::gfx::DepthStencilView* pDepthStencilView) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief サンプラを取得します
        //----------------------------------------------------------------
        nn::gfx::DescriptorSlot GetSamplerDescriptorSlot() NN_NOEXCEPT
        {
            return m_SamplerDescriptor;
        }

        //----------------------------------------------------------------
        //! @brief 画面のクリアカラーを取得します
        //! @param[in] color    クリアカラー
        //----------------------------------------------------------------
        nn::util::Color4u8 GetClearColor() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 画面のクリアカラーを設定します
        //! @param[in] color    クリアカラー
        //----------------------------------------------------------------
        void SetClearColor(const nn::util::Color4u8& color) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 描画色を設定します
        //! @param[in] color    描画色
        //----------------------------------------------------------------
        void SetColor(const nn::util::Color4u8& color) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 描画色を取得します
        //! @details    グラデーションが設定されていた場合変化前の色を返します
        //! @return 描画色
        //----------------------------------------------------------------
        nn::util::Color4u8 GetColor() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 線分の太さを設定します
        //! @param[in] width    太さ
        //----------------------------------------------------------------
        void SetLineWidth(const float width) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 線分の太さを取得します
        //----------------------------------------------------------------
        float GetLineWidth() const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 描画色を設定します(グラデーション)
        //! @details    グラデーション機能が有効な描画関数は
        //!             Draw2DRoundedRect と Draw2DLine の2点です
        //!             グラデーションが無効な図形の場合、始めに指定した色を使用します
        //! @param[in] dir      描画方向
        //! @param[in] color    グラデーション開始時の描画色
        //! @param[in] color2   グラデーション終了時の描画色
        //----------------------------------------------------------------
        void SetColor(const nns::hidfw::gfx::GraphicsDrawer::GradationDirection dir, const nn::util::Color4u8& color, const nn::util::Color4u8& color2) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 描画色を取得します
        //! @details    グラデーションが設定されていない場合、color2 には color と同じ色が設定されます
        //! @param[out] color   グラデーション開始時の描画色
        //! @param[out] color2  グラデーション終了時の描画色
        //! @return グラデーションの方向
        //----------------------------------------------------------------
        nns::hidfw::gfx::GraphicsDrawer::GradationDirection GetColor(nn::util::Color4u8* color, nn::util::Color4u8* color2) const NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief ジャギ消しを有効にするか否か設定します
        //! @param[in] dir      有効・無効状態
        //----------------------------------------------------------------
        //void SetAntiAliasingFlag(AntiAliasingFlag flag) { m_AntiAliasingFlag = flag; }

        //----------------------------------------------------------------
        //! @brief ジャギ消しが有効になっているか否か返します
        //! @return 有効・無効状態
        //----------------------------------------------------------------
        //AntiAliasingFlag GetAntiAliasingFlag() const { return m_AntiAliasingFlag; }

        //----------------------------------------------------------------
        //! @brief 描画コマンドの受付を開始します
        //! @details 画面のクリアも同時に実行されます
        //----------------------------------------------------------------
        void BeginDraw() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 描画コマンドの受付を終了します
        //! @details 画面への描画が実行されます
        //----------------------------------------------------------------
        void EndDraw() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 現在の画面を記憶します
        //! @details    現在の画面の内容を保存します
        //!             実行完了まで1フレーム分のレイテンシが発生します
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //! @param[in]  scale    スケール
        //----------------------------------------------------------------
        void ScreenShot(const nn::util::Float2& scale) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 現在の画面を記憶します
        //! @details    現在の画面の内容を保存します
        //!             実行完了まで1フレーム分のレイテンシが発生します
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void ScreenShot() NN_NOEXCEPT { ScreenShot(NN_UTIL_FLOAT_2_INITIALIZER(1.f, 1.f)); }

        //----------------------------------------------------------------
        //! @brief 現在の画面を記憶します
        //! @details    現在の画面の内容を保存します
        //!             実行完了まで1フレーム分のレイテンシが発生します
        //!             最大 8 つまで画面の内容を保持する事が出来ます
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //! @param[in]  slot     保存先のスロット (デフォルトではScreenShotSlot_Slot_1に保存されます)
        //! @param[in]  scale    スケール
        //----------------------------------------------------------------
        void ScreenShot(ScreenShotSlot slot, const nn::util::Float2& scale) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 現在の画面を記憶します
        //! @details    現在の画面の内容を保存します
        //!             実行完了まで1フレーム分のレイテンシが発生します
        //!             最大 8 つまで画面の内容を保持する事が出来ます
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //! @param[in]  slot     保存先のスロット (デフォルトではScreenShotSlot_Slot_1に保存されます)
        //----------------------------------------------------------------
        void ScreenShot(ScreenShotSlot slot) NN_NOEXCEPT { ScreenShot(slot, NN_UTIL_FLOAT_2_INITIALIZER(1.f, 1.f)); }

        //----------------------------------------------------------------
        //! @brief 直前のフレームの画面を保持するようにします
        //----------------------------------------------------------------
        void StartScanScreen() NN_NOEXCEPT { m_isEnableScreenScanJob = true; }

        //----------------------------------------------------------------
        //! @brief 直前のフレームの画面を保持しないようにします
        //----------------------------------------------------------------
        void StopScanScreen() NN_NOEXCEPT { m_isEnableScreenScanJob = false; }

        //
        // 描画関数
        //

        //----------------------------------------------------------------
        //! @brief 矩形を描画します
        //! @param[in]  pos     中心点
        //! @param[in]  radius  半径
        //! @param[in]  angle   三角形の回転角度
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DTriangle(const nn::util::Float2& center, const float radius, const float angle) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 矩形を描画します
        //! @param[in]  pos     中心点
        //! @param[in]  radius  半径
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DTriangle(const nn::util::Float2& center, const float radius) NN_NOEXCEPT { Draw2DTriangle(center, radius, 0.f); }

        //----------------------------------------------------------------
        //! @brief 矩形を描画します
        //! @param[in]  pos      左上の座標
        //! @param[in]  size     矩形のサイズ
        //! @param[in]  uvPos    テクスチャの切り出し座標 (0.f..1.f)
        //! @param[in]  uvSize   テクスチャの切り出しサイズ (0.f..1.f)
        //! @param[in]  pSlot    テクスチャの格納先
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DRect(const nn::util::Float2& pos, const nn::util::Float2& size, const nn::util::Float2& uvPos, const nn::util::Float2& uvSize, nn::gfx::DescriptorSlot* pSlot) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 矩形を描画します
        //! @param[in]  pos      左上の座標
        //! @param[in]  size     矩形のサイズ
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DRect(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 枠線を表示します
        //! @details    枠線は矩形の外側に向かって表示されます
        //!             borderSize を負の値にした場合は内側に向かって表示されます
        //!             本関数ではグラデーションの方向設定は無効化され
        //!             ボーダーの外側(borderSize が負の場合は内側)に向かってグラデーションがかかります
        //! @param[in]   pos          左上の座標
        //! @param[in]  size         矩形のサイズ
        //! @param[in]  borderSize   線分の太さ
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DFrame(const nn::util::Float2& pos, const nn::util::Float2& size, const float borderSize) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 角を丸めた矩形を描画します
        //! @details    円形の0・90・180・270度部分を引き伸ばして生成します
        //!             角の丸みは round に 1.f を設定した場合に最も強くなります
        //!             丸みが最大の場合、矩形の四隅は縦幅、横幅のサイズで
        //!             小さい方の値 / 2.f の半径を持つ円となります
        //! @param[in]  pos      左上の座標
        //! @param[in]  size     図形のサイズ
        //! @param[in]  round    角の丸み
        //! @param[in]  div      角の丸みの分割数 ( div >= 4 の4の倍数 )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DRoundedRect(const nn::util::Float2& pos, const nn::util::Float2& size, const float round, const int div) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief (TODO) 角を丸めた矩形の枠線を描画します
        //! @details    枠線は矩形の外側に向かって表示されます
        //!             borderSize を負の値にした場合は内側に向かって表示されます
        //!             本関数ではグラデーションの方向設定は無効化され
        //!             ボーダーの外側に向かってグラデーションがかかります
        //!             他の描画関数と比べ負荷が比較的大きいです
        //! @param[in]  pos          左上の座標
        //! @param[in]  size         矩形のサイズ
        //! @param[in]  borderSize   線分の太さ
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DRoundedFrame(const nn::util::Float2& pos, const nn::util::Float2& size, const float round, const int div, const float borderSize) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 線分を描画します
        //! @param[in]  begin    始点
        //! @param[in]  end      終点
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DLine(const nn::util::Float2& begin, const nn::util::Float2& end) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 線分を描画します
        //! @param[in]  num      頂点の数 ( num >= 2)
        //! @param[in]  begin    始点
        //! @param[in]  next     次の点
        //! @param[in]  ...      次の点
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DLine(int num, const nn::util::Float2& begin, const nn::util::Float2 next, ...) NN_NOEXCEPT;


        //----------------------------------------------------------------
        //! @brief (TODO) 線分を描画します
        //! @param[in]  num          頂点の数 ( num >= 2)
        //! @param[in]  pointList    頂点リスト ( array size >= 2 )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DLine(int num, const nn::util::Float2* pointList) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 円を描画します
        //! @param[in]  center   中心点
        //! @param[in]  radius   円の半径
        //! @param[in]  div      円の分割数 ( 3 .. 128 )
        //! @param[in]  angle    円形の傾き ( 0.f .. 359.f )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DCircle(const nn::util::Float2& center, const float radius, const int div, const float angle ) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 円を描画します
        //! @param[in]  center   中心点
        //! @param[in]  radius   円の半径
        //! @param[in]  div      円の分割数 ( 3 .. 128 )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DCircle(const nn::util::Float2& center, const float radius, const int div) NN_NOEXCEPT { Draw2DCircle(center, radius, div, 0.f); }

        //----------------------------------------------------------------
        //! @brief 円を描画します
        //! @param[in]  center   中心点
        //! @param[in]  radius   円の半径
        //! @param[in]  div      円の分割数 ( 3 .. 128 )
        //! @param[in]  angle    円形の傾き ( 0.f .. 359.f )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //! @pre        div >= 3 && div < 512
        //----------------------------------------------------------------
        void Draw2DCircle(const nn::util::Float2& center, const nn::util::Float2& radius, const int div, const float angle) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 円を描画します
        //! @param[in]  center   中心点
        //! @param[in]  radius   円の半径
        //! @param[in]  div      円の分割数 ( 3 .. 128 )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //! @pre        div >= 3 && div < 512
        //----------------------------------------------------------------
        void Draw2DCircle(const nn::util::Float2& center, const nn::util::Float2& radius, const int div) NN_NOEXCEPT { Draw2DCircle(center, radius, div, 0.f); }


        //----------------------------------------------------------------
        //! @brief 円の外周を描画します
        //! @param[in]  center      中心点
        //! @param[in]  radius      円の半径
        //! @param[in]  weight      線の太さ (マイナスの場合内周を描画)
        //! @param[in]  div         円の分割数 ( 3 .. 128 )
        //! @param[in]  angle       円形の傾き ( 0.f .. 359.f )
        //! @param[in]  length      描画する角度 (0.f ..360.f )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //! @pre        div >= 3 && div < 512
        //----------------------------------------------------------------
        void Draw2DCircleFrame(const nn::util::Float2& center, const nn::util::Float2& radius, const float weight, const int div, const float angle, const float length) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 円の外周を描画します
        //! @param[in]  center      中心点
        //! @param[in]  radius      円の半径
        //! @param[in]  weight      線の太さ (マイナスの場合内周を描画)
        //! @param[in]  div         円の分割数 ( 3 .. 128 )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //! @pre        div >= 3 && div < 512
        //----------------------------------------------------------------
        void Draw2DCircleFrame(const nn::util::Float2& center, const nn::util::Float2& radius, const float weight, const int div) NN_NOEXCEPT { Draw2DCircleFrame(center, radius, weight, div, 0.f, 360.f); }

        //----------------------------------------------------------------
        //! @brief 円の外周を描画します
        //! @param[in]  center      中心点
        //! @param[in]  radius      円の半径
        //! @param[in]  weight      線の太さ (マイナスの場合内周を描画)
        //! @param[in]  div         円の分割数 ( 3 .. 128 )
        //! @param[in]  angle       円形の傾き ( 0.f .. 359.f )
        //! @param[in]  length      描画する角度 (0.f ..360.f )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //! @pre        div >= 3 && div < 512
        //----------------------------------------------------------------
        void Draw2DCircleFrame(const nn::util::Float2& center, const float radius, const float weight, const int div, const float angle, const float length) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief 円の外周を描画します
        //! @param[in]  center      中心点
        //! @param[in]  radius      円の半径
        //! @param[in]  weight      線の太さ (マイナスの場合内周を描画)
        //! @param[in]  div         円の分割数 ( 3 .. 128 )
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //! @pre        div >= 3 && div < 512
        //----------------------------------------------------------------
        void Draw2DCircleFrame(const nn::util::Float2& center, const float radius, const float weight, const int div) NN_NOEXCEPT { Draw2DCircleFrame(center, radius, weight, div, 0.f, 360.f); }

        //----------------------------------------------------------------
        //! @brief 立方体を描画します
        //! @param[in]  center      中心点
        //! @param[in]  scale       立方体のスケール
        //! @param[in]  rot         立方体の回転度
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void DrawCube(const nn::util::Vector3fType& center, const nn::util::Vector3fType& scale, const nn::util::Quaternion& rot) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief ぼかしシェーダの有効/無効を切り替えます
        //! @param[in]  enable trueを指定すると有効化
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void EnableBlurhader(bool enable) NN_NOEXCEPT
        {
            m_pGraphicsSystem->GetPrimitiveRenderer().SetUserPixelShader(enable ? m_pGraphicsSystem->GetFrostedTGlassShader() : nullptr);
        }

        //----------------------------------------------------------------
        //! @brief ScreenShot() で保存した画面を描画します
        //! @param[in]  slot     画面を保持しているスロット
        //! @param[in]  pos      左上の座標
        //! @param[in]  size     図形のサイズ
        //! @param[in]  texcoord 描画領域
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DScreenShot(ScreenShotSlot slot, const nn::util::Float2& pos, const nn::util::Float2& size, const nn::util::Float4& texcoord) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief ScreenShot() で保存した画面を描画します
        //! @param[in]  pos      左上の座標
        //! @param[in]  size     図形のサイズ
        //! @param[in]  texcoord 描画領域
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DScreenShot(const nn::util::Float2& pos, const nn::util::Float2& size, const nn::util::Float4& texcoord) NN_NOEXCEPT { Draw2DScreenShot(ScreenShotSlot_Slot_Auto, pos, size, texcoord); }

        //----------------------------------------------------------------
        //! @brief ScreenShot() で保存した画面を描画します
        //! @param[in]  slot     画面を保持しているスロット
        //! @param[in]  pos      左上の座標
        //! @param[in]  size     図形のサイズ
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DScreenShot(ScreenShotSlot slot, const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT { Draw2DScreenShot(slot, pos, size, NN_UTIL_FLOAT_4_INITIALIZER(0.f, 0.f, 1.f, 1.f)); }

        //----------------------------------------------------------------
        //! @brief ScreenShot() で保存した画面を描画します
        //! @param[in]  pos      左上の座標
        //! @param[in]  size     図形のサイズ
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DScreenShot(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT { Draw2DScreenShot(ScreenShotSlot_Slot_Auto, pos, size, NN_UTIL_FLOAT_4_INITIALIZER(0.f, 0.f, 1.f, 1.f)); }

        //----------------------------------------------------------------
        //! @brief 直前のフレームで出画された画面を描画します
        //! @param[in]  pos      左上の座標
        //! @param[in]  size     図形のサイズ
        //! @param[in]  texcoord 描画領域
        //! @pre        BeginDraw() ～ EndDraw() の区間内で呼び出せます
        //----------------------------------------------------------------
        void Draw2DPrevScreen(const nn::util::Float2& pos, const nn::util::Float2& size,
            const nn::util::Float4& texcoord = NN_UTIL_FLOAT_4_INITIALIZER(0.f, 0.f, 1.f, 1.f)) NN_NOEXCEPT;

        void FlushCommandBuffer() NN_NOEXCEPT
        {
            m_pGraphicsSystem->DrawDone();
        }

        // TODO:
        //void InitializeFontTexture(nns::hidfw::gfx::FontSystem* pFontSystem)
        //{
        //    pFontSystem->SetTextureBuffer(m_TextBufferTextureView.ToData());
        //}

    private:
        //----------------------------------------------------------------
        //! @brief GetInstance の内部で呼び出されます
        //----------------------------------------------------------------
        GraphicsDrawer() NN_NOEXCEPT;

        void InitializeBuffer() NN_NOEXCEPT;

        void FinalizeBuffer() NN_NOEXCEPT;

    private:
        nns::hidfw::gfx::GraphicsSystem*   m_pGraphicsSystem;                           //! グラフィックスシステム
        State                       m_State;                                            //! 現在の描画状態
        nn::util::Color4u8          m_ClearColor;                                       //! 画面をクリアする際に利用される色
        nn::util::Color4u8          m_DrawColor[2];                                     //! 描画色です、配列の先頭以降はグラデーション描画の際に利用されます
        GradationDirection          m_GradationDirection;                               //! グラデーションの向き
        float                       m_LineWidth;                                        //! 線分の太さ
        int                         m_FrameRate;                                        //! フレームレート (デフォルト: 60 FPS)
        AntiAliasingType            m_AntiAliasingType;                                 //! アンチエイリアスを利用するか否か

        bool                        m_isEnableScreenScanJob;                            //! trueの場合EndDrawで常時の画面の内容を記憶します
        bool                        m_isEnableScreenShotJob;                            //! trueの場合EndDrawで現在の画面の内容を記憶します
        nn::util::Float2            m_ScreenShotScale;                                  //! 撮影した画面を拡大、縮小する倍率
        ScreenShotSlot              m_ScanScreenSlot;                                   //! キャプチャした画面を保持するスロット

        nn::gfx::Texture            m_ScanBuffer;                                       //! 画面のコピー
        nn::gfx::Texture            m_ScreenShotBuffer[ScreenShotSlot_Slot_Num];        //! スクリーンショット用のバッファ
        nn::gfx::Texture            m_WorkBuffer;                                       //! 画面のコピーの編集用
        nn::gfx::Texture            m_TextBuffer;                                       //! 描画範囲指定のテキスト描画に利用します

        nn::gfx::Sampler            m_Sampler;                                          //! Drawer内部で使用するサンプラ
        nn::gfx::DescriptorSlot     m_SamplerDescriptor;                                //! Drawer内部で使用するサンプラのデスクリプタスロット

        nn::gfx::DescriptorSlot     m_ScanScreenDescriptor;                             //! コピーした画面のデスクリプタ
        nn::gfx::DescriptorSlot     m_ScreenShotDescriptor[ScreenShotSlot_Slot_Num];    //! コピーした画面のデスクリプタ
        nn::gfx::DescriptorSlot     m_TextureDescriptor;
        nn::gfx::DescriptorSlot     m_FontTextureDescriptor;

        nn::gfx::ColorTargetView    m_WorkBufferColorTargetView;                        //! m_ScanBuffer の内容のクリアに使用
        nn::gfx::ColorTargetView    m_ScanBufferColorTargetView;                        //! m_WorkBuffer の内容のクリアに使用
        nn::gfx::ColorTargetView    m_TextBufferColorTargetView;

        nn::gfx::TextureView        m_TextBufferTextureView;
        nn::gfx::TextureView        m_ScanBufferTextureView;                            //! キャプチャした画面をテクスチャとして扱う
        nn::gfx::TextureView        m_ScreenShotTextureView[ScreenShotSlot_Slot_Num];   //! カラーターゲットの参照先をテクスチャとして扱う
        nn::gfx::TextureView        m_FontTextureView;
    };

}}}
