﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * Layout 機能のサンプルです
 */

#include "hidFw/gfx.h"
#include "hidFw/hid.h"
#include "hidFw/layout.h"

#include <nn/hid/hid_ControllerSupport.h>

namespace LayoutSample
{
    void CallControllerSupportApplet(void* pushButton, void* param);

    void StartSixAxisSensor(void* pushButton, void* param);

    void StopSixAxisSensor(void* pushButton, void* param);

    void SwapVibrationViewer(void* pushButton, void* param);

    extern "C" void nnMain()
    {
        //======================================================================
        // 初期化
        //======================================================================

        gGraphics.Initialize();                     //!< グラフィックス機能の初期化を行います
        gInput.Initialize();                        //!< 入力関連の機能の初期化を行います

        //======================================================================
        // レイアウトアイテムの初期設定
        // 全てのレイアウトアイテムを利用します
        //======================================================================

        enum ButtonGroup
        {
            ButtonGroup_VibrationViewer    = 0,
            ButtonGroup_TargetController   = 1,
            ButtonGroup_TargetAxis,
            ButtonGroup_CallFunction,
            ButtonGroup_SeekBar,
            ButtonGroup_Graph,
            ButtonGroup_Num
        };

        nns::hidfw::layout::ButtonGroup    buttonGroup[ButtonGroup::ButtonGroup_Num];        //!< ボタンのグループ

        //======================================================================
        // ボタンを作成します
        //======================================================================

        //-------------------------------------------------------
        // 振動のグラフ表示切替用のボタンを作成します
        //-------------------------------------------------------
        {
            const char* buttonText[] =
            {
                "BarChart",
                "LineChart",
                "Dual"
            };

            // 標準的なボタンの設定で作成します
            for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
            {
                nns::hidfw::layout::Button button;
                button.SetPos(50 + 220 * i, 100);
                button.SetText(buttonText[i]);
                button.ThroughCancel(true);                             // キャンセルを無効化します
                buttonGroup[ButtonGroup_VibrationViewer].Add(new nns::hidfw::layout::Button(button));
            }
        }
        //-------------------------------------------------------
        // 振動や六軸センサのグラフ表示のターゲットとなるコントローラの切り替えボタンを作成します
        //-------------------------------------------------------
        enum  TargetController
        {
            TargetController_Handheld = 0,
            TargetController_No1 = 1,
            TargetController_None,
            TargetController_Num
        };
        {
            const char* buttonText[TargetController_Num] =
            {
                "Handheld",
                "NpadId::No1",
                "None"
            };

            // タッチ操作でフォーカスを移動できないボタンで作成します
            for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
            {
                nns::hidfw::layout::Button button;
                button.SetPos(50 + 220 * i, 200);
                button.SetText(buttonText[i]);
                button.ForciblyFocus(false);                            // ボタンセットの前後関係を無視してフォーカスを変更する機能をOFFにします
                button.ThroughCancel(true);                             // キャンセルを無効化します
                buttonGroup[ButtonGroup_TargetController].Add(new nns::hidfw::layout::Button(button));
            }
        }

        //-------------------------------------------------------
        // グラフ描画で表示される軸を選択するボタンを作成します
        //-------------------------------------------------------
        {
            const char* buttonText[] =
            {
                "X",
                "Y",
                "Z"
            };
            const nn::util::Color4u8 buttonColor[] =
            {
                nn::util::Color4u8(248, 80, 72, 255),
                nn::util::Color4u8(96, 248, 96, 255),
                nn::util::Color4u8(80, 96, 248, 255),
            };

            // 色付きの複数選択が可能なボタンで作成します
            for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
            {
                nns::hidfw::layout::Button button;
                button.SetPos(50 + 220 * i, 300);
                button.SetText(buttonText[i]);
                button.MultiMode(true);                                 // マルチ選択モードをONにします
                button.SetMainColor(buttonColor[i]);                    // ボタンカラーを変更します
                button.SetTextColor(nn::util::Color4u8::White());       // ボタンのテキストカラーを変更します
                buttonGroup[ButtonGroup_TargetAxis].Add(new nns::hidfw::layout::Button(button));
            }
        }

        //-------------------------------------------------------
        // 関数呼び出しを実行するボタンを作成します
        //-------------------------------------------------------
        {
            const char* buttonText[] =
            {
                "CallControllerSupport",
                "StartSixAxisSensor",
                "StopSixAxisSensor"
            };
            nns::hidfw::layout::LayoutFunction pFunc[] =
            {
                CallControllerSupportApplet,
                StartSixAxisSensor,
                StopSixAxisSensor
            };

            for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
            {
                nns::hidfw::layout::Button button;
                button.SetPos(50 + 220 * i, 400);
                button.SetFunc(pFunc[i]);                               // 選択時に呼び出される関数を設定します
                button.SetParam(nullptr);                               // 第二引数に入る変数を設定します
                button.SetText(buttonText[i]);
                button.OnOffMode(false);                                // 押したら、自動で押される前の状態に戻るようにします
                buttonGroup[ButtonGroup_CallFunction].Add(new nns::hidfw::layout::Button(button));
            }
        }

        //-------------------------------------------------------
        // シークバーを作成します
        //-------------------------------------------------------
        {
            for (size_t i = 0; i < 3; ++i)
            {
                nns::hidfw::layout::SeekBar seekBar;
                seekBar.SetPos(50 + 220 * i, 600);
                seekBar.SetWidth(200.f);
                seekBar.SetText("");                                    // 内部で使用されるためテキストは設定しないでください
                seekBar.SetMaxValue((i + 1) * 100);                     // メーターの最大値です。最大値の上限は nns::hidfw::layout::Meter::LimitMaxValue までとなります
                seekBar.SetValue((i + 1) * 50);                         // Value の初期値
                seekBar.EnableAutoShowValue(true);                      // Value の値が更新された際に現在値をポップアップします
                buttonGroup[ButtonGroup_SeekBar].Add(new nns::hidfw::layout::SeekBar(seekBar));
            }
        }

        //======================================================================
        // 折れ線グラフを作成します
        //======================================================================

        enum  eChartList
        {
            eChartList_SixAxisSensorLineChart   = 0,
            eChartList_VibrationLineChart       = 1,
            eChartList_VibrationBarChart,
            eChartList_Num
        };

        size_t chartIndex[eChartList::eChartList_Num];

        buttonGroup[ButtonGroup_Graph].SetOrientation(nns::hidfw::layout::BaseItemSet::eOrientation_Vertical);     // 操作を縦方向操作に切り替えます

        // 6軸センサの値を表示するための折れ線グラフを初期化します
        nns::hidfw::layout::LineChart* pSixAxisSensorLineChart = new nns::hidfw::layout::LineChart();
        pSixAxisSensorLineChart->SetDefault();
        pSixAxisSensorLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Red(), "AccelX", 240, nn::hid::AccelerometerMax, -nn::hid::AccelerometerMax));
        pSixAxisSensorLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Green(), "AccelY", 240, nn::hid::AccelerometerMax, -nn::hid::AccelerometerMax));
        pSixAxisSensorLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Blue(), "AccelZ", 240, nn::hid::AccelerometerMax, -nn::hid::AccelerometerMax));
        pSixAxisSensorLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Magenta(), "VelocityX", 240, nn::hid::AngularVelocityMax, -nn::hid::AngularVelocityMax));
        pSixAxisSensorLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Cyan(), "VelocityY", 240, nn::hid::AngularVelocityMax, -nn::hid::AngularVelocityMax));
        pSixAxisSensorLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Yellow(), "VelocityZ", 240, nn::hid::AngularVelocityMax, -nn::hid::AngularVelocityMax));
        pSixAxisSensorLineChart->SetPos(750, 100);
        pSixAxisSensorLineChart->SetSize(480, 200);

        chartIndex[eChartList_SixAxisSensorLineChart] = buttonGroup[ButtonGroup_Graph].Add(pSixAxisSensorLineChart) - 1;

        // 振動の振幅を表示するための折れ線グラフを初期化します
        nns::hidfw::layout::LineChart* pVibrationLineChart = new nns::hidfw::layout::LineChart();
        pVibrationLineChart->SetDefault();
        pVibrationLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Blue(), "Low", 240, -1.f, 1.f));
        pVibrationLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Red(), "Hi", 240, 1.f, -1.f));
        pVibrationLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Green(), "Low", 240, -1.f, 1.f));
        pVibrationLineChart->AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8::Yellow(), "Hi", 240, 1.f, -1.f));
        pVibrationLineChart->SetFunc(SwapVibrationViewer);                                      // 選択時にグラフの表示状態を切り替える関数を追加します
        pVibrationLineChart->SetParam(&buttonGroup[ButtonGroup_VibrationViewer]);              // 振動表示切替ボタンと連動させます
        pVibrationLineChart->SetPos(750, 350);
        pVibrationLineChart->SetSize(480, 200);
        pVibrationLineChart->LeaveFrameDrawing(true);

        chartIndex[eChartList_VibrationLineChart] = buttonGroup[ButtonGroup_Graph].Add(pVibrationLineChart) - 1;

        //======================================================================
        // 棒グラフを作成します
        //======================================================================

        // 振動値を描画するための棒グラフを初期化します
        nns::hidfw::layout::BarChart* pVibrationBarChart = new nns::hidfw::layout::BarChart();
        pVibrationBarChart->SetDefault();
        pVibrationBarChart->SetViewerMode(nns::hidfw::layout::BarChart::ViewerMode_Music);             // 音楽モード (エフェクトが付きます)
        pVibrationBarChart->SetMaxHorizontalValue(400.f);                                       // 周波数の Max を 400Hz に
        pVibrationBarChart->SetMinHorizontalValue(80.f);                                        // 周波数の Min を 80Hz  に
        pVibrationBarChart->SetMaxVerticalValue(1.f);                                           // Max Gain を 1.0f に
        pVibrationBarChart->SetMinVerticalValue(0.f);                                           // Min Gain を 0.f  に
        pVibrationBarChart->SetDivisionCount(48);                                               // 周波数のMin～Maxを48分割して表示します
        pVibrationBarChart->SetChooseMode(nns::hidfw::layout::BaseItem::ChoiceMode::None::Mask);       // コントローラとタッチ操作を受け付けません
        pVibrationBarChart->SetPos(750, 350);                                                   // 意図的に pVibrationLineChart と同じ座標を設定しています
        pVibrationBarChart->SetSize(480, 200);
        pVibrationBarChart->LeaveFrameDrawing(true);                                            // Visble = false の時も枠線のみ描画します

        chartIndex[eChartList_VibrationBarChart] = buttonGroup[ButtonGroup_Graph].Add(pVibrationBarChart) - 1;

        // それぞれのボタンセットに前後関係を登録します
        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonGroup); ++i)
        {
            buttonGroup[i].SetItemSet(nullptr, &buttonGroup[(i + 1) % NN_ARRAY_SIZE(buttonGroup)], &buttonGroup[(i + NN_ARRAY_SIZE(buttonGroup) - 1) % NN_ARRAY_SIZE(buttonGroup)]);
        }

        // 最初のボタンセットにフォーカスをあわせます
        buttonGroup[ButtonGroup_VibrationViewer].Focus(true);
        // フォーカス変更後は全てのコマンドが無効化されるため、更新を行います
        buttonGroup[ButtonGroup_VibrationViewer].Update();
        // ボタンの初期選択状態を設定します
        buttonGroup[ButtonGroup_VibrationViewer].Select(0, true);
        buttonGroup[ButtonGroup_TargetController].Select(0, true);

        //======================================================================
        // メインループ
        //======================================================================

        nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

        bool                exit        = false;                         // trueになったらループ終了 (サンプルではtrueになることはありません)
        uint64_t            frameCount  = 0;                             // フレームカウント
        nn::util::Color4u8  bgColor     = nn::util::Color4u8::Black();   // 背景色

        do
        {
            // BeginPerfFrame() ～ EndPerfFrame() 間でパフォーマンスを計測します
            gGraphics.GetGraphicsSystem().BeginPerfFrame();

            //======================================================================
            // 入力処理
            //======================================================================
            // コントローラ・タッチスクリーンの入力を更新します
            gInput.Update();

            // 接続中のコントローラのボタンの押下げ状況を確認します
            for (
                std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
                it != gController.GetConnectedControllerList().end();
                ++it
                )
            {
                bool isUsedHandheldController = ((*it)->GetNpadId() == nn::hid::NpadId::Handheld) && buttonGroup[ButtonGroup_TargetController].GetItemSet().at(TargetController_Handheld)->IsSelected();
                bool isUsedWirelessController = ((*it)->GetNpadId() == nn::hid::NpadId::No1) && buttonGroup[ButtonGroup_TargetController].GetItemSet().at(TargetController_No1)->IsSelected();

                if (
                    isUsedHandheldController ||
                    isUsedWirelessController
                    )
                {
                    nn::hid::SixAxisSensorState sixAxisSensor;
                    size_t sixAxisSensorStateCount = (*it)->GetSixAxisSensorState(1, &sixAxisSensor);    // 6軸センサ
                    // 6軸センサの値を追加
                    if (sixAxisSensorStateCount > 0)
                    {
                        nns::hidfw::layout::LineChart* pLineChart = (nns::hidfw::layout::LineChart* )buttonGroup[ButtonGroup_Graph].GetItemSet().at(chartIndex[eChartList_SixAxisSensorLineChart]);
                        pLineChart->AddValue(0, sixAxisSensor.acceleration.x);
                        pLineChart->AddValue(3, sixAxisSensor.angularVelocity.x);
                        pLineChart->AddValue(1, sixAxisSensor.acceleration.y);
                        pLineChart->AddValue(4, sixAxisSensor.angularVelocity.y);
                        pLineChart->AddValue(2, sixAxisSensor.acceleration.z);
                        pLineChart->AddValue(5, sixAxisSensor.angularVelocity.z);
                    }
                    // 現在の振動値を追加
                    {
                        nns::hidfw::layout::LineChart* pLineChart = (nns::hidfw::layout::LineChart*)buttonGroup[ButtonGroup_Graph].GetItemSet().at(chartIndex[eChartList_VibrationLineChart]);
                        nns::hidfw::layout::BarChart* pBarChart  = (nns::hidfw::layout::BarChart*)buttonGroup[ButtonGroup_Graph].GetItemSet().at(chartIndex[eChartList_VibrationBarChart]);
                        nn::hid::VibrationValue leftVib = nn::hid::VibrationValue::Make();
                        nn::hid::VibrationValue rightVib = nn::hid::VibrationValue::Make();
                        (*it)->GetVibration().GetCurVibrationValue(&leftVib, &rightVib);
                        pLineChart->AddValue(0, rightVib.amplitudeLow);
                        pLineChart->AddValue(1, rightVib.amplitudeHigh);
                        pBarChart->AddValue(rightVib.frequencyLow, rightVib.amplitudeLow);
                        pBarChart->AddValue(rightVib.frequencyHigh, rightVib.amplitudeHigh);
                    }
                    if ((*it)->IsHold(nn::hid::NpadButton::B::Mask))
                    {
                        (*it)->GetVibration().Send(nn::hid::VibrationValue::Make(
                            0.05f + nn::util::SinEst(nn::util::DegreeToRadian(frameCount % 360)) * 0.05f, 160.f + nn::util::CosEst(nn::util::DegreeToRadian(frameCount % 360)) * 40.f,
                            0.05f + nn::util::SinEst(nn::util::DegreeToRadian(frameCount % 360)) * 0.05f, 320.f - nn::util::CosEst(nn::util::DegreeToRadian(frameCount % 360)) * 40.f));
                    }

                    {
                        nns::hidfw::layout::LineChart* pLineChart = (nns::hidfw::layout::LineChart*)buttonGroup[ButtonGroup_Graph].GetItemSet().at(chartIndex[eChartList_VibrationLineChart]);
                        nn::hid::VibrationValue vib[2];
                        (*it)->GetVibration().GetCurSendVibrationValue(vib, 2);
                        pLineChart->AddValue(2, vib[1].amplitudeLow);
                        pLineChart->AddValue(3, vib[1].amplitudeHigh);
                    }
                }
                if ((*it)->IsTrigger(nn::hid::NpadButton::X::Mask))
                {
                    // いずれかのコントローラでXボタンが押下げられたら背景色を変更します
                    bgColor = (bgColor == nn::util::Color4u8::Black()) ? nn::util::Color4u8::Gray() : nn::util::Color4u8::Black();
                }
                if ((*it)->IsTrigger(nn::hid::NpadButton::Plus::Mask) || (*it)->IsTrigger(nn::hid::NpadButton::Minus::Mask))
                {
                    nns::hidfw::layout::Dialog dialog;
                    nns::hidfw::layout::Dialog::DialogInfo info;
                    info.Buttons.Set<nns::hidfw::layout::Dialog::DialogButton::OK>();
                    info.Buttons.Set<nns::hidfw::layout::Dialog::DialogButton::Cancel>();
                    dialog.ShowDialog(info);
                }
                else
                {
                    // ボタン押下げの処理 (複数同時押下げ や 複数のコントローラでの押下げ などは考慮していません)
                    if ((*it)->IsTrigger(nn::hid::NpadButton::Y::Mask))
                    {
                        // いずれかのコントローラでYボタンが押下げられたら現在のCPU、GPU使用率を表示します
                        gGraphics.GetGraphicsSystem().SetEnablePerfMeter(!gGraphics.GetGraphicsSystem().IsEnablePerfMeter());
                    }
                }
            }

            //======================================================================
            // ツールの更新処理
            //======================================================================
            // 振動
            {
                nns::hidfw::layout::LineChart* pLineChart = (nns::hidfw::layout::LineChart*)buttonGroup[ButtonGroup_Graph].GetItemSet().at(chartIndex[eChartList_VibrationLineChart]);
                nns::hidfw::layout::BarChart* pBarChart = (nns::hidfw::layout::BarChart*)buttonGroup[ButtonGroup_Graph].GetItemSet().at(chartIndex[eChartList_VibrationBarChart]);

                pBarChart->Visible(buttonGroup[ButtonGroup_VibrationViewer].GetItemSet().at(0)->IsSelected() || buttonGroup[ButtonGroup_VibrationViewer].GetItemSet().at(2)->IsSelected());
                pLineChart->Visible(buttonGroup[ButtonGroup_VibrationViewer].GetItemSet().at(1)->IsSelected() || buttonGroup[ButtonGroup_VibrationViewer].GetItemSet().at(2)->IsSelected());

                if (!pLineChart->IsVisible())
                {
                    nn::util::Color4u8 color = pBarChart->GetMainColor();
                    color.SetA(255);
                    pBarChart->SetMainColor(color);
                    color = pBarChart->GetBorderColor();
                    color.SetA(255);
                    pBarChart->SetBorderColor(color);
                }
                else
                {
                    nn::util::Color4u8 color = pBarChart->GetMainColor();
                    color.SetA(0);
                    pBarChart->SetMainColor(color);
                    color = pBarChart->GetBorderColor();
                    color.SetA(0);
                    pBarChart->SetBorderColor(color);
                }

            }
            // 6軸
            {
                nns::hidfw::layout::LineChart* pLineChart = (nns::hidfw::layout::LineChart*)buttonGroup[ButtonGroup_Graph].GetItemSet().at(chartIndex[eChartList_SixAxisSensorLineChart]);

                pLineChart->VisibleElement(0, buttonGroup[2].GetItemSet().at(0)->IsSelected());
                pLineChart->VisibleElement(3, buttonGroup[2].GetItemSet().at(0)->IsSelected());
                pLineChart->VisibleElement(1, buttonGroup[2].GetItemSet().at(1)->IsSelected());
                pLineChart->VisibleElement(4, buttonGroup[2].GetItemSet().at(1)->IsSelected());
                pLineChart->VisibleElement(2, buttonGroup[2].GetItemSet().at(2)->IsSelected());
                pLineChart->VisibleElement(5, buttonGroup[2].GetItemSet().at(2)->IsSelected());
            }

            //======================================================================
            // アイテムセットの更新処理
            //======================================================================

            for (size_t i = 0; i < NN_ARRAY_SIZE(buttonGroup); ++i)
            {
                buttonGroup[i].Update();                     //!< ボタンセットの更新
            }

            //======================================================================
            // テキスト出力
            //======================================================================

            gTextWriter.SetScale(1.f, 1.f);
            gTextWriter.SetTextColor((bgColor == nn::util::Color4u8::Black()) ? nn::util::Color4u8::Gray() : nn::util::Color4u8::Black());

            gTextWriter.SetCursor(32, 68);
            gTextWriter.Print("VibrationViewer (Basic Button)");
            gTextWriter.SetCursor(32, 168);
            gTextWriter.Print("TargetController (Disable Touch Focus)");
            gTextWriter.SetCursor(32, 268);
            gTextWriter.Print("TargetAxis (Enable Multi Select)");
            gTextWriter.SetCursor(32, 368);
            gTextWriter.Print("CallFunction (Function Button)");
            gTextWriter.SetCursor(32, 568);
            gTextWriter.Print("MeterSet1 (Basic Meter)");
            gTextWriter.SetCursorX(752);
            gTextWriter.Print("(B) Play Vibration");
            //======================================================================
            // 描画処理
            //======================================================================

            // 背景色の切り替え
            gDrawer.SetClearColor(bgColor);

            gDrawer.BeginDraw();            // 描画関数の受付開始
            {
                for (size_t i = 0; i < NN_ARRAY_SIZE(buttonGroup); ++i)
                {
                    buttonGroup[i].Draw();
                }
            }
            gDrawer.EndDraw();              // 描画終了
            ++frameCount;

            // BeginPerfFrame() ～ EndPerfFrame() 間でパフォーマンスを計測します
            gGraphics.GetGraphicsSystem().EndPerfFrame();

        } while (!exit);

        //======================================================================
        // 終了処理
        //======================================================================

        gInput.Finalize();
        gGraphics.Finalize();

    } // NOLINT(impl/function_size)

    void CallControllerSupportApplet(void* pushButton, void* param)
    {
        nn::Result result;
        nn::hid::ControllerSupportArg arg;
        nn::hid::ControllerSupportResultInfo info;

        arg.SetDefault();
        arg.playerCountMin = arg.playerCountMax = 4;
        arg.enableIdentificationColor = true;
        arg.enableExplainText = true;

        arg.identificationColor[0] = nn::util::Color4u8::Red();
        arg.identificationColor[1] = nn::util::Color4u8::Green();
        arg.identificationColor[2] = nn::util::Color4u8::Blue();
        arg.identificationColor[3] = nn::util::Color4u8::Yellow();

        nn::hid::SetExplainText(&arg, "Controller (1)", nn::hid::NpadId::No1);
        nn::hid::SetExplainText(&arg, "Controller (2)", nn::hid::NpadId::No2);
        nn::hid::SetExplainText(&arg, "Controller (3)", nn::hid::NpadId::No3);
        nn::hid::SetExplainText(&arg, "Controller (4)", nn::hid::NpadId::No4);

        /*
        * コントローラーサポートアプレットでは、縦持ちと下記のスタイルの混在をサポートしていません。
        * 下記のスタイルが一つでも有効な場合は
        * コントローラサポートアプレットの呼び出し前にコントローラの持ち方を横持ちに変更する必要があります。
        * （縦持ちにする場合は下記のスタイルを無効にしてください。）
        *
        * nn::hid::NpadStyleHandheld
        * nn::hid::NpadStyleFullKey
        * nn::hid::NpadStyleJoyDual
        */
        if (nn::hid::GetNpadJoyHoldType() == nn::hid::NpadJoyHoldType_Vertical)
        {
            /*
             * アプリはコントローラーサポートアプレットの呼び出し前に
             * ホールドタイプの変更を行う必要が無いような実装にするべきです
             * 本サンプルではホールドタイプの維持よりもコントローラサポートアプレットの呼び出しを優先するため
             * コントローラーサポートアプレットの呼び出し前に適切なホールドタイプへの切り替えを行っています
             */
            nn::hid::NpadStyleSet style = nn::hid::GetSupportedNpadStyleSet();
            if ((style &
                (
                    nn::hid::NpadStyleHandheld::Mask |
                    nn::hid::NpadStyleFullKey::Mask |
                    nn::hid::NpadStyleJoyDual::Mask
                )).IsAnyOn())
            {
                nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Horizontal);
            }
        }

        // コントローラサポートアプレットの呼び出しを実行します
        result = nn::hid::ShowControllerSupport(&info, arg);
    }

    void StartSixAxisSensor(void* pushButton, void* param)
    {
        for (
            std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
            it != gController.GetConnectedControllerList().end();
            ++it
            )
        {
            (*it)->StartSixAxisSensor();
        }
    }

    void StopSixAxisSensor(void* pushButton, void* param)
    {
        for (
            std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
            it != gController.GetConnectedControllerList().end();
            ++it
            )
        {
            (*it)->StopSixAxisSensor();
        }
    }

    void SwapVibrationViewer(void* pushButton, void* param)
    {
        nns::hidfw::layout::ButtonGroup* buttonGroup = (nns::hidfw::layout::ButtonGroup* )param;
        if (buttonGroup != nullptr)
        {
            size_t itemIndex = 0;
            for (size_t i = 0; i < buttonGroup->GetListSize(); ++i)
            {
                if (buttonGroup->GetItemSet().at(i)->IsSelected())
                {
                    itemIndex = (i + 1) % buttonGroup->GetListSize();
                }
            }
            buttonGroup->Select(itemIndex);
        }
    }
}
