﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/time.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs.h>
#include <nn/htc.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nv_GemCoreDump.h>

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
#define GPU_CORE_DUMPER_TRACE( channel, ... ) NN_SDK_LOG( "[" ); NN_SDK_LOG( channel ); NN_SDK_LOG( "] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#define GPU_CORE_DUMPER_LOG(...) NN_LOG( "[GpuCoreDumper] - " ); NN_LOG( __VA_ARGS__ ); NN_LOG( "\n" )
#else
#define GPU_CORE_DUMPER_TRACE( channel, ... )
#define GPU_CORE_DUMPER_LOG(...)
#endif

#define GPU_CORE_DUMPER_EXTENSION ".nxgcd"

// ホスト PC へのダンプを有効化します。
#define GPU_CORE_DUMPER_ENABLE_HOSTFS_DUMP
// SD カードへのダンプを有効化します。
#define GPU_CORE_DUMPER_ENABLE_SDCARD_DUMP
// 実際のコアダンプデータの代わりにダミーデータを書き出します。
//#define GPU_CORE_DUMPER_DUMP_DUMMY_DATA

namespace
{
    enum
    {
        OUTPUT_FILE_NAME_SIZE = 1024,
        PROCESS_ARGS_SIZE = 1024,
        FILE_PATH_BUFFER_SIZE = 1024,
        HEAP_SIZE = 4 * 1024,
    };

    char sOutputFileName[OUTPUT_FILE_NAME_SIZE];
    char sArgs[PROCESS_ARGS_SIZE];

    NN_ALIGNAS(4096) char g_HeapBuffer[HEAP_SIZE];
    nn::lmem::HeapHandle g_HeapHandle;
    const char SdRootFsName[] = "sdmc";
    const char PcRootFsName[] = "host";
    bool s_MountHost = false;
    bool s_MountSdmc = false;

    GcdBlock s_GcdBlock;

    void* Allocate(size_t size)
    {
        GPU_CORE_DUMPER_TRACE("Allocate", "Allocating %lld", size);

        void* p = nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
        if (p == nullptr)
        {
            GPU_CORE_DUMPER_LOG("Alloc() returned NULL when trying to allocate %d bytes (%d bytes available)\n", size, nn::lmem::GetExpHeapTotalFreeSize(g_HeapHandle));
        }
        return p;
    }

    void Deallocate(void* Ptr, size_t size)
    {
        GPU_CORE_DUMPER_TRACE("Deallocate", "Deallocating %lld", size);
        (void)size;
        nn::lmem::FreeToExpHeap(g_HeapHandle, Ptr);
    }

    void ExtractDirectoryName(char* buffer, size_t bufferSize, const char* path)
    {
        std::strncpy(buffer, path, bufferSize);
        *(buffer + bufferSize - 1) = '\0';

        char* ptr = buffer + bufferSize - 1;
        while (ptr > buffer && *ptr != '\\' && *ptr != '/')
        {
            ptr--;
        }
        *ptr = '\0';
    }

    void ExtractFileName(char* buffer, size_t bufferSize, const char* path)
    {
        const char *fileNamePtr = path;
        while (*path != '\0')
        {
            if (*path == '\\' || *path == '/')
            {
                fileNamePtr = path + 1;
            }
            path++;
        }
        std::strncpy(buffer, fileNamePtr, bufferSize);
        *(buffer + bufferSize - 1) = '\0';
    }

    bool IsHostAvailable()
    {
        nn::htc::Initialize();
        const char EnvVarName[] = "NINTENDO_SDK_ROOT";
        size_t length;
        nn::Result result = nn::htc::GetEnvironmentVariableLength(&length, EnvVarName);
        nn::htc::Finalize();

        return !(nn::htc::ResultConnectionFailure().Includes(result));
    }

    void EnsureDirectory(char *path)
    {
        // If path points to network drive, do nothing and return
        // This is not because of technical issue but troublesomeness
        if (*path == '\\' || *path == '/')
        {
            return;
        }

        nn::fs::MountHostRoot();
        char *ptr = path;
        while (*ptr != '\0')
        {
            while (*ptr != '\0' && *ptr != '\\' && *ptr != '/')
            {
                ptr++;
            }
            char org = *ptr;
            *ptr = '\0';
            if (!(ptr > path && *(ptr - 1) == ':'))
            {
                nn::fs::CreateDirectory(path);
            }
            *ptr = org;
            ptr++;
        }
        nn::fs::UnmountHostRoot();
    }

    bool StringsEqual(const char* pStr1, const char* pStr2)
    {
        int Length = strlen(pStr1);
        if (strlen(pStr2) != Length)
        {
            return false;
        }

        for (int Index = 0; Index < Length; Index += 1)
        {
            if (tolower(pStr1[Index]) != (unsigned int)tolower(pStr2[Index]))
            {
                return false;
            }
        }

        return true;
    }

    bool FormatOutputFileName(char* OutputFileName)
    {
        bool Ret = false;
        if (OutputFileName != NULL && strlen(OutputFileName) > 0)
        {
            //Does it have a valid extension?
            char* pExtension = strrchr(OutputFileName, '.');
            if (pExtension != NULL && StringsEqual(pExtension, GPU_CORE_DUMPER_EXTENSION))
            {
                *pExtension = '\0'; //Remove extension once
            }
            Ret = true;
        }

        return Ret;
    }

    size_t ParseEnvironmentVariable(char *buffer, size_t bufferSize, char *src)
    {
        nn::htc::Initialize();
        int size = 0;
        while (*src != '\0' && bufferSize > size + 1)
        {
            if (*src == '%')
            {
                src++;
                const char *envName = src;
                while (*src != '\0' && *src != '%')
                    src++;
                *src = '\0';
                src++;

                size_t envSize;
                nn::Result result = nn::htc::GetEnvironmentVariable(&envSize, buffer + size, bufferSize - size, envName);
                if (result.IsFailure() || envSize == 0)
                {
                    NN_LOG("[Dump Error] Failed to read environment variable %s.\n", envName);
                }
                else
                {
                    size += envSize - 1;
                }
            }
            else
            {
                *(buffer + size) = *src;
                src++;
                size++;
            }
        }
        buffer[size] = '\0';
        size++;
        nn::htc::Finalize();
        return size;
    }

    bool CreateDefaultFileName(char *buffer, size_t bufferSize)
    {
        const int SettingsValueSize = 256;
        char settingsValue[SettingsValueSize];
        nn::settings::fwdbg::GetSettingsItemValue(settingsValue, SettingsValueSize, "snap_shot_dump", "output_dir");
        size_t size = ParseEnvironmentVariable(buffer, bufferSize, settingsValue);

        const int FileNameLength = 1 + 14 + 1 + 14; // /SN_YYYYMMDDhhmmss
        NN_SDK_ASSERT(size + FileNameLength <= bufferSize, "Too long path\n");
        if (size <= 0 || size + FileNameLength > bufferSize)
        {
            return false;
        }

        nn::time::PosixTime timeStamp;
        nn::time::StandardUserSystemClock::GetCurrentTime(&timeStamp);
        nn::time::CalendarTime time;
        nn::time::ToCalendarTime(&time, nullptr, timeStamp);

        nn::settings::factory::SerialNumber serialNumber;
        serialNumber.string[0] = '\0';

        const nn::Result result = nn::settings::factory::GetSerialNumber(&serialNumber);
        if (result.IsFailure())
        {
            NN_LOG("[GpuCoreDumper] Failed to get Serial Number of the device.\n");
        }

        nn::util::TSNPrintf(buffer + size - 1, bufferSize - size + 1, "\\%s_%04d%02d%02d%02d%02d%02d",
            serialNumber.string,
            time.year, time.month, time.day, time.hour, time.minute, time.second);
        return true;
    }
}

bool GetCommandLineParams(char* pOutputFileName, size_t OutputFileNameSize, char* pProcessArgs)
{
    int NumArgs = nn::os::GetHostArgc();
    char** pArgs = nn::os::GetHostArgv();
    for (int ArgIndex = 0; ArgIndex < NumArgs; ArgIndex += 1)
    {
        GPU_CORE_DUMPER_TRACE("GpuCoreDumper", "GetCommandLineParams:  arg %d:  %s", ArgIndex, pArgs[ArgIndex]);
    }

    if (NumArgs < 1)
    {
        return false;
    }

    //============================================================
    // Get our required inputs.
    //============================================================
    int ArgIndex = 1;

    if (NumArgs > 1 && pArgs[1][0] != '-')
    {
        strcpy(pOutputFileName, pArgs[1]);
        ArgIndex++;
    }
    else
    {
        if (!CreateDefaultFileName(pOutputFileName, OutputFileNameSize))
        {
            return false;
        }
    }

    for (; ArgIndex < NumArgs; ArgIndex++)
    {
        // currently, there is no enable options with starting "-".
    }

    return FormatOutputFileName(pOutputFileName);
}

nn::Result CreateSuffixedFile(char *filePathBuffer, size_t bufferSize, const char* driveName, const char* fileName)
{
    nn::Result result;
    const int SuffixEnd = 100;

    for (int i = 0; i < SuffixEnd; i++)
    {
        nn::util::SNPrintf(filePathBuffer, bufferSize, "%s:/%s_%02d%s", driveName, fileName, i, GPU_CORE_DUMPER_EXTENSION);

        result = nn::fs::CreateFile(filePathBuffer, 0);
        if (result.IsSuccess())
        {
            return nn::ResultSuccess();
        }
    }
    return result;
}

nn::Result FsInit(char *filePathBuffer, size_t bufferSize)
{
    g_HeapHandle = nn::lmem::CreateExpHeap(&g_HeapBuffer, sizeof(g_HeapBuffer), nn::lmem::CreationOption_NoOption);

    nn::fs::SetAllocator(Allocate, Deallocate);

    nn::Result result = nn::ResultSuccess();

#if defined(GPU_CORE_DUMPER_ENABLE_HOSTFS_DUMP)
    if (IsHostAvailable())
    {
        char fileName[bufferSize];
        ExtractFileName(fileName, bufferSize, filePathBuffer);
        char dirBuffer[bufferSize];
        ExtractDirectoryName(dirBuffer, bufferSize, filePathBuffer);
        EnsureDirectory(dirBuffer);

        result = nn::fs::MountHost(PcRootFsName, dirBuffer);
        if (result.IsSuccess())
        {
            nn::util::SNPrintf(filePathBuffer, bufferSize, "%s:/%s%s", PcRootFsName, fileName, GPU_CORE_DUMPER_EXTENSION);
            result = nn::fs::CreateFile(filePathBuffer, 0);

            if (nn::fs::ResultPathAlreadyExists::Includes(result))
            {
                result = CreateSuffixedFile(filePathBuffer, bufferSize, PcRootFsName, fileName);
            }

            if (result.IsSuccess())
            {
                NN_LOG("[GpuCoreDumper] Start dumping to %s...\n", dirBuffer);
                s_MountHost = true;
                return nn::ResultSuccess();
            }
            nn::fs::Unmount(PcRootFsName);
        }

        NN_LOG("[GpuCoreDumper] Failed to dump to Host PC (%s).\nTrying SD card.\n", filePathBuffer);
    }
#endif

#if defined(GPU_CORE_DUMPER_ENABLE_SDCARD_DUMP)
    const char SdRootDirName[] = "NXDMP";
    result = nn::fs::MountSdCardForDebug(SdRootFsName);
    if (result.IsSuccess())
    {
        {
            char rootDir[bufferSize];
            nn::util::SNPrintf(rootDir, bufferSize, "%s:/%s", SdRootFsName, SdRootDirName);
            result = nn::fs::CreateDirectory(rootDir);
            if (result.IsFailure() && !nn::fs::ResultPathAlreadyExists().Includes(result))
            {
                nn::fs::Unmount(SdRootFsName);
                return result;
            }
        }
        char fileName[bufferSize];
        ExtractFileName(fileName, bufferSize, filePathBuffer);
        char path[bufferSize];
        nn::util::SNPrintf(path, bufferSize, "%s/%s", SdRootDirName, fileName);

        nn::util::SNPrintf(filePathBuffer, bufferSize, "%s:/%s%s", SdRootFsName, path, GPU_CORE_DUMPER_EXTENSION);
        result = nn::fs::CreateFile(filePathBuffer, 0);

        if (nn::fs::ResultPathAlreadyExists::Includes(result))
        {
            result = CreateSuffixedFile(filePathBuffer, bufferSize, SdRootFsName, path);
        }

        if (result.IsSuccess())
        {
            NN_LOG("[GpuCoreDumper] Start dumping to SD card...\n");
            s_MountSdmc = true;
        }
    }
#endif

    return result;
}

void FsClose()
{
    if (s_MountHost)
    {
        nn::fs::Unmount(PcRootFsName);
    }
    if (s_MountSdmc)
    {
        nn::fs::Unmount(SdRootFsName);
    }
}

void ReportUsage()
{
    NN_LOG("usage: GpuCoreDumper [<output file name>]\n");
}

extern "C" void nninitStartup()
{
}

extern "C" void nnMain()
{
    nn::fs::InitializeWithMultiSessionForTargetTool();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::time::Initialize()
    );

    if (GetCommandLineParams(sOutputFileName, OUTPUT_FILE_NAME_SIZE, sArgs) == false)
    {
        ReportUsage();
        return;
    }

    GPU_CORE_DUMPER_LOG("Dump file path = %s", sOutputFileName);

    // コアダンプクライアントを初期化します。
    nv::gem::Result result;
    nv::gemcoredump::Client client;
    {
        result = client.Initialize();
        NN_ABORT_UNLESS(result == nv::gem::Result_Success, "Failed to initialize gpu core dump client.\n");
    }

    // ファイルシステムを初期化します。
    // ダンプデータの出力先とするファイルを生成し、パスを獲得します。
    {
        nn::Result result = FsInit(sOutputFileName, OUTPUT_FILE_NAME_SIZE);
        if (result.IsFailure())
        {
            NN_LOG("[GpuCoreDumper] Failed to initialize FS (0x%08X). Is SD card inserted?\n", result.GetInnerValueForDebug());
            return;
        }
    }

    GPU_CORE_DUMPER_LOG("Dump file path (final) = %s", sOutputFileName);

    // コアダンプデータを書き込みます。
    {
        nn::fs::FileHandle fileHandle;

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::OpenFile(&fileHandle, sOutputFileName, nn::fs::OpenMode::OpenMode_Write | nn::fs::OpenMode::OpenMode_AllowAppend)
        );

#if defined(GPU_CORE_DUMPER_DUMP_DUMMY_DATA)
        const char dummyData[] = "This is core dump dummy data";

        nn::fs::WriteOption writeOption = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::WriteFile(fileHandle, 0, dummyData, sizeof(dummyData), writeOption)
            );

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::FlushFile(fileHandle)
        );
        nn::fs::CloseFile(fileHandle);
#else
        nn::fs::WriteOption writeOption = {};

        nn::applet::AppletResourceUserId aruid;
        result = client.GetAruid(&aruid.lower);
        NN_ABORT_UNLESS(result == nv::gem::Result_Success, "Failed to get applet user id.\n");
        NN_UNUSED(aruid);

        int64_t offset = 0;
        while (true)
        {
            result = client.ReadNextBlock(&s_GcdBlock);
            NN_ABORT_UNLESS(result == nv::gem::Result_Success, "Failed to get gpu core dump block data.\n");

            if (s_GcdBlock.size == 0)
            {
                // size == 0 は、すべてのデータの読み出し完了を意味します。
                break;
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::fs::WriteFile(fileHandle, offset, s_GcdBlock.data, s_GcdBlock.size, writeOption)
            );

            GPU_CORE_DUMPER_LOG("Write %d bytes (offset = %lld).", s_GcdBlock.size, offset);

            offset += s_GcdBlock.size;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::FlushFile(fileHandle)
        );
        nn::fs::CloseFile(fileHandle);
#endif
    }

    // ファイルシステムを終了します。
    {
        FsClose();
    }

    NN_LOG("[GpuCoreDumper] Dump completed\n");
}
