﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/i2c/i2c.h>
#include <nn/oe/oe_PowerStateControlApi.private.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include "FuelGaugeDumper_Glv.h"
#include "FuelGaugeDumper_SdCardLog.h"

namespace {

const int MajorVersion = 0;
const int MinorVersion = 1;

const uint64_t DumpIntervalMilliSeconds = 1000;

const uint8_t AddressRegionForDump[][2] =
{
    { 0x00, 0x0B },
    { 0x0D, 0x13 },
    { 0x16, 0x1F },
    { 0x21, 0x25 },
    { 0x27, 0x2F },
    { 0x32, 0x32 },
    { 0x36, 0x3A },
    { 0x3D, 0x3F },
    { 0x42, 0x42 },
    { 0x45, 0x46 },
    { 0x4D, 0x4D },
    { 0x80, 0xAF },
    { 0xFB, 0xFB },
    { 0xFF, 0xFF },
};

void Terminate() NN_NOEXCEPT;

class TerminationButton : public glv::Button
{
    bool onEvent(glv::Event::t e, glv::GLV& g) NN_NOEXCEPT
    {
        switch (e)
        {
        case glv::Event::WindowCreate:
            Terminate();
            break;
        case glv::Event::WindowDestroy:
            Terminate();
            break;
        case glv::Event::MouseDown:
            Terminate();
            break;
        default:
            break;
        }

        return true;
    }
};

glv::Label g_TitleLabel;
glv::Label g_VersionLabel;
glv::Label g_RegistersAddressLabel[0x100];
glv::Label g_RegistersLabel[0x100];
TerminationButton g_TerminationButton;
glv::Label g_TerminationLabel;

nn::Result UpdateFuelGaugeRegisters(nn::fs::FileHandle* pFileHandle, nn::i2c::I2cSession* pI2c) NN_NOEXCEPT
{

    char writeBuffer[0x2000] = "";

    char* pTail = &writeBuffer[0];

    {
        nn::os::Tick tick       = nn::os::GetSystemTick();
        int32_t hours           = static_cast<uint32_t>(nn::os::ConvertToTimeSpan(tick).GetHours());
        int32_t minutes         = static_cast<uint32_t>(nn::os::ConvertToTimeSpan(tick).GetMinutes() % 60);
        int32_t seconds         = static_cast<uint32_t>(nn::os::ConvertToTimeSpan(tick).GetSeconds() % 60);
        int32_t milliSeconds    = static_cast<uint32_t>(nn::os::ConvertToTimeSpan(tick).GetMilliSeconds() % 1000);

        auto size = nn::util::SNPrintf(pTail, 20, "%d:%02d:%02d.%03d, ", hours, minutes, seconds, milliSeconds);
        NN_ABORT_UNLESS(size > 0);
        pTail += size;
    }

    for ( int region = 0; region < (sizeof(AddressRegionForDump) / sizeof(AddressRegionForDump[0])); region++ )
    {
        for ( uint8_t address = AddressRegionForDump[region][0]; address <= AddressRegionForDump[region][1]; address++ )
        {
            uint16_t value = 0;

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&value, *pI2c, &address));

            // Label への反映
            const size_t BufferSize = 0x20;
            char buffer[BufferSize];
            nn::util::SNPrintf(buffer, BufferSize, "%04x", value);
            g_RegistersLabel[address].setValue(std::string(buffer));

            // Log 出力フォーマットバッファへの書込み
            auto size = nn::util::SNPrintf(pTail, 14, "0x%02x:0x%04x", address, value);
            NN_ABORT_UNLESS(size > 0);
            pTail += size;

            // Log 出力フォーマットバッファ改行処理のスキップ
            if ( address == 0xFF )
            {
                break;
            }

            size = nn::util::SNPrintf(pTail, 3, ", ");
            NN_ABORT_UNLESS(size > 0);
            pTail += size;
        }
    }

    auto size = nn::util::SNPrintf(pTail, 2, "\n");
    NN_ABORT_UNLESS(size > 0);
    pTail += size;

    int length = pTail - &writeBuffer[0];

    return AppendLog(pFileHandle, writeBuffer, length);
}

bool g_Running = true;

void Terminate() NN_NOEXCEPT
{
    if ( g_Running )
    {
        NN_LOG("Terminate.\n");
        nn::oe::RequestToShutdown();
//        g_Running = false;
    }
}

void AddToolInformationLabels(const char* toolName, int majorVersion, int minorVersion) NN_NOEXCEPT
{
    const size_t BufferSize = 0x400;
    char buffer[BufferSize];

    nn::util::SNPrintf(buffer, BufferSize, "%s", toolName);
    g_TitleLabel.setValue(std::string(buffer));
    g_TitleLabel.pos(20, 20);
    g_TitleLabel.size(28.0f);
    AddItemToView(g_TitleLabel);

    nn::util::SNPrintf(buffer, BufferSize, "Version %02d.%02d", majorVersion, minorVersion);
    g_VersionLabel.setValue(std::string(buffer));
    g_VersionLabel.pos(1000, 20);
    g_VersionLabel.size(28.0f);
    AddItemToView(g_VersionLabel);
}

} // namespace

extern "C" void nnMain()
{
    InitializeGlv();

    // TODO: Result 分類の表示
    if ( MountSdCard().IsFailure() )
    {
        return;
    }

    if ( CreateLogFile().IsFailure() )
    {
        return;
    }

    nn::fs::FileHandle fileHandle;
    if ( OpenLogFile(&fileHandle).IsFailure() )
    {
        return;
    }

    nn::i2c::Initialize();
    nn::i2c::I2cSession i2c;
    nn::i2c::OpenSession(&i2c, nn::i2c::I2cDevice_Max17050);

    AddToolInformationLabels("FuelGaugeDumper", MajorVersion, MinorVersion);

    // TODO: 変数整理、定数整理
    int offsetX = 0;
    int offsetY = 0;
    for ( glv::Label& label : g_RegistersLabel )
    {
        label.size(18);
        label.pos(120 * offsetX + 20 + 40, 18 * offsetY + 60);

        const size_t BufferSize = 0x20;
        char buffer[BufferSize];
        nn::util::SNPrintf(buffer, BufferSize, "----");
        label.setValue(std::string(buffer));

        AddItemToView(label);

        offsetY++;
        offsetX += offsetY / 0x20;
        offsetY %= 0x20;
    }

    offsetX = 0;
    offsetY = 0;
    for ( glv::Label& label : g_RegistersAddressLabel )
    {
        label.size(18);
        label.pos(120 * offsetX + 20, 18 * offsetY + 60);

        const size_t BufferSize = 0x20;
        char buffer[BufferSize];
        nn::util::SNPrintf(buffer, BufferSize, "%02x:", offsetY + offsetX * 0x20);
        label.setValue(std::string(buffer));

        AddItemToView(label);

        offsetY++;
        offsetX += offsetY / 0x20;
        offsetY %= 0x20;
    }

    g_TerminationButton.pos(980, 60);
    g_TerminationButton.extent(300, 300);
    AddItemToView(g_TerminationButton);

    {
        g_TerminationLabel.size(36);
        g_TerminationLabel.pos(980 + 50, 60 + 132);
        const size_t BufferSize = 0x20;
        char buffer[BufferSize];
        nn::util::SNPrintf(buffer, BufferSize, "TERMINATE");
        g_TerminationLabel.setValue(std::string(buffer));
        AddItemToView(g_TerminationLabel);
    }

    while ( g_Running )
    {
        (void)UpdateFuelGaugeRegisters(&fileHandle, &i2c);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(DumpIntervalMilliSeconds));
    }

    CloseLogFile(&fileHandle);

    UnmountSdCard();

    NN_LOG("Finalize.\n");

    FinalizeGlv();
}
