﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <map>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/ae/ae.h>
#include <nn/gfx.h>
#include <nn/nn_Log.h>
#include <stdint.h>
#include <cstring>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet.h>
#include <nn/ec/detail/ec_ShowShopPageTypes.h>
#include <nn/util/util_StringUtil.h>

#include <cstdlib>

// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Assert.h>
#include <nn/vi.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nv/nv_MemoryManagement.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/web/common/web_CommonApi.h>
#include <nn/web/common/web_CommonTypes.h>
#include <nn/web/common/web_CommonArgData.h>
#include <nn/la/la_CommonArgumentsReader.h>
#include <nn/la/la_CommonArguments.h>
#include <nn/la/la_AppletToNifmArgumentsWriter.h>

const int DisplayWidth = 1280;
const int DisplayHeight = 720;

//#include "HidController.h"

// ショップ上のシーン
struct Scene
{
    enum class Type
    {
        Top,                // トップ
        ProductDetail,      // タイトル・AOC・バンドル詳細
        Aocs,               // AOC一覧
        Subscriptions,      // 汎用期間券コース一覧/詳細
        Consumption,        // 消費型アイテム
        ConsumptionItem,    // 消費型アイテム詳細
        Setting,            // アカウント情報
        List,               // ニュースから指定された棚表示
        MembershipService,  // 会員サービス案内
        Preinstall,         // プリインストール引換

        Unknown,            // 不明または設定されていない
    } type;

    static Type GetScene(const char* string, int size)
    {
        const std::pair<const char*, Type> Map[] =
        {
            { "top", Type::Top },
            { "product_detail", Type::ProductDetail },
            { "aocs", Type::Aocs },
            { "subscriptions", Type::Subscriptions },
            { "consumption", Type::Consumption },
            { "consumption_item", Type::ConsumptionItem },
            { "setting", Type::Setting },
            { "list", Type::List },
            { "membership_service", Type::MembershipService },
            { "preinstall", Type::Preinstall },
        };

        for (const auto& value : Map)
        {
            if (nn::util::Strncmp(value.first, string, size) == 0)
            {
                return value.second;
            }
        }

        return Type::Unknown;
    }
};

// DummyEcApplet の戻り値
struct DummyShopPageReturnValue
{
    uint64_t m_ExitReason;
    char m_LastUrl[4096];
    uint64_t m_LastUrlSize;

    void SetLastUrl(const char* lastUrl, uint64_t lastUrlSize) NN_NOEXCEPT
    {
        NN_ASSERT_LESS_EQUAL(lastUrlSize, sizeof(m_LastUrl));

        nn::util::Strlcpy(m_LastUrl, lastUrl, sizeof(m_LastUrl));
        m_LastUrlSize = lastUrlSize;
    }
};

// 購入処理成功時のコールバックURL
const char ShopSuccessLastUrl[] = "eshop://success";
const size_t ShopSuccessLastUrlSize = sizeof(ShopSuccessLastUrl);

// 入出力用バッファ
uint8_t g_InData[nn::web::common::CommonArgData::DataSize_Max];
char g_OutData[1024];

//void ReadInData(char* inData, size_t inDataSize) NN_NOEXCEPT;
void ExecuteLibraryAppletFunction(char* outData, const char* inData) NN_NOEXCEPT;
//void WriteOutData(const char* outData, size_t outDataSize) NN_NOEXCEPT;

namespace {
    NN_ALIGNAS(4096) char g_MallocBuffer[120 * 1024 * 1024]; // 本当はshopと揃えるべきなのだけどとりあえず120MB確保。60MBではメモリが足りない
}

void SetReturnValueAsSuccess(DummyShopPageReturnValue* returnValue) NN_NOEXCEPT
{
    returnValue->m_ExitReason = nn::ec::detail::ShopExitReason_CallbackUrlReached;
    returnValue->SetLastUrl(ShopSuccessLastUrl, ShopSuccessLastUrlSize);
}

//------------------------------------------------------------------------------
//  レイヤを初期化
//------------------------------------------------------------------------------
nn::vi::Display* g_pDisplay;
nn::vi::Layer* g_pLayer;


void InitializeLayer()
{
    nn::Result result = nn::vi::OpenDefaultDisplay(&g_pDisplay);
    NN_ASSERT(result.IsSuccess());
    NN_UNUSED(result);

    int width;
    int height;

    width = DisplayWidth;
    height = DisplayHeight;

    result = nn::vi::CreateLayer(&g_pLayer, g_pDisplay);
    NN_ASSERT(result.IsSuccess());

    result = nn::vi::SetLayerScalingMode(g_pLayer, nn::vi::ScalingMode_FitToLayer);
    NN_ASSERT(result.IsSuccess());
}

//------------------------------------------------------------------------------
//  デバイスを初期化
//------------------------------------------------------------------------------
nn::gfx::Device g_Device;

void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
    g_Device.Initialize(info);
}

//------------------------------------------------------------------------------
//  メモリプールを初期化
//------------------------------------------------------------------------------
static const size_t g_VisiblePoolMemorySize = 16 * 1024 * 1024;
static const size_t g_InvisiblePoolMemorySize = 20 * 1024 * 1024;
void* g_pVisiblePoolMemory = NULL;
void* g_pInvisiblePoolMemory = NULL;
void* g_pMemoryPoolStart = NULL;
ptrdiff_t g_MemoryPoolOffset = 0;
void* g_pInvisibleMemoryPoolStart = NULL;
ptrdiff_t g_InvisibleMemoryPoolOffset = 0;
nn::gfx::MemoryPool g_MemoryPool;
nn::gfx::MemoryPool g_InvisibleMemoryPool;

void InitializeMemoryPool()
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached
                               | nn::gfx::MemoryPoolProperty_GpuCached);

    // Determine alignment to allocate space on an alignment boundary
    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
    g_pVisiblePoolMemory = malloc(g_VisiblePoolMemorySize + alignment);

    g_pMemoryPoolStart = nn::util::BytePtr(g_pVisiblePoolMemory).AlignUp(alignment).Get();
    info.SetPoolMemory(g_pMemoryPoolStart, nn::util::align_down(g_VisiblePoolMemorySize,
                                                                nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info)));
    g_MemoryPool.Initialize(&g_Device, info);

    g_MemoryPoolOffset = 0;
}

void InitializeInvisibleMemoryPool()
{

    nn::gfx::MemoryPool::InfoType info;

    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible
                               | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);

    // Determine alignment to allocate space on an alignment boundary
    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
    g_pInvisiblePoolMemory = malloc(g_InvisiblePoolMemorySize + alignment);

    g_pInvisibleMemoryPoolStart = nn::util::BytePtr(g_pInvisiblePoolMemory).AlignUp(alignment).Get();

    info.SetPoolMemory(g_pInvisibleMemoryPoolStart, nn::util::align_down(g_InvisiblePoolMemorySize,
                                                                         nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info)));

    g_InvisibleMemoryPool.Initialize(&g_Device, info);

    g_InvisibleMemoryPoolOffset = 0;
}

//------------------------------------------------------------------------------
//  スワップチェーンを初期化
//------------------------------------------------------------------------------
nn::gfx::SwapChain g_SwapChain;

void InitializeSwapChain()
{
    int width;
    int height;

    width = DisplayWidth;
    height = DisplayHeight;


    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer(g_pLayer);
    info.SetWidth(width);
    info.SetHeight(height);
    info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
    info.SetBufferCount(2);
    if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
    {
        size_t size = g_SwapChain.CalculateScanBufferSize(&g_Device, info);
        g_InvisibleMemoryPoolOffset = nn::util::align_up(g_InvisibleMemoryPoolOffset,
                                                         nn::gfx::SwapChain::GetScanBufferAlignment(&g_Device, info));
        g_SwapChain.Initialize(&g_Device, info, &g_InvisibleMemoryPool, g_InvisibleMemoryPoolOffset, size);
        g_InvisibleMemoryPoolOffset += size;
    }
    else
    {
        g_SwapChain.Initialize(&g_Device, info, NULL, 0, 0);
    }
}

//------------------------------------------------------------------------------
//  キューを初期化
//------------------------------------------------------------------------------
nn::gfx::Queue g_Queue;

void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);
    g_Queue.Initialize(&g_Device, info);
}

//------------------------------------------------------------------------------
//  コマンドバッファを初期化
//------------------------------------------------------------------------------
nn::gfx::CommandBuffer g_CommandBuffer;

void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
    g_CommandBuffer.Initialize(&g_Device, info);
}

//------------------------------------------------------------------------------
//  ビューポートシザーを初期化
//------------------------------------------------------------------------------
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    int width;
    int height;
    //nn::Result result = nn::vi::GetDisplayResolution(&width, &height, g_pDisplay);
    //NN_ASSERT(result.IsSuccess());
    //NN_UNUSED(result);
    width = DisplayWidth;
    height = DisplayHeight;

    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth(static_cast<float>(width));
        viewportInfo.SetHeight(static_cast<float>(height));
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth(width);
        scissorInfo.SetHeight(height);
    }
    info.SetViewportStateInfoArray(&viewportInfo, 1);
    info.SetScissorStateInfoArray(&scissorInfo, 1);
    g_ViewportScissor.Initialize(&g_Device, info);
}

//------------------------------------------------------------------------------
//  サンプラディスクリプタプールの初期化
//------------------------------------------------------------------------------
nn::gfx::DescriptorPool g_SamplerDescriptorPool;
int g_SamplerDescriptorBaseIndex = 0;

void InitializeSamplerDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
    info.SetSlotCount(g_SamplerDescriptorBaseIndex + 1);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
    g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
                                            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
    g_SamplerDescriptorPool.Initialize(&g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, size);
    g_MemoryPoolOffset += size;
}

//------------------------------------------------------------------------------
//  テクスチャディスクリプタプールの初期化
//------------------------------------------------------------------------------
nn::gfx::DescriptorPool g_TextureDescriptorPool;
int g_TextureDescriptorBaseIndex = 0;

void InitializeTextureDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
    info.SetSlotCount(g_TextureDescriptorBaseIndex + 1);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
    g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
                                            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
    g_TextureDescriptorPool.Initialize(&g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, size);
    g_MemoryPoolOffset += size;
}

//------------------------------------------------------------------------------
//  gfx オブジェクトの初期化
//------------------------------------------------------------------------------
nn::util::BytePtr g_pMemoryHeap(NULL);
nn::util::BytePtr g_pMemory(NULL);

void InitializeGfxObjects()
{

    g_pMemoryHeap.Reset(malloc(1024 * 1024 * 32));
    g_pMemory = g_pMemoryHeap;

    InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(g_Device);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
                        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
                        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex);
#endif

    InitializeMemoryPool();
    InitializeInvisibleMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    NN_ASSERT(g_pMemoryHeap.Distance(g_pMemory.Get()) < 1024 * 1024 * 32);

    NN_ASSERT(g_InvisibleMemoryPoolOffset < g_InvisiblePoolMemorySize);
}

//------------------------------------------------------------------------------
//  gfx オブジェクトの終了処理
//------------------------------------------------------------------------------

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* Allocate(size_t size, size_t alignment, void*)
{
    return aligned_alloc(alignment, size);
}
void Free(void* addr, void*)
{
    free(addr);
}
void* Reallocate(void* addr, size_t newSize, void*)
{
    return realloc(addr, newSize);
}
#endif

//------------------------------------------------------------------------------
//  起動パラメータパース 関数
//------------------------------------------------------------------------------

std::map<std::string, std::string> ParseParameters(char* params)
{
    std::map<std::string, std::string> paramList;

    char* pDivided;

    auto AddParamList = [&](char* p) {
        std::string dividedString(pDivided);
        size_t delimiter = dividedString.find("=", 0);

        std::string key = dividedString.substr(0, delimiter);
        std::string value = dividedString.substr(delimiter + 1);

        paramList[key] = value;
    };

    pDivided = strtok(params, "?&");
    if (pDivided != NULL) {
        AddParamList(pDivided);
    }
    while (pDivided != NULL) {
        pDivided = strtok(nullptr, "?&");
        if (pDivided != NULL) {
            AddParamList(pDivided);
        }
    }

    return paramList;
}

//------------------------------------------------------------------------------
//  プリント 関数
//------------------------------------------------------------------------------

void PrintParameter(nn::gfx::util::DebugFontTextWriter& writer, char* str)
{
    nn::util::Color4u8Type color0 = { { 255, 255, 255, 255 } };
    writer.SetScale(1.0f, 1.0f);

    writer.SetTextColor(color0);
    writer.SetCursor(0, 0);
    writer.Print(str);
}

extern "C" void nninitStartup()
{
    // メモリヒープ領域の確保（最初は 0 にしておく）
    auto result = nn::os::SetMemoryHeapSize(0);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // malloc を使えるようにするための初期化
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

// URLパラメータ表示
bool ShowParameter(nn::gfx::util::DebugFontTextWriter& writer, std::pair<std::string, std::string> param, int yCursor) {

    const auto& key = param.first;
    const auto& value = param.second;

    char str[nn::web::common::CommonArgData::DataSize_Max];

    writer.SetCursor(10, yCursor);

    auto IsMatch = [](const char* pStr1, const char* pStr2, int count)
    {
        return nn::util::Strncmp(pStr1, pStr2, count) == 0;
    };

    if (IsMatch(key.c_str(), "scene", sizeof("scene")))
    {
        const std::pair<const char*, const char*> Values[] =
        {
            { "top", "Top menu" },
            { "aocs", "Add-on content list" },
            { "product_detail", "Product detail" },
            { "subscriptions", "Subscription list" },
            { "consumption", "Consumable Item list" },
            { "consumption_item", "Consumable Item Detail" },
            { "membership_service", "Membership Service" },
            { "preinstall", "Preinstall" },
        };

        auto Print = [&writer, &str](const char* string)
        {
            nn::util::SNPrintf(str, sizeof(str), "Scene: %s (Dummy)", string);
            writer.Print(str);
        };

        for (const auto& var : Values)
        {
            if (IsMatch(var.first, value.c_str(), value.length()))
            {
                Print(var.second);
            }
        }
    }
    else if (IsMatch(key.c_str(), "from_scene", sizeof("from_scene")))
    {
        const std::pair<const char*, const char*> Values[] =
        {
            { "news", "News" },
            { "qlc", "Quick launcher" },
            { "demo", "Demo" },
            { "kntf", "Killer notification" },
            { "phld", "Place hoLder" },
            { "my", "My page" },
            { "set", "Settings" },
            { "pre", "Preinstall" },
        };

        auto Print = [&writer, &str](const char* string)
        {
            nn::util::SNPrintf(str, sizeof(str), "From: %s", string);
            writer.Print(str);
        };

        for (const auto& var : Values)
        {
            if (IsMatch(var.first, value.c_str(), value.length()))
            {
                Print(var.second);
            }
        }
    }
    else if (IsMatch(key.c_str(), "dst_app_id", sizeof("dst_app_id")))
    {
        nn::util::SNPrintf(str, sizeof(str), "AppID: %s", value.c_str());
        writer.Print(str);
    }
    else if (IsMatch(key.c_str(), "dst_nsuid", sizeof("dst_nsuid")))
    {
        nn::util::SNPrintf(str, sizeof(str), "NSUID: %s", value.c_str());
        writer.Print(str);
    }
    else if (IsMatch(key.c_str(), "dst_course_id", sizeof("dst_course_id")))
    {
        nn::util::SNPrintf(str, sizeof(str), "CourseID: %s", value.c_str());
        writer.Print(str);
    }
    else if (IsMatch(key.c_str(), "dst_consumption_id", sizeof("dst_consumption_id")))
    {
        nn::util::SNPrintf(str, sizeof(str), "ConsumableID: %s", value.c_str());
        writer.Print(str);
    }
    else if (IsMatch(key.c_str(), "dst_consumption_item_nsuid", sizeof("dst_consumption_item_nsuid")))
    {
        nn::util::SNPrintf(str, sizeof(str), "ConsumableItemNSUID: %s", value.c_str());
        writer.Print(str);
    }
    else {
        // デバッグ用。選択肢に当てはまらない情報を表示
        //nn::util::SNPrintf(str, sizeof(str), "%15s = %s", key.c_str(), value.c_str());
        //NN_LOG(str);
        //writer.SetCursor(0, yCursor);
        //writer.Print(str);
        return false;
    }
    return true;
}

// 購入が成功したとして終了するかを表示するシーンか判定する
bool GetReturnValueOptionOfPurchase(Scene scene)
{
    if (scene.type == Scene::Type::MembershipService)  // 会員サービス案内のみ
    {
        return true;
    }
    else
    {
        return false;
    }
}

void ReadInData(uint8_t* aBuffer)
{
    //SYS_ASSERT_POINTER(aBuffer);

    ::nn::applet::StorageHandle storageHandle;
    bool isSuccess = ::nn::ae::TryPopFromInChannel(&storageHandle);
    NN_ABORT_UNLESS(isSuccess);
    int inStorageSize = static_cast<int>(GetStorageSize(storageHandle));
    auto result = ::nn::applet::ReadFromStorage(storageHandle, 0, const_cast<uint8_t*>(aBuffer), inStorageSize);
    NN_ABORT_UNLESS(result.IsSuccess());
    ::nn::applet::ReleaseStorage(storageHandle);

}

// メイン処理のループ
void HandleMessageAutoLoop(DummyShopPageReturnValue* returnValue) NN_NOEXCEPT
{
    nn::hid::NpadIdType g_NpadIds[] = { nn::hid::NpadId::No1,
                                        nn::hid::NpadId::No2,
                                        nn::hid::NpadId::No3,
                                        nn::hid::NpadId::No4,
                                        nn::hid::NpadId::Handheld,
    };
    const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(nn::hid::NpadIdType);
    nn::hid::DebugPadState DebugpadState = {};
    nn::hid::NpadHandheldState NpadHandheldState;
    nn::hid::NpadFullKeyState NpadFullKeyState[NpadIdCountMax];
    nn::hid::NpadJoyDualState NpadJoyDualState[NpadIdCountMax * 2];
    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);
    // 使用する Npad を設定
    nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);

    for (bool firstFlag=true;;)
    {
        nn::ae::Message message = nn::ae::GetNotificationMessage();

        switch (message)
        {
        case nn::ae::Message_ChangeIntoForeground:
            nn::ae::AcquireForegroundRights();

            if (firstFlag) {
                g_Queue.Present(&g_SwapChain, 1);
                firstFlag = false;
            }

            break;

        case nn::ae::Message_ChangeIntoBackground:
            // リソースアクセス権開放前の処理を行なう
            // （現在のシーンの処理を停止するなど）
            nn::ae::ReleaseForegroundRights();
            break;

        case nn::ae::Message_Exit:
            // ライブラリアプレットとしての終了準備を行なう（データ保存など）
            // その後はループを抜けて、nnMain() からリターンするか、
            //  nn::ae::ExitLibraryApplet() を発行して下さい。
            return;

        case  nn::ae::Message_None:
            nn::hid::GetDebugPadState(&DebugpadState);
            if (DebugpadState.buttons.Test<nn::hid::DebugPadButton::A>()) {
                //NN_LOG("DebugPadでコントローラのAボタン押されたよ\n");
                SetReturnValueAsSuccess(returnValue);
                return;
            }
            else if (DebugpadState.buttons.Test<nn::hid::DebugPadButton::B>())
            {
                return;
            }
            for (int i = 0; i < NpadIdCountMax; i++)
            {
                //現在有効な操作形態(NpadStyleSet)を取得
                nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(g_NpadIds[i]);
                // フルキー操作が有効な場合
                if (style.Test<nn::hid::NpadStyleFullKey>() == true)
                {
                    //最新のNpadのステートを取得
                    nn::hid::GetNpadState(&(NpadFullKeyState[i]), g_NpadIds[i]);
                    if (NpadFullKeyState[i].buttons.Test<nn::hid::NpadButton::A>())
                    {
                        SetReturnValueAsSuccess(returnValue);
                        return;
                    }
                    else if (NpadFullKeyState[i].buttons.Test<nn::hid::NpadButton::B>())
                    {
                        return;
                    }
                }
                // ジョイコン操作が有効な場合
                if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
                {
                    //最新のNpadのステートを取得
                    nn::hid::GetNpadState(&(NpadJoyDualState[i]), g_NpadIds[i]);
                    if (NpadJoyDualState[i].buttons.Test<nn::hid::NpadButton::A>())
                    {
                        SetReturnValueAsSuccess(returnValue);
                        return;
                    }
                    else if (NpadJoyDualState[i].buttons.Test<nn::hid::NpadButton::B>())
                    {
                        return;
                    }
                }
                // 携帯機コントローラー操作が有効な場合
                if (style.Test<nn::hid::NpadStyleHandheld>() == true)
                {
                    //最新のNpadのステートを取得
                    nn::hid::GetNpadState(&(NpadHandheldState), g_NpadIds[i]);
                    if (NpadHandheldState.buttons.Test<nn::hid::NpadButton::A>())
                    {
                        SetReturnValueAsSuccess(returnValue);
                        return;
                    }
                    else if (NpadHandheldState.buttons.Test<nn::hid::NpadButton::B>())
                    {
                        return;
                    }
                }
            }

            break;

        default:
            //NN_LOG("message: default\n");
            // 未知のメッセージは何もせずにスルーして下さい
            break;

            //TODO: 終了条件を定義し、実行可能にする
            //TODO: 上記ができたらreturnは消す
            //return;
        }
        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / 60));
    }
}

void ExecuteLibraryAppletFunction(DummyShopPageReturnValue* outReturnValue, char* outData, const char* inData) NN_NOEXCEPT
{
    *outReturnValue = {};

    // ライブラリアプレットとしての初期化やシーンの起動などを行なう

    //メモリアロケート周りのバグ回避処理
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nv::SetGraphicsAllocator(Allocate, Free, Reallocate, NULL);
    nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);

    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();

    nn::vi::Initialize();
    InitializeLayer();
    nn::gfx::Initialize();
    InitializeGfxObjects();

    // デバッグフォント初期化
    const bool userMemoryPoolEnable = false;    // true にすると、ユーザーのメモリプールを使用します
    const int charCountMax = 1024;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charCountMax);

    info.SetUserMemoryPoolEnabled(userMemoryPoolEnable);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(&g_Device, info);
    nn::util::BytePtr debugFontHeap(new uint8_t[debugFontHeapSize]);

    size_t debugFontMemoryPoolSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(&g_Device, info);


    nn::gfx::util::DebugFontTextWriter writer;
    writer.Initialize(
        &g_Device,
        info,
        debugFontHeap.Get(),
        debugFontHeapSize,
        userMemoryPoolEnable ? &g_MemoryPool : nullptr,
        userMemoryPoolEnable ? g_MemoryPoolOffset : 0,
        userMemoryPoolEnable ? debugFontMemoryPoolSize : 0
        );

    g_MemoryPoolOffset += userMemoryPoolEnable ? debugFontMemoryPoolSize : 0;

    int width;
    int height;

    width = DisplayWidth;
    height = DisplayHeight;

    writer.SetDisplayWidth(width);
    writer.SetDisplayHeight(height);
    writer.SetTextureDescriptor(&g_TextureDescriptorPool, g_TextureDescriptorBaseIndex);
    writer.SetSamplerDescriptor(&g_SamplerDescriptorPool, g_SamplerDescriptorBaseIndex);

    // 表示内容変更の必要は無いため描画は1回のみで十分
    //for (int frame = 0; frame < 60 * 1; ++frame)
    {
        // フォント表示
        nn::util::Color4u8Type color = { { 255, 255, 255, 255 } }; //白文字
        writer.SetScale(1.5f, 1.5f);
        writer.SetTextColor(color);

        // 念のためinDataは直接触らない
        char copyInData[nn::web::common::CommonArgData::DataSize_Max];
        strcpy(copyInData, inData);
        NN_LOG("[DummyECApplet] LaunchParameter: %s\n", copyInData);

        int yCursor = 60;
        auto paramList = ParseParameters(copyInData);
        for (auto param : paramList)
        {
            NN_LOG("[DummyECApplet] Parameter: %s = %s\n", param.first.c_str(), param.second.c_str());
            if (ShowParameter(writer, param, yCursor))
            {
                yCursor += 40;
            }
        }

        // 呼び出されているショップのシーンを取得する
        Scene scene;
        {
            auto itr = paramList.find("scene");
            if (itr != paramList.end())
            {
                scene.type = Scene::GetScene(itr->second.c_str(), itr->second.length());
            }
            else
            {
                scene.type = Scene::Type::Unknown;
            }
        }

        // 実行中のアプリケーションと異なる ApplicationId が "dst_app_id" パラメータに指定されていた場合は警告を表示する
        {
            auto itr = paramList.find("dst_app_id");
            if (itr != paramList.end())
            {
                nn::ncm::ApplicationId appId = { std::strtoull(itr->second.c_str(), NULL, 16) };

                nn::applet::AppletIdentityInfo info = nn::ae::GetMainAppletIdentityInfo();

                if (info.IsApplication() && info.applicationId != appId)
                {
                    writer.SetCursor(10, 300);
                    writer.Print("Warning: Designated application ID on calling show-shop API differs from running application's one.");
                    writer.SetCursor(10, 340);
                    writer.Print("             If expected, please ignore the message.");
                }
            }
        }

        // アプレットの終了するボタンの案内の表示
        // 購入が成功したかにより戻り値を返す必要があるシーンの場合、2パターンの終了の案内を表示する
        if (GetReturnValueOptionOfPurchase(scene))
        {
            writer.SetCursor(760, 640);
            writer.Print("A: End this applet as purchase success.");
            writer.SetCursor(760, 680);
            writer.Print("B: End this applet.");
        }
        else
        {
            writer.SetCursor(760, 640);
            writer.Print("A: End this applet.");
        }

        // コマンド生成
        g_CommandBuffer.Reset();
        g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
                                                nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
        g_CommandBuffer.AddCommandMemory(&g_MemoryPool, g_MemoryPoolOffset, 1024 * 1024);
        g_pMemory.AlignUp(256);
        g_CommandBuffer.AddControlMemory(g_pMemory.Get(), 256);
        g_CommandBuffer.Begin();
        {
            nn::gfx::ColorTargetView* pTarget = g_SwapChain.AcquireNextScanBufferView();
            g_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);
            g_CommandBuffer.SetDescriptorPool(&g_TextureDescriptorPool);
            g_CommandBuffer.SetDescriptorPool(&g_SamplerDescriptorPool);
            g_CommandBuffer.ClearColor(pTarget, 0.1f, 0.1f, 0.1f, 1.0f, NULL);
            g_CommandBuffer.SetRenderTargets(1, &pTarget, NULL);
            g_CommandBuffer.SetViewportScissorState(&g_ViewportScissor);

            // デバッグフォント用のコマンド生成
            writer.Draw(&g_CommandBuffer);
        }
        g_CommandBuffer.End();

        // コマンドの実行
        g_Queue.ExecuteCommand(&g_CommandBuffer, NULL);
    }

    HandleMessageAutoLoop(outReturnValue);
} // NOLINT(impl/function_size)

void LibraryAppletMenuMain(const nn::ae::LibraryAppletSelfInfo& info)
{
    // 起動属性のチェック
    NN_ABORT_UNLESS(info.appletId == nn::applet::AppletId_LibraryAppletShop);
    NN_ABORT_UNLESS(info.libraryAppletMode == nn::applet::LibraryAppletMode_AllForeground);
    NN_ABORT_UNLESS(info.isUnwound == false);

    //呼出し元からのデータ読込み

    //共通データpop
    ::nn::la::CommonArgumentsReader commonArgumentsReader;

    commonArgumentsReader.TryPopFromInChannel();

    //cruiserデータ読込み
    ReadInData(g_InData);
    struct nn::web::common::CommonArgData::Data formattedData;

    struct nn::web::common::CommonArgData::Header header;
    header = nn::web::common::CommonArgData::GetHeader(g_InData);

    formattedData = nn::web::common::CommonArgData::GetData(g_InData, nn::web::common::CommonArgData::ArgKind::RequestRelativeURL);

    DummyShopPageReturnValue returnValue;

    //メイン処理
    ExecuteLibraryAppletFunction(&returnValue, g_OutData, (char*)formattedData.pBuffer);

    nn::applet::StorageHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&handle, sizeof(struct DummyShopPageReturnValue)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::WriteToStorage(handle, 0, &returnValue, sizeof(struct DummyShopPageReturnValue)));
    nn::ae::PushToOutChannel(handle);

    ::nn::ae::ExitLibraryApplet();
}

extern "C" void nnMain()
{
    //アプレット実処理の呼出し
    nn::ae::InvokeLibraryAppletMain(LibraryAppletMenuMain);
}
