﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace ui{

    struct Position
    {
    public:
        Position operator+(const Position& v) const NN_NOEXCEPT {
            return { this->x + v.x, this->y + v.y };
        }

        Position& operator+=(const Position& v) NN_NOEXCEPT
        {
            this->x += v.x;
            this->y += v.y;
            return *this;
        }

        Position operator-(const Position& v) const NN_NOEXCEPT
        {
            return { this->x - v.x, this->y - v.y };
        }

        Position& operator-=(const Position& v) NN_NOEXCEPT
        {
            this->x -= v.x;
            this->y -= v.y;
            return *this;
        }

    public:
        int x;
        int y;
    };

    struct Size
    {
        int width;
        int height;
    };

    struct Rectangle
    {
    public:
        bool Contains(const Position& point) NN_NOEXCEPT
        {
            if(point.x < this->position.x
                || point.y < this->position.y
                || point.x >= this->position.x + this->size.width
                || point.y >= this->position.y + this->size.height
                )
            {
                return false;
            }
            return true;
        }

    public:
        Position position;
        Size size;
    };

    struct Aabb
    {
    public:
        static Aabb FromRectangle(const Rectangle& rect) NN_NOEXCEPT
        {
            return
            {
                rect.position.x,
                rect.position.x + rect.size.width,
                rect.position.y,
                rect.position.y + rect.size.height
            };
        }

        // 有効な領域を表していない場合 true 。
        bool IsEmpty() const NN_NOEXCEPT
        {
            return (this->xBeg >= this->xEnd) || (this->yBeg >= this->yEnd);
        }

        // 点が有効な範囲に入っている場合 true 。
        bool Contains(const Position& point) NN_NOEXCEPT
        {
            if(point.x < this->xBeg
                || point.x >= this->xEnd
                || point.y < this->yBeg
                || point.y >= this->yEnd
                )
            {
                return false;
            }
            return true;
        }

        // 共通部分を計算する。
        // この関数で取得できる AABB は裏返っている場合がある。
        // 有効な領域を表しているかの判定には !IsEmpty() を使用する。
        Aabb Intersect(const Aabb& v) const NN_NOEXCEPT
        {
            return
            {
                std::max(this->xBeg, v.xBeg),
                std::min(this->xEnd, v.xEnd),
                std::max(this->yBeg, v.yBeg),
                std::min(this->yEnd, v.yEnd)
            };
        }

    public:
        int xBeg;
        int xEnd;
        int yBeg;
        int yEnd;
    };

    // 上右下左の順 (style-sheet における指定と同様)
    struct Margin
    {
        int top;
        int right;
        int bottom;
        int left;
    };

}
