﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ui_MenuItemRow.h"

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "util/ui_FindFocusAcceptingItem.h"

namespace ui{

    MenuItemRow::MenuItemRow(const MenuItemTableStyle& style) NN_NOEXCEPT
        : MenuItemTableBase(style)
    {
    }

    MenuItemRow::~MenuItemRow() NN_NOEXCEPT
    {
    }

    void MenuItemRow::UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT
    {
        // 子を更新しながら最大の高さを計算
        int cellHeightMax = 0;
        for(size_t i = 0; i < m_CellList.size(); i++)
        {
            UpdateCellLayout1(i, context);
            auto size = GetExpectedCellSize(i);
            cellHeightMax = std::max(cellHeightMax, size.height);
        }

        // 子の位置を設定
        int curPosX = m_Style.GetBorderStyle().GetLeftWidth();
        const int posY = m_Style.GetBorderStyle().GetTopWidth();
        for(size_t i = 0; i < m_CellList.size(); i++)
        {
            auto minSize = GetExpectedCellSize(i);
            Size cellSize = {minSize.width, cellHeightMax};
            UpdateCellLayout2(i, cellSize, context);

            m_CellList[i]->SetLayoutPosition({curPosX, posY});
            curPosX += cellSize.width + m_Style.GetBorderStyle().GetVerticalWidth();
        }

        int width  = curPosX - m_Style.GetBorderStyle().GetVerticalWidth() + m_Style.GetBorderStyle().GetRightWidth();
        int height = cellHeightMax + m_Style.GetBorderStyle().GetTopWidth() + m_Style.GetBorderStyle().GetBottomWidth();

        SetLayoutSize({width, height});
    }

    MenuButtonResult MenuItemRow::HandleLeftButtonPressed(const MenuButtonHandleContext& context)    NN_NOEXCEPT
    {
        auto pChild =context.GetPreviousHandlerItem();
        if(pChild == nullptr)
        {
            return MenuButtonResult::GetNotHandled();
        }

        auto info = pChild->GetParentInfo();
        NN_SDK_ASSERT_EQUAL(this, info.pParent.lock().get());
        NN_SDK_ASSERT_RANGE(info.index, 0u, m_CellList.size());
        NN_SDK_ASSERT_EQUAL(m_CellList[info.index], pChild);

        for(int index = info.index - 1; index >= 0; index--)
        {
            if(auto pToFocus = util::FindFocusAcceptingItem::Find(m_CellList[index], context.GetCurrentFocusedItem(), util::FindFocusAcceptingItem::Distance))
            {
                context.GetPage()->ChangeFocus(pToFocus);
                return MenuButtonResult::GetHandled();
            }
        }

        // 範囲外になる場合は NotHandled
        return MenuButtonResult::GetNotHandled();
    }

    MenuButtonResult MenuItemRow::HandleRightButtonPressed(const MenuButtonHandleContext& context)  NN_NOEXCEPT
    {
        auto pChild = context.GetPreviousHandlerItem();
        if(pChild == nullptr)
        {
            return MenuButtonResult::GetNotHandled();
        }

        auto info = pChild->GetParentInfo();
        NN_SDK_ASSERT_EQUAL(this, info.pParent.lock().get());
        NN_SDK_ASSERT_RANGE(info.index, 0u, m_CellList.size());
        NN_SDK_ASSERT_EQUAL(m_CellList[info.index], pChild);

        for(int index = info.index + 1; index < m_CellList.size(); index++)
        {
            if(auto pToFocus = util::FindFocusAcceptingItem::Find(m_CellList[index], context.GetCurrentFocusedItem(), util::FindFocusAcceptingItem::Distance))
            {
                context.GetPage()->ChangeFocus(pToFocus);
                return MenuButtonResult::GetHandled();
            }
        }

        // 範囲外になる場合は NotHandled
        return MenuButtonResult::GetNotHandled();
    }

}

