﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "../../panel/panel_PanelContainer.h"
#include "../ui_Macro.h"
#include "../ui_StyleTypes.h"
#include "../util/ui_CalculateItemSize.h"
#include "../util/ui_CalculateAlignedPosition.h"
#include "../util/ui_CalculateBackgroundVisibility.h"
#include "../base/ui_MenuItemAttributeNoFocus.h"

namespace ui{ namespace base{

    template<typename TBaseType>
    class MenuItemBoxBase
        : public base::MenuItemAttributeNoFocus<TBaseType>
    {
    public:
        typedef typename TBaseType::ConcreteType ConcreteType;

    public:
        explicit MenuItemBoxBase() NN_NOEXCEPT
        {
            m_pPanel = std::make_shared<panel::PanelContainer>();
        }

        virtual ~MenuItemBoxBase() NN_NOEXCEPT
        {
            RemoveInnerItem();
        }

        virtual int GetChildCount() const NN_NOEXCEPT UI_FINAL
        {
            return m_pInnerItem ? 1 : 0;
        }

        virtual std::shared_ptr<IMenuItem> GetChild(int index) NN_NOEXCEPT UI_FINAL
        {
            if(index != 0)
            {
                return nullptr;
            }

            return m_pInnerItem;
        }

        virtual std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT UI_FINAL
        {
            return m_pPanel;
        }

        virtual void UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT UI_FINAL
        {
            UpdateInnerLayout(context);
            UpdateOuterLayout(GetExpectedOuterSize(), context);
        }

        void UpdateInnerLayout(const MenuLayoutUpdateContext& context) NN_NOEXCEPT
        {
            auto margin = this->GetCurrentMargin();

            Size innerSize = {};
            if(m_pInnerItem)
            {
                m_pInnerItem->UpdateLayoutRecursively(context);
                innerSize = m_pInnerItem->GetLayoutSize();
            }

            m_InnerLayout.m_ItemPosition = Position{margin.left, margin.top};
            m_InnerLayout.m_Size  = Size{
                innerSize.width  + margin.left + margin.right,
                innerSize.height + margin.top + margin.bottom
            };
            m_InnerLayout.m_ExpectedOuterSize = util::CalculateItemSize::Calculate(
                [this](){ return m_InnerLayout.m_Size; },
                this->GetCurrentSizeStyle()
            );
        }

        Size GetExpectedOuterSize() const NN_NOEXCEPT
        {
            return m_InnerLayout.m_ExpectedOuterSize;
        }

        void UpdateOuterLayout(const Size& outerSize, const MenuLayoutUpdateContext& context) NN_NOEXCEPT
        {
            NN_UNUSED(context);
            auto innerPos   = m_InnerLayout.m_ItemPosition;
            auto layoutSize = m_InnerLayout.m_Size;

            if(m_pInnerItem)
            {
                auto alignment = this->GetCurrentAlignment();

                // 位置を決定
                auto layoutPos = util::CalculateAlignedPosition::Calculate(layoutSize, outerSize, alignment);
                m_pInnerItem->SetLayoutPosition(layoutPos + innerPos);
            }

            this->SetLayoutSize(outerSize);
        }

        virtual void UpdatePanel() NN_NOEXCEPT UI_FINAL
        {
            auto bgStyle = this->GetCurrentBackgroundStyle();
            auto bgVisibility = util::CalculateBackgroundVisibility::Calculate(bgStyle, m_pInnerItem);
            m_pPanel->SetVisibility(bgVisibility.visibility);
            m_pPanel->SetColor(bgVisibility.color);

            auto pos = this->GetLayoutPosition();
            auto size = this->GetLayoutSize();
            m_pPanel->SetPosition(pos.x, pos.y);
            m_pPanel->SetSize(size.width, size.height);
        }

        virtual BackgroundStyle GetCurrentBackgroundStyle() const NN_NOEXCEPT = 0;
        virtual SizeStyle       GetCurrentSizeStyle()       const NN_NOEXCEPT = 0;
        virtual Margin          GetCurrentMargin()          const NN_NOEXCEPT = 0;
        virtual ItemAlignment   GetCurrentAlignment()       const NN_NOEXCEPT = 0;

        void RemoveInnerItem() NN_NOEXCEPT
        {
            if(m_pInnerItem)
            {
                m_pInnerItem->OnParentItemChanged(nullptr, 0);
            }
            m_pInnerItem.reset();
            m_pPanel->ClearChildren();
        }

        std::shared_ptr<ConcreteType> SetInnerItem(const std::shared_ptr<IMenuItem>& item) NN_NOEXCEPT
        {
            // 以前のものを除外
            RemoveInnerItem();

            // 新しいものを設定
            if(item)
            {
                m_pInnerItem = item;
                item->OnParentItemChanged(this->shared_from_this(), 0);
                m_pPanel->AddChild(item->GetPanel());
            }

            return this->shared_from_this();
        }

        std::shared_ptr<ConcreteType> SetPanelName(const std::string& value) NN_NOEXCEPT
        {
            m_pPanel->SetPanelName(value);
            return this->shared_from_this();
        }

    protected:
        std::shared_ptr<IMenuItem> m_pInnerItem;
        std::shared_ptr<panel::PanelContainer> m_pPanel;

        struct
        {
            Position m_ItemPosition = {};
            Size m_Size = {};
            Size m_ExpectedOuterSize = {};
        } m_InnerLayout = {};
    };

}}
