﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NotificationSnapShotDumperProgress.h"

#include <nn/util/util_FormatString.h>
#include "../../config/NotificationParameter.h"

namespace scene{ namespace overlay{

    namespace {
        enum Priority
        {
            Priority_Highest = 0,
            Priority_Failure,
            Priority_Complete,
            Priority_Progress,
            Priority_Lowest,
        };
    }

    NotificationSnapShotDumperProgress::NotificationSnapShotDumperProgress() NN_NOEXCEPT
    {
        typedef config::NotificationStyle Style;

        m_pPanel = std::make_shared<panel::PanelContainer>();
        m_pPanel->SetPosition(nn::util::Vector3f(0, 0, 0));
        m_pPanel->SetSize(Style::GetSizeF());
        m_pPanel->SetColor(Style::GetBackgroundColor());
        m_pPanel->SetVisibility(panel::PanelVisibility::Visible);

        m_pBackground = std::make_shared<panel::PanelText>();
        m_pBackground->SetPosition(Style::GetSnapShotDumperProgressIndicatorPosition());
        m_pBackground->SetSize(Style::GetSnapShotDumperProgressIndicatorSize());
        m_pBackground->SetColor(Style::GetSnapShotDumperProgressIndicatorBackgroundColor());
        m_pBackground->SetVisibility(panel::PanelVisibility::Visible);

        m_pForeground = std::make_shared<panel::PanelText>();
        m_pForeground->SetPosition(Style::GetSnapShotDumperProgressIndicatorPosition());
        m_pForeground->SetSize(Style::GetSnapShotDumperProgressIndicatorSize());
        m_pForeground->SetColor(Style::GetSnapShotDumperProgressIndicatorForegroundColor());
        m_pForeground->SetVisibility(panel::PanelVisibility::Visible);

        m_pText = std::make_shared<panel::PanelText>();
        m_pText->SetPosition(nn::util::Vector3f(0, 0, 0));
        m_pText->SetSize(Style::GetSizeF());
        m_pText->SetColor(Style::GetBackgroundColor());
        m_pText->SetVisibility(panel::PanelVisibility::Visible);
        m_pText->SetTextPosition(Style::GetAudioDeviceTextPositionX(), Style::GetAudioDeviceTextPositionY());
        m_pText->SetTextSize(Style::GetAudioDeviceTextSize());
        m_pText->SetTextColor(Style::GetAudioDeviceTextColor());

        m_pPanel->AddChild(m_pText);
        m_pPanel->AddChild(m_pBackground);
        m_pPanel->AddChild(m_pForeground);
    }

    std::shared_ptr<panel::IPanel> NotificationSnapShotDumperProgress::GetPanel() NN_NOEXCEPT
    {
        return m_pPanel;
    }

    void NotificationSnapShotDumperProgress::ReadyToAppear() NN_NOEXCEPT
    {
    }

    bool NotificationSnapShotDumperProgress::SetupProgress(uint32_t percentage) NN_NOEXCEPT
    {
        char str[256] = "Dump completed.";
        nn::TimeSpan ttl = config::NotificationTtl::GetTtlSnapShotDumperComplete();
        Priority priority = Priority_Complete;
        float barRatio = 1;

        if(percentage >= std::numeric_limits<uint32_t>::max())
        {
            nn::util::SNPrintf(str, sizeof(str), "Dump failed.");
            ttl = config::NotificationTtl::GetTtlSnapShotDumperFailure();
            priority = Priority_Failure;
            barRatio = 0;
        }
        else
        {
            uint32_t value = std::min(100u, percentage);
            if(value < 100)
            {
                nn::util::SNPrintf(str, sizeof(str), "Now dumping... %d%%", value);
                ttl = config::NotificationTtl::GetTtlSnapShotDumperProgress();
                priority = Priority_Progress;
                barRatio = value / 100.f;
            }
        }

        m_pText->SetText(str);

        auto sizeMax = config::NotificationStyle::GetSnapShotDumperProgressIndicatorSize();
        m_pForeground->SetSize(nn::util::MakeFloat2(sizeMax.x * barRatio, sizeMax.y));

        m_TimeToLive = ttl;
        m_Category   = NotificationCategory_SnapShotDumperProgress;
        m_Priority   = priority;
        return true;
    }


}}
