﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneMenuPageBase.h"

#include "../../Config.h"
#include "../../framework/Hid.h"
#include "../../framework/Framework.h"
#include "../../AppletMessageThread.h"

#include "../../ui/util/ui_FindFocusAcceptingItem.h"
#include "../../ui/util/ui_FindTouchAcceptingItem.h"
#include "../../ui/util/ui_ProcessButtonHandling.h"
#include "../../ui/util/ui_ProcessAnalogStickHandling.h"
#include "../../ui/util/ui_ProcessTouchHandling.h"
#include "../../ui/util/ui_ProcessUpdatePanel.h"
#include "../../ui/util/ui_ProcessNotifyFocusedItemChanged.h"
#include "../../ui/util/ui_ProcessLoadStoreValue.h"

#include "SceneStackedItemMenuPage.h"

namespace scene{ namespace menu{

    SceneMenuPageBase::SceneMenuPageBase(
        const std::shared_ptr<void>& pArg,
        ui::ItemAlignment itemAlignment
    ) NN_NOEXCEPT
    {
        m_pParameter = std::static_pointer_cast<SceneMenuPageParameter>(pArg);

        m_pMenuItemRoot = std::make_shared<ui::MenuItemBox>(
            ui::MenuItemBoxStyle()
            .SetSizeStyle(ui::SizeStyle()
                .SetFixedSize({ScreenWidth, ScreenHeight})
            )
            .SetBackgroundStyle(
                ui::BackgroundStyle()
                .SetMode(ui::BackgroundMode_Transparent)
            )
            .SetAlignment(itemAlignment)
        );
    }

    SceneMenuPageBase::~SceneMenuPageBase() NN_NOEXCEPT
    {
        NN_ASSERT_EQUAL(m_PageActivity.m_State, PageState_Closed);
    }

    //-------------------------------------------
    // Scene
    //-------------------------------------------

    void SceneMenuPageBase::OnDestroying() NN_NOEXCEPT
    {
        CloseImpl();
    }

    SceneUpdateResult SceneMenuPageBase::Update() NN_NOEXCEPT
    {
        return UpdateImpl();
    }

    std::shared_ptr<panel::IPanel> SceneMenuPageBase::GetPanel() NN_NOEXCEPT
    {
        return m_pMenuItemRoot->GetPanel();
    }

    //-------------------------------------------

    void SceneMenuPageBase::SetMenuItem(const std::shared_ptr<ui::IMenuItem>& value) NN_NOEXCEPT
    {
        m_pMenuItemRoot->SetInnerItem(value);
        m_pFocusedItem.reset();
    }

    void SceneMenuPageBase::SetPanelName(const std::string& value) NN_NOEXCEPT
    {
        m_pMenuItemRoot->SetPanelName(value);
    }

    std::shared_ptr<ui::IMenuItem> SceneMenuPageBase::GetRootMenuItem() NN_NOEXCEPT
    {
        return m_pMenuItemRoot;
    }

    void SceneMenuPageBase::OnEnteringPage(const std::shared_ptr<ui::IMenuPage>&) NN_NOEXCEPT
    {
    }

    void SceneMenuPageBase::OnLeavingPage(const std::shared_ptr<ui::IMenuPage>&) NN_NOEXCEPT
    {
    }

    void SceneMenuPageBase::OnClosingPage(const std::shared_ptr<ui::IMenuPage>&) NN_NOEXCEPT
    {
    }

    //---------------------------------------------------

    SceneUpdateResult SceneMenuPageBase::UpdateImpl() NN_NOEXCEPT
    {
        const auto currentPageState = m_PageActivity.m_State;
        if(currentPageState == PageState_Closed)
        {
            return SceneUpdateResult::GetPopResult(); // 既に Close しているので除外する
        }

        auto pReactor = std::make_shared<MenuPageReactor>();

        bool       isRetryRequired;
        PageState  nextPageState;
        if(currentPageState == PageState_Created)
        {
            // レイアウトの初回計算を要求
            pReactor->UpdateLayout();

            isRetryRequired = true;
            nextPageState   = PageState_EnteringPage;
        }
        else if(currentPageState == PageState_OutPage)
        {
            // 子ページ呼び出しから返ってきた。
            GatherRequestFromChildPageResultImpl(pReactor);

            // 子ページから返ってきた場合は Enter する。
            isRetryRequired = true;
            nextPageState   = PageState_EnteringPage;
        }
        else if(currentPageState == PageState_WaitLayer)
        {
            // レイヤ呼出から返ってきた場合は Enter せずにそのまま InPage になる

            // W/A: Panel の描画の不具合の対処のため再描画させる
            if(auto pPanel = GetPanel())
            {
                pPanel->RequestRedraw();
            }

            isRetryRequired = true;
            nextPageState   = PageState_InPage;
        }
        else if(currentPageState == PageState_EnteringPage)
        {
            // 誰もフォーカスを持っていない場合、誰かフォーカスを取れないか探す
            if(m_pFocusedItem.lock() == nullptr)
            {
                pReactor->ChangeFocus(m_pMenuItemRoot);
            }

            // W/A: Panel の描画の不具合の対処のため再描画させる
            if(auto pPanel = GetPanel())
            {
                pPanel->RequestRedraw();
            }

            // コールバック呼出
            OnEnteringPage(pReactor);

            isRetryRequired = true;
            nextPageState   = PageState_InPage;
        }
        else //if(currentPageState == PageState_InPage)
        {
            // ユーザー入力を処理
            GatherRequestFromUserInputImpl(pReactor);

            // TORIAEZU: InPage の場合毎フレーム描画を更新させる
            pReactor->UpdatePanel();

            isRetryRequired = false;
            nextPageState   = PageState_InPage;
        }

        // ページ内要求を処理
        ProcessPageRequestImpl(
            pReactor,
            currentPageState == PageState_EnteringPage
        );


        // 遷移要求を処理
        return ProcessTransitionRequestImpl(
            pReactor,
            std::make_pair(
                nextPageState,
                isRetryRequired ? SceneUpdateResult::GetIncompleteResult() : SceneUpdateResult::GetKeepResult()
            )
        );
    }

    void SceneMenuPageBase::CloseImpl() NN_NOEXCEPT
    {
        if(m_PageActivity.m_State == PageState_Closed)
        {
            return;
        }

        auto pReactor = std::make_shared<MenuPageReactor>();
        pReactor->ClosePage();

        ProcessPageRequestImpl(pReactor, false);
        ProcessTransitionRequestImpl(pReactor, std::make_pair(PageState_InPage, SceneUpdateResult::GetKeepResult()));
        NN_ASSERT_EQUAL(m_PageActivity.m_State, PageState_Closed);
    }

    //---------------------------------------------------------------------------------------

    namespace {
        ui::MenuButtonResult GatherRequestFromUserInputTouchImpl(
            const std::shared_ptr<MenuPageReactor>& pReactor,
            SceneMenuPageBase::TouchState* pTouchState,
            const std::shared_ptr<ui::IMenuItem>& pRootItem,
            const std::shared_ptr<ui::IMenuItem>& pFocusedItem
        ) NN_NOEXCEPT
        {
            auto touchState = Hid::GetTouchState();
            nn::util::Vector3f pos = nn::util::Vector3f(touchState.touchPositionX, touchState.touchPositionY, 0);

            bool isHandled = false;
            // Up
            if(pTouchState->m_IsTouching && (touchState.isReleased || !touchState.isHeld))
            {
                auto pHeldItem = pTouchState->m_pHeldItem.lock();

                // 離されたことを通知
                if(pHeldItem)
                {
                    auto result = ui::util::ProcessTouchHandling::HandleTouchUp(pHeldItem, ui::MenuButtonHandleContext(pFocusedItem, pHeldItem, nullptr, pReactor), pos);
                    isHandled |= result.IsHandled();
                }

                // タッチを通知
                auto pTargetItem = ui::util::FindTouchAcceptingItem::Find(pRootItem, pos);
                if(pTargetItem && (pHeldItem == pTargetItem || pHeldItem == nullptr))
                {
                    auto result = ui::util::ProcessTouchHandling::HandleTouched(pTargetItem, ui::MenuButtonHandleContext(pFocusedItem, pTargetItem, nullptr, pReactor), pos);
                    isHandled |= result.IsHandled();
                }

                // タッチ状態の追跡を解除
                pTouchState->m_IsTouching = false;
                pTouchState->m_pHeldItem.reset();
                pTouchState->m_Position = {};
            }

            // Move
            if(pTouchState->m_IsTouching && touchState.isHeld)
            {
                if(auto pHeldItem = pTouchState->m_pHeldItem.lock())
                {
                    NN_ASSERT(pHeldItem->IsTouchAcceptable());

                    // 位置が変わっていたら通知
                    if(pos.GetX() != pTouchState->m_Position.GetX() || pos.GetY() != pTouchState->m_Position.GetY())
                    {
                        auto result = ui::util::ProcessTouchHandling::HandleTouchMove(pHeldItem, ui::MenuButtonHandleContext(pFocusedItem, pHeldItem, nullptr, pReactor), pos);
                        isHandled |= result.IsHandled();
                    }

                    pTouchState->m_Position = pos;
                }
            }

            // Down
            if(touchState.isPressed)
            {
                pTouchState->m_IsTouching = true;

                auto pTargetItem = ui::util::FindTouchAcceptingItem::Find(pRootItem, pos);
                if(pTouchState->m_pHeldItem.expired() && pTargetItem)
                {
                    // タッチ状態の追跡を開始
                    pTouchState->m_pHeldItem= pTargetItem;
                    pTouchState->m_Position = pos;

                    // 押されたことを通知
                    auto result = ui::util::ProcessTouchHandling::HandleTouchDown(pTargetItem, ui::MenuButtonHandleContext(pFocusedItem, pTargetItem, nullptr, pReactor), pos);
                    isHandled |= result.IsHandled();
                }
            }

            return isHandled ? ui::MenuButtonResult::GetHandled() : ui::MenuButtonResult::GetNotHandled();
        }

        ui::MenuButtonResult GatherRequestFromUserInputButtonImpl(
            const std::shared_ptr<MenuPageReactor>& pReactor,
            const std::shared_ptr<ui::IMenuItem>& pFocusedItem
        ) NN_NOEXCEPT
        {
            auto buttonState = Hid::GetButtonState();

            // SIGLO-76763
            // メニューが閉じられた直後に FG のアプレットが A, B ボタンをハンドリングするのを避けるため
            // A, B ボタンは離したときに判定する。

            if(buttonState.isUp.B)
            {
                if(pFocusedItem)
                {
                    ui::MenuButtonHandleContext context(pFocusedItem, pFocusedItem, nullptr, pReactor);
                    auto result = ui::util::ProcessButtonHandling::HandleNegativeButtonPressed(pFocusedItem, context);
                    if(result.GetCode() != ui::MenuButtonResultCode_NotHandled)
                    {
                        return result;
                    }
                }

                // 誰もハンドルしなかったらページを閉じる
                pReactor->ClosePage();
                return ui::MenuButtonResult::GetHandled();
            }

            if(buttonState.isUp.A && pFocusedItem)
            {
                ui::MenuButtonHandleContext context(pFocusedItem, pFocusedItem, nullptr, pReactor);
                auto result = ui::util::ProcessButtonHandling::HandlePositiveButtonPressed(pFocusedItem, context);
                if(result.GetCode() != ui::MenuButtonResultCode_NotHandled)
                {
                    return result;
                }
            }

            if(buttonState.isTriggered.Up && !buttonState.isPressed.Down && pFocusedItem)
            {
                ui::MenuButtonHandleContext context(pFocusedItem, pFocusedItem, nullptr, pReactor);
                auto result = ui::util::ProcessButtonHandling::HandleUpButtonPressed(pFocusedItem, context);
                if(result.GetCode() != ui::MenuButtonResultCode_NotHandled)
                {
                    return result;
                }
            }

            if(buttonState.isTriggered.Down && !buttonState.isPressed.Up && pFocusedItem)
            {
                ui::MenuButtonHandleContext context(pFocusedItem, pFocusedItem, nullptr, pReactor);
                auto result = ui::util::ProcessButtonHandling::HandleDownButtonPressed(pFocusedItem, context);
                if(result.GetCode() != ui::MenuButtonResultCode_NotHandled)
                {
                    return result;
                }
            }

            if(buttonState.isTriggered.Right && !buttonState.isPressed.Left && pFocusedItem)
            {
                ui::MenuButtonHandleContext context(pFocusedItem, pFocusedItem, nullptr, pReactor);
                auto result = ui::util::ProcessButtonHandling::HandleRightButtonPressed(pFocusedItem, context);
                if(result.GetCode() != ui::MenuButtonResultCode_NotHandled)
                {
                    return result;
                }
            }

            if(buttonState.isTriggered.Left && !buttonState.isPressed.Right && pFocusedItem)
            {
                ui::MenuButtonHandleContext context(pFocusedItem, pFocusedItem, nullptr, pReactor);
                auto result = ui::util::ProcessButtonHandling::HandleLeftButtonPressed(pFocusedItem, context);
                if(result.GetCode() != ui::MenuButtonResultCode_NotHandled)
                {
                    return result;
                }
            }

            // スティック入力
            if(pFocusedItem)
            {
                ui::MenuButtonHandleContext context(pFocusedItem, pFocusedItem, nullptr, pReactor);
                (void)ui::util::ProcessAnalogStickHandling::HandleLeftAnalogStickInput (pFocusedItem, context, {buttonState.analogSticks.stickL.x, buttonState.analogSticks.stickL.y, 0});
                (void)ui::util::ProcessAnalogStickHandling::HandleRightAnalogStickInput(pFocusedItem, context, {buttonState.analogSticks.stickR.x, buttonState.analogSticks.stickR.y, 0});
            }

            return ui::MenuButtonResult::GetNotHandled();
        }

    }

    ui::MenuButtonResult SceneMenuPageBase::GatherRequestFromUserInputImpl(const std::shared_ptr<MenuPageReactor>& pReactor) NN_NOEXCEPT
    {
        auto touchResult = GatherRequestFromUserInputTouchImpl(pReactor, &m_TouchState, m_pMenuItemRoot, m_pFocusedItem.lock());
        if(touchResult.IsHandled())
        {
            return touchResult;
        }

        auto buttonResult = GatherRequestFromUserInputButtonImpl(pReactor, m_pFocusedItem.lock());
        if(buttonResult.IsHandled())
        {
            return buttonResult;
        }

        return ui::MenuButtonResult::GetNotHandled();
    }

    void SceneMenuPageBase::GatherRequestFromChildPageResultImpl(const std::shared_ptr<MenuPageReactor>& pReactor) NN_NOEXCEPT
    {
        if(auto pChildResult = m_PageActivity.m_pChildParameter)
        {
            if(auto message = pChildResult->GetRequestedMessageToSend())
            {
                pReactor->SendMessage(*message);
            }
            if(pChildResult->IsCloseMenuRequested())
            {
                pReactor->CloseMenu();
            }
        }
        m_PageActivity.m_pChildParameter.reset();
    }

    //---------------------------------------------------------------------------------------

    namespace {

        void ProcessLoadValueRequestImpl(
            const std::shared_ptr<MenuPageReactor>& pReactor,
            const std::shared_ptr<ui::IMenuItem>& pRootItem
        ) NN_NOEXCEPT
        {
            if(pReactor->IsLoadValueRequested())
            {
                ui::util::ProcessLoadStoreValue::Load(pRootItem, pReactor);
            }
        }

        void ProcessStoreValueRequestImpl(
            const std::shared_ptr<MenuPageReactor>& pReactor,
            const std::shared_ptr<ui::IMenuItem>& pRootItem
        ) NN_NOEXCEPT
        {
            if(pReactor->IsStoreValueRequested())
            {
                ui::util::ProcessLoadStoreValue::Store(pRootItem);
            }
        }

        void ProcessEnablityChangeRequestImpl(
            const std::shared_ptr<MenuPageReactor>& pReactor
        ) NN_NOEXCEPT
        {
            int n = pReactor->GetEnablityChangeRequestCount();
            for(int i = 0; i < n; i++)
            {
                auto e = pReactor->GetEnablityChangeRequestEntry(i);
                if(e.first == nullptr)
                {
                    continue;
                }

                e.first->SetEnablity(e.second);
            }
        }

        void ProcessFocusChangeRequestImpl(
            const std::shared_ptr<MenuPageReactor>& pReactor,
            std::weak_ptr<ui::IMenuItem>* ppFocusedItem,
            const std::shared_ptr<ui::IMenuItem>& pRootItem,
            const std::shared_ptr<ui::IMenuItem>& pPrevFocused
        ) NN_NOEXCEPT
        {
            auto request = pReactor->GetFocusChangeRequest();
            if(request)
            {
                auto pNewFocused = *request;

                if(pPrevFocused == pNewFocused)
                {
                    return;
                }

                if(pPrevFocused)
                {
                    pPrevFocused->OnOutFocus();
                    ppFocusedItem->reset();
                }

                if(pNewFocused)
                {
                    if(auto p = ui::util::FindFocusAcceptingItem::Find(pNewFocused, pPrevFocused, ui::util::FindFocusAcceptingItem::Distance))
                    {
                        *ppFocusedItem = p;
                        p->OnInFocus();
                    }
                }

                ui::util::ProcessNotifyFocusedItemChanged::Notify(pRootItem, ui::MenuFocusedItemChangedContext(ppFocusedItem->lock(), pPrevFocused, pReactor));
            }
        }

        void ProcessUpdateLayoutRequestImpl(
            const std::shared_ptr<MenuPageReactor>& pReactor,
            const std::shared_ptr<ui::IMenuItem>& pRootItem,
            const std::shared_ptr<ui::IMenuItem>& pFocusedItem
        ) NN_NOEXCEPT
        {
            if(pReactor->IsUpdateLayoutRequested())
            {
                pRootItem->UpdateLayoutRecursively(ui::MenuLayoutUpdateContext(pFocusedItem));
            }
        }

        void ProcessUpdatePanelRequestImpl(
            const std::shared_ptr<MenuPageReactor>& pReactor,
            const std::shared_ptr<ui::IMenuItem>& pRootItem
        ) NN_NOEXCEPT
        {
            if(pReactor->IsUpdatePanelRequested())
            {
                ui::util::ProcessUpdatePanel::Update(pRootItem);
            }
        }
    }

    void SceneMenuPageBase::ProcessPageRequestImpl(const std::shared_ptr<MenuPageReactor>& pReactor, bool isLoadValueEnabled) NN_NOEXCEPT
    {
        if(isLoadValueEnabled)
        {
            ProcessLoadValueRequestImpl(pReactor, m_pMenuItemRoot);
        }

        ProcessStoreValueRequestImpl(pReactor, m_pMenuItemRoot);

        ProcessEnablityChangeRequestImpl(pReactor);

        ProcessFocusChangeRequestImpl(pReactor, &m_pFocusedItem, m_pMenuItemRoot, m_pFocusedItem.lock());

        ProcessUpdateLayoutRequestImpl(pReactor, m_pMenuItemRoot, m_pFocusedItem.lock());

        ProcessUpdatePanelRequestImpl(pReactor, m_pMenuItemRoot);
    }

    //---------------------------------------------------------------------------------------

    namespace {
        enum TransitionMode
        {
            TransitionMode_NoTransition = 0,
            TransitionMode_ClosePage,
            TransitionMode_OpenChild,
            TransitionMode_PushLayer,
        };

        std::pair<TransitionMode, SceneUpdateResult> ProcessSceneTransitionRequestImpl(
            const std::shared_ptr<MenuPageReactor>& pRequest,
            std::shared_ptr<SceneMenuPageParameter>* ppChildPageParameter,
            const std::shared_ptr<SceneMenuPageBase>& pMyPage,
            const std::shared_ptr<SceneMenuPageParameter>& pMyPageParameter
        ) NN_NOEXCEPT
        {
            // AppletMessage の送信要求を処理
            if(auto request = pRequest->GetAppletResponseRequest())
            {
                if(auto pReporter = pMyPageParameter->GetReporter())
                {
                    auto pResponse = std::static_pointer_cast<AppletMessageMenuResponse>(pReporter->GetUserData());
                    NN_SDK_ASSERT_NOT_NULL(pResponse);
                    *pResponse = *request;
                    pReporter->NotifyCompletion(ThreadMessageResult_Success);
                }
                else
                {
                    pMyPageParameter->RequestSendMessage(*request);
                }
                return std::make_pair(TransitionMode_ClosePage, SceneUpdateResult::GetPopResult());
            }

            // Menu を閉じる
            if(pRequest->IsCloseMenuRequested())
            {
                pMyPageParameter->RequestCloseMenu();
                return std::make_pair(TransitionMode_ClosePage, SceneUpdateResult::GetPopResult());
            }

            // Page を閉じる
            if(pRequest->IsClosePageRequested())
            {
                return std::make_pair(TransitionMode_ClosePage, SceneUpdateResult::GetPopResult());
            }

            // 子ページを開く
            if(auto request = pRequest->GetPageOpenRequest())
            {
                auto pParam = std::make_shared<SceneMenuPageParameter>(pMyPage, request->second);
                *ppChildPageParameter = pParam;
                return std::make_pair(TransitionMode_OpenChild, SceneUpdateResult::GetPushResult(static_cast<scene::SceneIndex>(request->first), std::move(pParam)));
            }

            // レイヤを積む
            if(auto request = pRequest->GetLayerPushRequest())
            {
                auto pParam = std::make_shared<SceneMenuPageParameter>(pMyPage, request->second);
                return std::make_pair(TransitionMode_PushLayer, SceneUpdateResult::GetPushResult(static_cast<scene::SceneIndex>(request->first), std::move(pParam)));
            }

            return std::make_pair(TransitionMode_NoTransition, SceneUpdateResult::GetKeepResult());
        }
    }

    SceneUpdateResult SceneMenuPageBase::ProcessTransitionRequestImpl(const std::shared_ptr<MenuPageReactor>& pReactor, const std::pair<PageState, SceneUpdateResult>& defaultValue) NN_NOEXCEPT
    {
        auto transition = ProcessSceneTransitionRequestImpl(pReactor, &m_PageActivity.m_pChildParameter, this->shared_from_this(), m_pParameter);
        if(transition.first != TransitionMode_NoTransition)
        {
            // Leave
            if(m_PageActivity.m_State != PageState_OutPage &&   // WaitLayer の状態で Close される場合があるので状態をチェック
                ( transition.first == TransitionMode_ClosePage
                || transition.first == TransitionMode_OpenChild
                )
            )
            {
                auto pLeavingReactor = std::make_shared<MenuPageReactor>();

                // タッチ状態をキャンセル
                CancelUserInputTouchImpl();

                // コールバック呼出
                OnLeavingPage(pLeavingReactor);

                ProcessPageRequestImpl(pLeavingReactor, false);
                m_PageActivity.m_State = PageState_OutPage;
            }
            else if(transition.first == TransitionMode_PushLayer)
            {
                // タッチ状態をキャンセル
                CancelUserInputTouchImpl();

                m_PageActivity.m_State = PageState_WaitLayer;
            }

            // Close
            if(transition.first == TransitionMode_ClosePage)
            {
                auto pClosingReactor = std::make_shared<MenuPageReactor>();
                // フォーカスを外す
                pClosingReactor->ChangeFocus(nullptr);

                // コールバックを呼び出す
                OnClosingPage(pClosingReactor);

                ProcessPageRequestImpl(pClosingReactor, false);
                m_PageActivity.m_State = PageState_Closed;
            }

            return transition.second;
        }

        m_PageActivity.m_State = defaultValue.first;
        return defaultValue.second;
    }

    //--------------------------------------------------------------------------

    void SceneMenuPageBase::CancelUserInputTouchImpl() NN_NOEXCEPT
    {
        if(auto pItem = m_TouchState.m_pHeldItem.lock())
        {
            ui::util::ProcessTouchHandling::HandleTouchCanceled(pItem);
        }

        m_TouchState.m_IsTouching = false;
        m_TouchState.m_pHeldItem.reset();
        m_TouchState.m_Position = {};
    }


    //---------------------------------------------------

}}
