﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "debug_SystemSharedTextureViewer.h"

#include <array>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../../framework/Framework.h"

namespace scene{ namespace debug{

    namespace {
        int GetBorderWidth(const ActivitySystemSharedTextureViewer& act) NN_NOEXCEPT
        {
            switch(act.scale)
            {
            case ActivitySystemSharedTextureViewer::Scale_Div4:
                return 4;
            default:
                return 0;
            }
        }

        int GetColumnCount(const ActivitySystemSharedTextureViewer& act) NN_NOEXCEPT
        {
            switch(act.scale)
            {
            case ActivitySystemSharedTextureViewer::Scale_Div4:
                return 4;
            default:
                return 8;
            }
        }

        int GetCellWidth(const ActivitySystemSharedTextureViewer& act) NN_NOEXCEPT
        {
            switch(act.scale)
            {
            case ActivitySystemSharedTextureViewer::Scale_Div4:
                return 315;
            default:
                return 160;
            }
        }

        int GetCellHeight(const ActivitySystemSharedTextureViewer& act) NN_NOEXCEPT
        {
            switch(act.scale)
            {
            case ActivitySystemSharedTextureViewer::Scale_Div4:
                return 177;
            default:
                return 90;
            }
        }

        std::array<int, 8> GetTextureOrder(const ActivitySystemSharedTextureViewer& act) NN_NOEXCEPT
        {
            switch(act.scale)
            {
            case ActivitySystemSharedTextureViewer::Scale_Div4:
                return {{-1, 0, 1, 2, 3, 4, 5, 6}};
            default:
                return {{0, 1, 2, 3, 4, 5, 6, -1}};
            }
        }
    }

    SystemSharedTextureViewer::SystemSharedTextureViewer() NN_NOEXCEPT
    {
        m_pRootPanel = std::make_shared<panel::PanelContainer>();
        m_pRootPanel->SetPanelName("shrtex");

        int n = Framework::GetSharedTextureCount();
        for(int i = 0; i < n; i++)
        {
            auto p = std::make_shared<panel::PanelSharedTexture>();
            p->SetTexture(Framework::AcquireSharedTexture(i));
            m_TextureList.push_back(p);
            m_pRootPanel->AddChild(p);
        }
    }

    void SystemSharedTextureViewer::SetActivity(const ActivitySystemSharedTextureViewer& value) NN_NOEXCEPT
    {
        m_Activity = value;
    }

    void SystemSharedTextureViewer::Update() NN_NOEXCEPT
    {
        m_pRootPanel->SetVisibility(panel::PanelVisibility::Visible);
        m_pRootPanel->SetPosition(m_Activity.common.positionX, m_Activity.common.positionY);
        m_pRootPanel->SetSize(m_Activity.GetWidth(), m_Activity.GetHeight());
        m_pRootPanel->SetColor({0, 0, 0, .5f});

        // レイアウト計算
        int border      = GetBorderWidth(m_Activity);
        int columnCount = GetColumnCount(m_Activity);
        int cellWidth   = GetCellWidth(m_Activity);
        int cellHeight  = GetCellHeight(m_Activity);
        auto textureOrder = GetTextureOrder(m_Activity);

        // 一旦全テクスチャを非表示にする
        // 描画キャッシュが破棄されるが、どうせ再描画するので構わない。
        // （描画キャッシュを残す場合は order に存在するかをチェックする）
        for(auto e : m_TextureList)
        {
            e->SetVisibility(panel::PanelVisibility::Invisible);
        }

        // サイズがおかしければ非表示のままおわり
        if(cellWidth <= 0 || cellHeight <= 0)
        {
            return;
        }

        for(int layoutIdx = 0; layoutIdx < textureOrder.size(); layoutIdx++)
        {
            int texIdx = textureOrder[layoutIdx];
            if(texIdx < 0 || texIdx >= m_TextureList.size())
            {
                continue;
            }

            auto& e = m_TextureList[texIdx];
            int row = layoutIdx / columnCount;
            int col = layoutIdx % columnCount;

            int x = border + (cellWidth  + border) * col;
            int y = border + (cellHeight + border) * row;

            e->SetVisibility(panel::PanelVisibility::Visible);
            e->SetColor({0, 0, 0, 0});
            e->SetPosition(x, y);
            e->SetSize(cellWidth, cellHeight);

            // 外部で更新されるので毎フレーム描画を要求する
            e->RequestRedraw();
        }
    }

    std::shared_ptr<panel::IPanel> SystemSharedTextureViewer::GetPanel() NN_NOEXCEPT
    {
        return m_pRootPanel;
    }

}}
