﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "panel_IPanel.h"
#include "../Config.h"

namespace panel{

    template<typename TBaseType>
    class PanelBase
        : public TBaseType
    {
        NN_DISALLOW_COPY(PanelBase);
    public:
        explicit PanelBase(PanelTypeType type)
            : m_Type(type)
            , m_Visibility(PanelVisibility::Visible)
        {
            m_PanelId  = detail::PanelIdentifier::AcquireNewPanelIdentifier();
            m_Rectangle = {};
            m_Color    = nn::util::Color4f(0, 0, 0, 0);
            m_RecycleCache.Invalidate();
            m_IsRedrawForced = false;
        }

        virtual PanelTypeType GetType() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Type;
        }

        virtual int64_t GetPanelIdentifier() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_PanelId;
        }

        virtual std::string GetPanelName() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_PanelName;
        }

        virtual std::shared_ptr<IPanelContainer> CastToContainer() NN_NOEXCEPT NN_OVERRIDE
        {
            return nullptr;
        }

        virtual nn::util::Vector3f GetPosition() const NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::util::Vector3f(m_Rectangle.x, m_Rectangle.y, 0);
        }

        virtual nn::util::Vector3f GetSize() const NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::util::Vector3f(m_Rectangle.width, m_Rectangle.height, 0);
        }

        virtual nn::util::Color4f GetColor() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Color;
        }

        virtual PanelVisibility GetVisibility() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Visibility;
        }

        virtual std::shared_ptr<IPanelContainer> GetParent() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Parent.lock();
        }

        virtual void RequestRedraw() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_DEVOVL_PANEL_LOG_PANEL("p%llu:RequestRedraw()\n", GetPanelIdentifier());
            InvalidateRecursiveImpl();
        }

        virtual void NotifyParentChanged(const std::shared_ptr<IPanelContainer>& value) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_DEVOVL_PANEL_LOG_PANEL("p%llu:NotifyParentChanged()\n", GetPanelIdentifier());
            m_Parent = value;
            InvalidateRecursiveImpl();
        }

        virtual void NotifyRenderingComplete(bool isForeground, const PanelRectangle& localRect, const PanelRectangle& globalRect) NN_NOEXCEPT NN_OVERRIDE
        {
            if(
                !m_RecycleCache.IsValid()
                || m_RecycleCache.IsForegroud() != isForeground
                || m_RecycleCache.GetLocalRectangle() != localRect
                || m_RecycleCache.GetGlobalRectangle() != globalRect
                )
            {
                NN_DEVOVL_PANEL_LOG_PANEL("p%llu:NotifyRenderingComplete(fg=%s)\n", GetPanelIdentifier(), isForeground ? "y" : "n");
            }
            m_RecycleCache.Validate(isForeground, localRect, globalRect);
        }

        virtual bool CheckRedrawRequired() const NN_NOEXCEPT NN_OVERRIDE
        {
            if(m_IsRedrawForced)
            {
                return true;
            }
            return !m_RecycleCache.IsValid();
        }

        virtual detail::RecycleCache GetRenderingRecycleCache() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_RecycleCache;
        }

    public:
        void SetPanelName(const std::string& value) NN_NOEXCEPT
        {
            m_PanelName = value;
        }

    public:
        void SetPosition(const nn::util::Vector3f& value) NN_NOEXCEPT
        {
            SetPositionImpl(static_cast<int>(value.GetX()), static_cast<int>(value.GetY()));
        }

        void SetPosition(float x, float y, float = 0) NN_NOEXCEPT
        {
            SetPositionImpl(static_cast<int>(x), static_cast<int>(y));
        }

        void SetPosition(int x, int y, int = 0) NN_NOEXCEPT
        {
            SetPositionImpl(x, y);
        }

    private:
        void SetPositionImpl(int x, int y) NN_NOEXCEPT
        {
            if(m_Rectangle.x == x && m_Rectangle.y == y)
            {
                return;
            }

            NN_DEVOVL_PANEL_LOG_PANEL("p%llu:SetPosition(%d,%d)\n", GetPanelIdentifier(), x, y);
            m_Rectangle.x = x;
            m_Rectangle.y = y;
            InvalidateRecursiveImpl(); // 子の位置も変わる
        }

    public:
        void SetSize(const nn::util::Vector3f& value) NN_NOEXCEPT
        {
            SetSizeImpl(static_cast<int>(value.GetX()), static_cast<int>(value.GetY()));
        }

        void SetSize(const nn::util::Float2& value) NN_NOEXCEPT
        {
            SetSizeImpl(static_cast<int>(value.x), static_cast<int>(value.y));
        }

        void SetSize(float w, float h) NN_NOEXCEPT
        {
            SetSizeImpl(static_cast<int>(w), static_cast<int>(h));
        }

        void SetSize(int w, int h) NN_NOEXCEPT
        {
            SetSizeImpl(w, h);
        }

    private:
        void SetSizeImpl(int w, int h) NN_NOEXCEPT
        {
            if(m_Rectangle.width == w && m_Rectangle.height == h)
            {
                return;
            }

            NN_DEVOVL_PANEL_LOG_PANEL("p%llu:SetSize(%d,%d)\n", GetPanelIdentifier(), w, h);
            m_Rectangle.width = w;
            m_Rectangle.height = h;
            InvalidateRecursiveImpl(); // 子の crop 範囲が変わる
        }

    public:
        void SetColor(const nn::util::Color4f& value) NN_NOEXCEPT
        {
            if(m_Color == value)
            {
                return;
            }

            NN_DEVOVL_PANEL_LOG_PANEL("p%llu:SetColor(%f,%f,%f;%f)\n", GetPanelIdentifier(), value.GetR(), value.GetG(), value.GetB(), value.GetA());
            m_Color = value;
            InvalidateSelfImpl();
        }

        void SetVisibility(PanelVisibility value) NN_NOEXCEPT
        {
            if(m_Visibility == value)
            {
                return;
            }

            NN_DEVOVL_PANEL_LOG_PANEL("p%llu:SetVisibility(%s)\n",
                GetPanelIdentifier(),
                value == PanelVisibility::Invisible ? "invisible" : (value == PanelVisibility::Transparent ? "transparent" : "visible")
            );
            m_Visibility = value;
            InvalidateRecursiveImpl(); // 子の可視状態も変わる
        }

    protected:
        void InvalidateSelfImpl() NN_NOEXCEPT
        {
            m_RecycleCache.Invalidate();
        }

        virtual void InvalidateRecursiveImpl() NN_NOEXCEPT
        {
            m_RecycleCache.Invalidate();
        }

        void SetForceRedrawImpl(bool value) NN_NOEXCEPT
        {
            m_IsRedrawForced = value;
        }

    private:
        PanelTypeType      m_Type;
        int64_t            m_PanelId;
        std::string        m_PanelName;
        PanelRectangle     m_Rectangle;
        nn::util::Color4f  m_Color;
        PanelVisibility    m_Visibility;
        std::weak_ptr<IPanelContainer> m_Parent;

        detail::RecycleCache m_RecycleCache;
        bool m_IsRedrawForced;
    };

}
