﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <string>
#include <nn/nn_Common.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Color.h>
#include "panel_PanelType.h"
#include "panel_PanelRectangle.h"
#include "panel_PanelVisibility.h"

#include "detail/panel_RecycleCache.h"

#define PANEL_CAST_FUNCTION(PanelType)   \
    static std::shared_ptr<Panel##PanelType> Is##PanelType(std::shared_ptr<IPanel> value) NN_NOEXCEPT   \
    {                                                   \
        if(value->GetType() == PanelType_##PanelType)   \
        {                                               \
            return std::static_pointer_cast<Panel##PanelType>(value);   \
        }                                               \
        else                                            \
        {                                               \
            return nullptr;                             \
        }                                               \
    }

namespace panel{
    class IPanelContainer;

    // 描画用オブジェクト
    class IPanel
    {
    public:
        virtual ~IPanel() NN_NOEXCEPT
        {
        }

        // Panel の種類を取得する。
        virtual PanelTypeType GetType() const NN_NOEXCEPT = 0;

        // このパネルの識別番号を取得する。
        virtual int64_t GetPanelIdentifier() const NN_NOEXCEPT = 0;

        // このパネルの名称を取得する。
        virtual std::string GetPanelName() const NN_NOEXCEPT = 0;

        // 自身の IPanelContainer としてのポインタを取得する。コンテナでなければ nullptr を返す。
        virtual std::shared_ptr<IPanelContainer> CastToContainer() NN_NOEXCEPT = 0;

        // ローカル座標系における Panel の左上座標を取得する。 z 成分は 0。
        virtual nn::util::Vector3f GetPosition() const NN_NOEXCEPT = 0;

        // Panel の幅と高さを取得する。 z 成分は 0。
        virtual nn::util::Vector3f GetSize() const NN_NOEXCEPT = 0;

        // Panel の背景色を取得する。
        virtual nn::util::Color4f GetColor() const NN_NOEXCEPT = 0;

        // Panel の可視性を取得する。
        virtual PanelVisibility GetVisibility() const NN_NOEXCEPT = 0;

        // 親 Panel を取得する。このパネルが根の場合 nullptr を返す。
        virtual std::shared_ptr<IPanelContainer> GetParent() const NN_NOEXCEPT = 0;

        // このパネルと子パネルの再描画を要求する。
        virtual void RequestRedraw() NN_NOEXCEPT = 0;

        // このパネルの再描画が必要か取得する。
        virtual bool CheckRedrawRequired() const NN_NOEXCEPT = 0;

        // 親 Panel が変更されたことを通知する。
        // この関数は PanelContainer が呼び出す。
        // ツリー構造を編集する場合にはこの関数を直接呼び出さずに GetParent()->RemoveChild() を使用すること。
        virtual void NotifyParentChanged(const std::shared_ptr<IPanelContainer>& value) NN_NOEXCEPT = 0;

    //@name Renderer 用
    //@{
        // 正常に描画完了したことを通知する。
        // この関数は Renderer が呼び出す。
        virtual void NotifyRenderingComplete(bool isForeground, const PanelRectangle& localRectangle, const PanelRectangle& globalRectangle) NN_NOEXCEPT = 0;

        virtual detail::RecycleCache GetRenderingRecycleCache() const NN_NOEXCEPT = 0;

    //@}

    };

    class IPanelContainer
        : public IPanel
    {
    public:
        // 子の数を返す。
        virtual int GetChildrenCount() const NN_NOEXCEPT = 0;

        // 子のインデックスを取得する。 child がこのパネルの子でない場合、負値を返す。
        virtual int GetChildIndex(const std::shared_ptr<IPanel>& child) const NN_NOEXCEPT = 0;

        // 指定したインデックスの子を取得する。 index が範囲外の場合 nullptr を返す。
        virtual std::shared_ptr<IPanel> GetChild(int index) NN_NOEXCEPT = 0;

        // すべての子を削除する。
        virtual void ClearChildren() NN_NOEXCEPT = 0;

        // 子を設定する。
        virtual void SetChildren(const std::vector<std::shared_ptr<IPanel>>& list) NN_NOEXCEPT = 0;

        // 子を追加する。子は末尾に追加される。
        // @pre value->GetParent() == nullptr
        virtual void AddChild(const std::shared_ptr<IPanel>& value) NN_NOEXCEPT = 0;

        // 子を削除する。
        virtual void RemoveChild(const std::shared_ptr<IPanel>& value) NN_NOEXCEPT = 0;

    };


    namespace detail{

        class PanelIdentifier
        {
        public:
            static int64_t AcquireNewPanelIdentifier() NN_NOEXCEPT
            {
                static std::atomic_int64_t s_NextValue;
                return s_NextValue++;
            }
        };

    }
}
