﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Framework.h"
#include "FrameworkImpl.h"

#include <memory>
#include <vector>
#include <array>
#include <nn/nn_Assert.h>
#include <nn/pl.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_CharacterEncoding.h>

#include "gfx_GraphicsFramework.h"
#include "Audio.h"
#include "Hid.h"

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

namespace framework{
    nns::gfx::GraphicsFramework g_Gfw;
    FrameworkImpl g_Impl;
}

namespace {
    using namespace ::framework;
    static const int CommandBufferCount = 1;
    static const int FrameBufferCount = 2;
    static const int Width = 1280;
    static const int Height = 720;

    // フレームバッファを Width x Height よりも大きく作るピクセル数。横方向。
    static const int FrameBufferMarginX = 0;
    // フレームバッファを Width x Height よりも大きく作るピクセル数。縦方向。
    // 幅 1280px なら高さ 742 (= 720 + 22) pxまでは使用メモリ量が変わらないはず。
    static const int FrameBufferMarginY = 8;

    static const size_t GraphicsDonateMemorySize     =   3 * 256 * 1024;
    static const size_t RootCommandMemorySize        =       256 * 1024;
    static const size_t RootControlMemorySize        =         2 * 1024;

    static const size_t TextureMemoryPoolSize        =  8 * 1024 * 1024;
    static const size_t CommandBufferMemoryPoolSize  =       256 * 1024;
    static const size_t ConstantBufferMemoryPoolSize =  1 * 1024 * 1024;
    static const size_t ShaderMemoryPoolSize         =        16 * 1024;
    static const size_t DataMemoryPoolSize           =                0;

    static const int PoolTextureWidth = 400;
    static const int PoolTextureHeight = 72;
    static const int PoolTextureCount  = 2;

    // LogViewer に表示する 1 行あたりの最大ログ文字数は i(4px) を 227文字
    static const int TextWriterCharCountMax  = 256;
    static const int TextWriterInstanceCount = 20;


    class MemoryPoolManager
    {
    public:
        explicit MemoryPoolManager(nns::gfx::GraphicsFramework::MemoryPoolType poolType, size_t poolSize) NN_NOEXCEPT
            : m_PoolType(poolType)
            , m_PoolSize(poolSize)
            , m_TotalAllocatedSize(0)
            , m_TotalAllocatedSizeMax(0)
        {
        }

        void UpdateAllocatedSize() NN_NOEXCEPT
        {
            if(m_PoolSize > 0)
            {
                auto pAllocator = g_Gfw.GetMemoryPoolAllocator(m_PoolType);
                m_TotalAllocatedSize = m_PoolSize - pAllocator->GetTotalFreeSize();
                if(m_TotalAllocatedSize > m_TotalAllocatedSizeMax)
                {
                    m_TotalAllocatedSizeMax = m_TotalAllocatedSize;
                }
            }
        }

        ptrdiff_t Allocate(size_t size, size_t alignment) NN_NOEXCEPT
        {
            auto offset = g_Gfw.AllocatePoolMemory(m_PoolType, size, alignment);
            if(offset < 0)
            {
                return -1;
            }

            UpdateAllocatedSize();
            return offset;
        }

        void Free(ptrdiff_t offset) NN_NOEXCEPT
        {
            g_Gfw.FreePoolMemory(m_PoolType, offset);

            UpdateAllocatedSize();
        }

        nn::gfx::MemoryPool* GetPool() NN_NOEXCEPT
        {
            return g_Gfw.GetMemoryPool(m_PoolType);
        }

        size_t GetTotalAllocatedSize() const NN_NOEXCEPT
        {
            return m_TotalAllocatedSize;
        }

        size_t GetTotalAllocatedSizeMax() const NN_NOEXCEPT
        {
            return m_TotalAllocatedSizeMax;
        }
    private:
        nns::gfx::GraphicsFramework::MemoryPoolType m_PoolType;
        size_t m_PoolSize;
        size_t m_TotalAllocatedSize;
        size_t m_TotalAllocatedSizeMax;
    };


    MemoryPoolManager g_TextureMemoryPoolManager        = MemoryPoolManager(nns::gfx::MemoryPoolType_RenderTarget, TextureMemoryPoolSize);
    MemoryPoolManager g_CommandBufferMemoryPoolManager  = MemoryPoolManager(nns::gfx::MemoryPoolType_CommandBuffer, CommandBufferMemoryPoolSize);
    MemoryPoolManager g_ConstantBufferMemoryPoolManager = MemoryPoolManager(nns::gfx::MemoryPoolType_ConstantBuffer, ConstantBufferMemoryPoolSize);
    MemoryPoolManager g_ShaderMemoryPoolManager         = MemoryPoolManager(nns::gfx::MemoryPoolType_Shader, ShaderMemoryPoolSize);
    MemoryPoolManager g_DataMemoryPoolManager           = MemoryPoolManager(nns::gfx::MemoryPoolType_Data, DataMemoryPoolSize);

    int g_SamplerDescriptorSlotIndex;
    nn::gfx::DescriptorSlot g_DefaultSamplerDescriptorSlot;
}


void Framework::Initialize() NN_NOEXCEPT
{
    {
        nns::gfx::GraphicsFramework::FrameworkInfo info;
        //info.SetDebugMode(nn::gfx::DebugMode_Enable);
        info.SetBufferCount(CommandBufferCount);
        info.SetColorBufferFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        info.SetDisplayWidth(Width + FrameBufferMarginX);
        info.SetDisplayHeight(Height + FrameBufferMarginY);
        info.SetSwapChainBufferCount(FrameBufferCount);
        info.SetSwapChainFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
        info.SetRootCommandBufferCommandMemorySize(RootCommandMemorySize);
        info.SetRootCommandBufferControlMemorySize(RootControlMemorySize);
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_RenderTarget, TextureMemoryPoolSize);
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_CommandBuffer, CommandBufferMemoryPoolSize);
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_ConstantBuffer, ConstantBufferMemoryPoolSize);
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_Shader, ShaderMemoryPoolSize);
        info.SetMemoryPoolSize(nns::gfx::MemoryPoolType_Data, DataMemoryPoolSize);
        g_Gfw.InitializeGraphicsSystem(GraphicsDonateMemorySize);
        g_Gfw.Initialize(info);
    }
    SetWindowCrop(GetScreenRectangle());

    {
        auto pPool = g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler);
        pPool->BeginUpdate();
        {
            g_SamplerDescriptorSlotIndex = g_Gfw.GetDescriptorPoolAllocator(nn::gfx::DescriptorPoolType_Sampler)->Allocate(1);
            NN_ASSERT_NOT_EQUAL(g_SamplerDescriptorSlotIndex, -1);
            int index = g_SamplerDescriptorSlotIndex;
            pPool->SetSampler(index, g_Gfw.GetSampler(nns::gfx::SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Border));
            pPool->GetDescriptorSlot(&g_DefaultSamplerDescriptorSlot, index);
        }
        pPool->EndUpdate();
    }
    {
        g_TextureMemoryPoolManager.UpdateAllocatedSize();
        g_CommandBufferMemoryPoolManager.UpdateAllocatedSize();
        g_ConstantBufferMemoryPoolManager.UpdateAllocatedSize();
        g_ShaderMemoryPoolManager.UpdateAllocatedSize();
        g_DataMemoryPoolManager.UpdateAllocatedSize();
    }

    // Hid を初期化
    Hid::Initialize();

    // Audio を初期化
    Audio::Initialize();

    // TexturePool を初期化
    {
        g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->BeginUpdate();
        NN_UTIL_SCOPE_EXIT{
            g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->EndUpdate();
        };

        framework::TexturePoolInfo info;
        info.textureWidth  = PoolTextureWidth;
        info.textureHeight = PoolTextureHeight;
        info.textureFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
        info.textureCount  = PoolTextureCount;

        g_Impl.texturePool.Initialize(info);
    }

    // TextWriterPool を初期化
    {
        g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->BeginUpdate();
        g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler)->BeginUpdate();
        NN_UTIL_SCOPE_EXIT{
            g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->EndUpdate();
            g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler)->EndUpdate();
        };
        // 共有フォントを取得
        auto loadSharedFont = [](nn::pl::SharedFontType type)
        {
            nn::pl::RequestSharedFontLoad(type);
            while(nn::pl::GetSharedFontLoadState(type) != nn::pl::SharedFontLoadState_Loaded)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            }
        };
        loadSharedFont(nn::pl::SharedFontType_Standard);            // 日米欧
        loadSharedFont(nn::pl::SharedFontType_NintendoExtension);   // 任天堂外字(http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=204994761)

        framework::TextWriterPoolInfo info;
        info.charCountMax = TextWriterCharCountMax;
        info.writerCount = TextWriterInstanceCount;
        info.pFontDataList   [0] = nn::pl::GetSharedFontAddress(nn::pl::SharedFontType_Standard);
        info.fontDataSizeList[0] = nn::pl::GetSharedFontSize   (nn::pl::SharedFontType_Standard);
        info.pFontDataList   [1] = nn::pl::GetSharedFontAddress(nn::pl::SharedFontType_NintendoExtension);
        info.fontDataSizeList[1] = nn::pl::GetSharedFontSize   (nn::pl::SharedFontType_NintendoExtension);
        NN_SDK_ASSERT_NOT_NULL(info.pFontDataList[0]);
        NN_SDK_ASSERT_GREATER(info.fontDataSizeList[0], 0u);
        NN_SDK_ASSERT_NOT_NULL(info.pFontDataList[1]);
        NN_SDK_ASSERT_GREATER(info.fontDataSizeList[1], 0u);

        g_Impl.textWriterPool.Initialize(info);
    }

#ifdef NN_DEVOVL_IS_SYSTEM
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    // SharedTexturePool を初期化
    {
        g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->BeginUpdate();
        NN_UTIL_SCOPE_EXIT{
            g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->EndUpdate();
        };
        g_Impl.sharedTexturePool.Initialize();
    }
#endif

    // 描画履歴を初期化
    {
        g_Impl.renderedRegion.resize(FrameBufferCount);
        for(auto& e : g_Impl.renderedRegion)
        {
            e.clearColor = nn::util::Color4f(0, 0, 0, 0);
        }
    }

    g_Impl.rootCommandMemoryUsedSizeMax = 0;
    g_Impl.rootControlMemoryUsedSizeMax = 0;
}

void Framework::Finalize() NN_NOEXCEPT
{
    Audio::Finalize();
    Hid::Finalize();

#ifdef NN_DEVOVL_IS_SYSTEM
    //g_Impl.sharedTexturePool.Finalize();
#endif

    g_Impl.textWriterPool.Finalize();
    g_Impl.texturePool.Finalize();
    g_Impl.lockList.clear();
    {
        g_Gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, g_SamplerDescriptorSlotIndex);
    }
    g_Gfw.Finalize();
}


void Framework::MakeInitializeCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor);
    //pCommandBuffer->SetDescriptorPool(g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_BufferView));
    pCommandBuffer->SetDescriptorPool(g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView));
    pCommandBuffer->SetDescriptorPool(g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler));

    // フレームバッファのクリア
    {
        nn::gfx::ColorTargetView* viewList[FrameBufferCount];
        int n = g_Gfw.GetSwapChain()->GetScanBufferViews(&viewList[0], FrameBufferCount);
        for(int i = 0; i < n; i++)
        {
            nn::gfx::ColorTargetView* pView = viewList[i];
            pCommandBuffer->SetRenderTargets(1, &pView, nullptr);
            MakeSetViewportScissorCommand(pCommandBuffer, GetExtendedScreenRectangle());
            MakeClearCommand(pCommandBuffer, nn::util::Color4f(0, 0, 0, 0));
        }
    }
}

//-----------------------------------------

framework::Rectangle Framework::GetScreenRectangle() NN_NOEXCEPT
{
    return Rectangle(0, 0, Width, Height);
}

framework::Rectangle Framework::GetExtendedScreenRectangle() NN_NOEXCEPT
{
    return Rectangle(0, 0, Width + FrameBufferMarginX, Height + FrameBufferMarginY);
}

framework::Rectangle Framework::GetIdleCropRectangle() NN_NOEXCEPT
{
    NN_STATIC_ASSERT(FrameBufferMarginY >= 2);
    return Rectangle(0, Height, 2, 2);
}

//-----------------------------------------

void Framework::MakeSetViewportCommand(nn::gfx::CommandBuffer* pCommandBuffer, const Rectangle& rect) NN_NOEXCEPT
{
    nn::gfx::ViewportStateInfo vinfo;
    vinfo.SetDefault();
    vinfo.SetOriginX(static_cast<float>(rect.x));
    vinfo.SetOriginY(static_cast<float>(rect.y));
    vinfo.SetWidth(static_cast<float>(rect.width));
    vinfo.SetHeight(static_cast<float>(rect.height));
    pCommandBuffer->SetViewports(0, 1, &vinfo);
}

void Framework::MakeSetScissorCommand(nn::gfx::CommandBuffer* pCommandBuffer, const Rectangle& rect) NN_NOEXCEPT
{
    nn::gfx::ScissorStateInfo sinfo;
    sinfo.SetDefault();
    sinfo.SetOriginX(rect.x);
    sinfo.SetOriginY(rect.y);
    sinfo.SetWidth(rect.width);
    sinfo.SetHeight(rect.height);
    pCommandBuffer->SetScissors(0, 1, &sinfo);
}

void Framework::MakeSetViewportScissorCommand(nn::gfx::CommandBuffer* pCommandBuffer, const Rectangle& rect) NN_NOEXCEPT
{
    Framework::MakeSetViewportCommand(pCommandBuffer, rect);
    Framework::MakeSetScissorCommand(pCommandBuffer, rect);
}

void Framework::MakeClearCommand(nn::gfx::CommandBuffer* pCommandBuffer, const nn::util::Color4f& color) NN_NOEXCEPT
{
    float c[4];
    std::memcpy(c, &color, 4 * sizeof(float));
    nvnCommandBufferClearColor(
        pCommandBuffer->ToData()->pNvnCommandBuffer,
        0, /* colorTargetIndex */
        c,
        NVN_CLEAR_COLOR_MASK_RGBA
    );
}

//-----------------------------------------

nn::gfx::Device* Framework::GetDevice() NN_NOEXCEPT
{
    return g_Gfw.GetDevice();
}

int Framework::GetCurrentFrameCount() NN_NOEXCEPT
{
    return g_Impl.frameCount;
}

nn::gfx::ColorTargetView* Framework::GetCurrentColorTargetView() NN_NOEXCEPT
{
    return g_Gfw.GetColorTargetView();
}

nn::gfx::Texture* Framework::GetCurrentTargetTexture() NN_NOEXCEPT
{
    nn::gfx::Texture* pTextureList[FrameBufferCount] = {};
    auto n = g_Gfw.GetSwapChain()->GetScanBuffers(pTextureList, FrameBufferCount);
    NN_UNUSED(n);

    return pTextureList[g_Gfw.GetSwapChain()->ToData()->currentScanBufferIndex];
}

nn::gfx::ViewportScissorState* Framework::GetCurrentViewportScissorState() NN_NOEXCEPT
{
    return g_Gfw.GetViewportScissorState();
}

size_t Framework::GetFirmwareMemorySize() NN_NOEXCEPT
{
    return GraphicsDonateMemorySize;
}

nn::gfx::CommandBuffer* Framework::GetCommandBuffer() NN_NOEXCEPT
{
    return g_Gfw.GetRootCommandBuffer(0);
}

size_t Framework::GetRootCommandMemorySize() NN_NOEXCEPT
{
    return RootCommandMemorySize;
}

size_t Framework::GetRootControlMemorySize() NN_NOEXCEPT
{
    return RootControlMemorySize;
}

size_t Framework::GetRootCommandMemoryUsedSizeMax() NN_NOEXCEPT
{
    return g_Impl.rootCommandMemoryUsedSizeMax;
}

size_t Framework::GetRootControlMemoryUsedSizeMax() NN_NOEXCEPT
{
    return g_Impl.rootControlMemoryUsedSizeMax;
}

const nn::gfx::DescriptorSlot& Framework::GetDefaultSamplerDescriptorSlot() NN_NOEXCEPT
{
    return g_DefaultSamplerDescriptorSlot;
}

//-----------------------------------------

void* Framework::Allocate(size_t size, size_t alignment) NN_NOEXCEPT
{
    auto p = g_Gfw.AllocateMemory(size, alignment);
    NN_SDK_ASSERT_NOT_NULL(p);
    return p;
}

void Framework::Free(void* p) NN_NOEXCEPT
{
    g_Gfw.FreeMemory(p);
}

//-----------------------------------------

size_t Framework::GetTexturePoolMemorySize() NN_NOEXCEPT
{
    return TextureMemoryPoolSize;
}

size_t Framework::GetCommandBufferPoolMemorySize() NN_NOEXCEPT
{
    return CommandBufferMemoryPoolSize;
}

size_t Framework::GetConstantBufferPoolMemorySize() NN_NOEXCEPT
{
    return ConstantBufferMemoryPoolSize;
}

size_t Framework::GetShaderPoolMemorySize() NN_NOEXCEPT
{
    return ShaderMemoryPoolSize;
}

size_t Framework::GetDataPoolMemorySize() NN_NOEXCEPT
{
    return DataMemoryPoolSize;
}

#define FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(Type) \
    nn::gfx::MemoryPool* Framework::Get##Type##MemoryPool() NN_NOEXCEPT         \
    {                                                                           \
        return g_##Type##MemoryPoolManager.GetPool();                           \
    }                                                                           \
    ptrdiff_t Framework::Allocate##Type##PoolMemory(size_t size, size_t alignment) NN_NOEXCEPT  \
    {                                                                           \
        auto offset = g_##Type##MemoryPoolManager.Allocate(size, alignment);    \
        NN_ABORT_UNLESS_GREATER_EQUAL(offset, 0);                                     \
        return offset;                                                          \
    }                                                                           \
    ptrdiff_t Framework::Allocate##Type##PoolMemoryNoCheck(size_t size, size_t alignment) NN_NOEXCEPT  \
    {                                                                           \
        auto offset = g_##Type##MemoryPoolManager.Allocate(size, alignment);    \
        return offset;                                                          \
    }                                                                           \
    void Framework::Free##Type##PoolMemory(ptrdiff_t offset) NN_NOEXCEPT        \
    {                                                                           \
        g_##Type##MemoryPoolManager.Free(offset);                               \
    }                                                                           \
    size_t Framework::GetTotalAllocated##Type##PoolMemorySize() NN_NOEXCEPT     \
    {                                                                           \
        return g_##Type##MemoryPoolManager.GetTotalAllocatedSize();             \
    }                                                                           \
    size_t Framework::GetTotalAllocated##Type##PoolMemorySizeMax() NN_NOEXCEPT  \
    {                                                                           \
        return g_##Type##MemoryPoolManager.GetTotalAllocatedSizeMax();          \
    }

FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(Texture);
FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(CommandBuffer);
FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(ConstantBuffer);
FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(Shader);
FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS(Data);

#undef FRAMEWORK_DEFINE_MEMORYPOOL_FUNCTIONS

//-----------------------------------------

nn::gfx::DescriptorPool* Framework::GetTextureDescriptorPool() NN_NOEXCEPT
{
    return g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView);
}

int Framework::AllocateTextureDescriptorSlot() NN_NOEXCEPT
{
    auto index = g_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    NN_ABORT_UNLESS_NOT_EQUAL(index, -1);
    return index;
}

int Framework::AllocateTextureDescriptorSlotNoCheck() NN_NOEXCEPT
{
    auto index = g_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    return index;
}

void Framework::FreeTextureDescriptorSlot(int index) NN_NOEXCEPT
{
    g_Gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, index);
}

nn::gfx::DescriptorPool* Framework::GetSamplerDescriptorPool() NN_NOEXCEPT
{
    return g_Gfw.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler);
}

int Framework::AllocateSamplerDescriptorSlot() NN_NOEXCEPT
{
    auto index = g_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
    NN_ABORT_UNLESS_NOT_EQUAL(index, -1);
    return index;
}

int Framework::AllocateSamplerDescriptorSlotNoCheck() NN_NOEXCEPT
{
    auto index = g_Gfw.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
    return index;
}

void Framework::FreeSamplerDescriptorSlot(int index) NN_NOEXCEPT
{
    g_Gfw.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, index);
}

//--------------------------------------

std::shared_ptr<framework::PoolTexture> Framework::AcquirePoolTexture(int width, int height, nn::gfx::ImageFormat format) NN_NOEXCEPT
{
    return g_Impl.texturePool.Acquire(width, height, format);
}

std::shared_ptr<framework::PoolTextWriterHolder> Framework::AcquirePoolTextWriter() NN_NOEXCEPT
{
    return g_Impl.textWriterPool.Acquire();
}

nn::font::ScalableFont* Framework::GetTextWriterScalableFont() NN_NOEXCEPT
{
    return g_Impl.textWriterPool.GetScalableFont(TextWriterUsage_Text);
}

void Framework::CalculateTextRenderingSize(int* pOutWidth, int* pOutHeight, const std::string& text, float fontSize, framework::TextWriterUsage usage) NN_NOEXCEPT
{
    std::vector<uint32_t> text32;
    text32.resize(text.size() + 1);
    auto result = nn::util::ConvertStringUtf8ToUtf32(text32.data(), static_cast<int>(text32.size()), text.c_str());
    if(result != nn::util::CharacterEncodingResult_Success)
    {
        NN_DEVOVL_LOG_ERR("[fw] CalculateTextRenderingSize failed -> result=%d\n", result);
        *pOutWidth = 0;
        *pOutHeight = 0;
        return;
    }

    auto pScalableFont = g_Impl.textWriterPool.GetScalableFont(usage);
    const float scale = fontSize / pScalableFont->GetFontSize();

    float widthMax = 0;
    float lineWidth = 0;
    int   lineCount = 1;
    for(auto it = text32.begin(); it != text32.end(); it++)
    {
        auto c = *it;

        // 終端文字で終わり
        if(c == 0)
        {
            break;
        }

        // 改行文字で次の行へ
        if(c == '\n')
        {
            widthMax = std::max<float>(widthMax, lineWidth);
            lineCount++;
            lineWidth = 0;
            continue;
        }

        auto w = pScalableFont->GetCharWidths(c);
        lineWidth += scale * w.charWidth;
    }

    widthMax = std::max<float>(widthMax, lineWidth);

    int lineHeight = static_cast<int>(std::ceilf(fontSize * pScalableFont->GetLineFeed() / pScalableFont->GetFontSize()));

    *pOutWidth = static_cast<int>(std::ceilf(widthMax));
    *pOutHeight = lineHeight * lineCount;
}

//--------------------------------------

int Framework::GetSharedTextureCount() NN_NOEXCEPT
{
    return g_Impl.sharedTexturePool.GetTextureCount();
}

std::shared_ptr<framework::SharedTexture> Framework::AcquireSharedTexture(int index) NN_NOEXCEPT
{
    return g_Impl.sharedTexturePool.GetTexture(index);
}

//--------------------------------------

