﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nn/nn_SdkAssert.h>
#include <nn/audio.h>
#include "Audio.h"

namespace {
nn::audio::AudioOut g_AudioOut;
nn::audio::AudioOutBuffer g_AudioOutBuffer;
NN_AUDIO_ALIGNAS_AUDIO_OUT_BUFFER_ALIGN uint8_t g_WaveDataBufferSilence[nn::audio::AudioOutBuffer::SizeGranularity];
NN_AUDIO_ALIGNAS_AUDIO_OUT_BUFFER_ALIGN uint8_t g_WaveDataBufferSine[nn::audio::AudioOutBuffer::SizeGranularity];
} // namespace

void Audio::Initialize() NN_NOEXCEPT
{
    // デフォルトの出力先を指定して AudioOut をオープンしスタートする
    const int channelCount = 2;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    parameter.channelCount = channelCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&g_AudioOut, parameter));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioOut(&g_AudioOut));

    // サイン波のデータを生成
    const int frequency = 440;
    const int sampleRate = nn::audio::GetAudioOutSampleRate(&g_AudioOut);
    const float Pi = 3.1415926535897932384626433f;
    int16_t* p = reinterpret_cast<int16_t*>(g_WaveDataBufferSine);
    NN_SDK_ASSERT(nn::audio::GetAudioOutSampleFormat(&g_AudioOut) == nn::audio::SampleFormat_PcmInt16);
    for(int i = 0; i < sizeof(g_WaveDataBufferSine) / channelCount / sizeof(int16_t); ++i)
    {
        const int16_t sample = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
        *p++ = sample;
        *p++ = sample;
    }
}

void Audio::Finalize() NN_NOEXCEPT
{
    // AudioOut をストップしてクローズする
    nn::audio::StopAudioOut(&g_AudioOut);
    nn::audio::CloseAudioOut(&g_AudioOut);
}

void Audio::Playback(SeId id) NN_NOEXCEPT
{
    // リリース済み AudioOutBuffer があったら取り出しておく
    nn::audio::GetReleasedAudioOutBuffer(&g_AudioOut);

    // AudioOutBuffer がまだ再生中だった場合は再生しない
    if(nn::audio::ContainsAudioOutBuffer(&g_AudioOut, &g_AudioOutBuffer))
    {
        return;
    }

    // AudioOutBuffer を準備
    switch(id)
    {
    case SeId::Silence:
        nn::audio::SetAudioOutBufferInfo(&g_AudioOutBuffer, g_WaveDataBufferSilence, sizeof(g_WaveDataBufferSilence), sizeof(g_WaveDataBufferSilence));
        break;
    case SeId::Sine:
        nn::audio::SetAudioOutBufferInfo(&g_AudioOutBuffer, g_WaveDataBufferSine, sizeof(g_WaveDataBufferSine), sizeof(g_WaveDataBufferSine));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // AudioOutBuffer を AudioOut に登録
    nn::audio::AppendAudioOutBuffer(&g_AudioOut, &g_AudioOutBuffer);
}
