﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ResourceUsage.h"

#include "Config.h"
#include "MemoryManagement.h"
#include "framework/Framework.h"

#include <nv/nv_MemoryManagement.h>

#define NN_DEVOVL_LOG_MEMORY(...) NN_DEVOVL_LOG("[memory]" __VA_ARGS__)

#ifndef NN_DEVOVL_LOG_MEMORY
#define NN_DEVOVL_LOG_MEMORY(...)
#endif

namespace
{
    int GetPercent(int64_t used, int64_t total) NN_NOEXCEPT
    {
        if(total <= 0)
        {
            return 0;
        }
        return static_cast<int>(static_cast<float>(used) / static_cast<float>(total) * 100.f);
    }

    void PrintUsage(const char* name, int64_t used, int64_t total) NN_NOEXCEPT
    {
        int percent = GetPercent(used, total);
        NN_DEVOVL_LOG_MEMORY("%s\n", name);
        NN_DEVOVL_LOG_MEMORY("  %lld / %lld (%d%%)\n", used, total, GetPercent(used, total));
        NN_UNUSED(name);
        NN_UNUSED(percent);
    }
}

void DumpResourceUsage() NN_NOEXCEPT
{
    {
        int64_t used  = GetTotalAllocatedSizeMax();
        int64_t total = GetApplicationHeapSize();
        PrintUsage("ApplicationHeap", used, total);
    }
    NN_DEVOVL_LOG_MEMORY("----------------------\n");
    {
        int64_t used  = Framework::GetTotalAllocatedTexturePoolMemorySizeMax();
        int64_t total = Framework::GetTexturePoolMemorySize();
        PrintUsage("MemoryPool(Texture)", used, total);
    }
    {
        int64_t used  = Framework::GetTotalAllocatedCommandBufferPoolMemorySizeMax();
        int64_t total = Framework::GetCommandBufferPoolMemorySize();
        PrintUsage("MemoryPool(CommandBuffer)", used, total);
    }
    {
        int64_t used  = Framework::GetTotalAllocatedConstantBufferPoolMemorySizeMax();
        int64_t total = Framework::GetConstantBufferPoolMemorySize();
        PrintUsage("MemoryPool(ConstantBuffer)", used, total);
    }
    {
        int64_t used  = Framework::GetTotalAllocatedShaderPoolMemorySizeMax();
        int64_t total = Framework::GetShaderPoolMemorySize();
        PrintUsage("MemoryPool(Shader)", used, total);
    }
    {
        int64_t used  = Framework::GetTotalAllocatedDataPoolMemorySizeMax();
        int64_t total = Framework::GetDataPoolMemorySize();
        PrintUsage("MemoryPool(Data)", used, total);
    }
    NN_DEVOVL_LOG_MEMORY("----------------------\n");
    {
        int64_t used  = Framework::GetRootCommandMemoryUsedSizeMax();
        int64_t total = Framework::GetRootCommandMemorySize();
        PrintUsage("RootCommandMemory", used, total);
    }
    {
        int64_t used  = Framework::GetRootControlMemoryUsedSizeMax();
        int64_t total = Framework::GetRootControlMemorySize();
        PrintUsage("RootControlMemory", used, total);
    }
    NN_DEVOVL_LOG_MEMORY("----------------------\n");
    {
        nv::MemoryInfo info;
        nv::GetGraphicsMemoryInfo(&info);
        int64_t total = Framework::GetFirmwareMemorySize();
        int64_t used  = total - info.firmwareMemoryMinimumFreeSize;
        PrintUsage("FirmwareMemory", used, total);
    }


}
