﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * インクルード
 */
#include <cstring>

#include "QREL.h"

/*
 * マクロ定義
 */
/* #define DEBUG_LOG        デバッグログ出力用マクロ */

/*
 * 内部関数定義
 */
static bool SetRsBlock(RSBLOCK *rsblock, ERRCORUNIT *rsinfo, bool torsblock);
static bool MemoryAllocate(SINGLERSBLOCK *rsb,
                           int32_t nblocks, int32_t num, int32_t codelen, int32_t datacodelen);
static bool CreateRSBlock(ERRCORUNIT *rsinfo, RSBLOCK *rsblock,
                           uint8_t *code);
static bool DestroyRSBlock(ERRCORUNIT *rsinfo, RSBLOCK *rsblock);
static bool MemoryFree(SINGLERSBLOCK *rsblock);

/****************************************************************
 * 外部関数
 ****************************************************************/
/*
 * 関数名 QREL_AddErrorCorrectData
 * 機能   データコード語に誤り訂正コード語を付加し最終コード語とする
 * 引数   versioninfo(IN)        型番情報
 *        ecclevel(IN)           誤り訂正レベル
 *        mode(IN)               モード指示子へのポインタ
 *        ncodes(IN)             データコード語数
 *        code(IN)               データコード語へのポインタ
 * 戻り値 正常終了時0以外を異常終了時0を返す
 *          異常終了時のエラーコードはQREL_GetLastError関数で取得できる。
 *          エラーコードは以下。
 *          ERR_VERSIONINFOERROR    指定された型番が不正
 * 説明   データコード語に誤り訂正コード語を付加し最終コード語とする。
 */
bool QREL_AddErrorCorrectData(VERSIONINFO versioninfo, ECCLEVEL ecclevel,
                               uint8_t *code)
{
    RSBLOCK rsblock = {};
    int32_t i = 0, j = 0;
    ERRCORUNIT rsinfo = {};
    int32_t al;         /* al 総コード語数 */
    int32_t el;         /* el 誤り訂正コード語数 */
    int32_t dl;         /* dl データコード語数 */
    SINGLERSBLOCK *rs;
    uint8_t data[NN];

    QREL_SetLastError(ERR_NOERROR, true);
    /* ERRCORUNITを取得する */
    if (!VER_GetRSInfo(versioninfo.version, ecclevel, &rsinfo)) {
        DestroyRSBlock(&rsinfo, &rsblock);
        QREL_SetLastError(ERR_VERSIONINFOERROR, true);
        return false;
    }
    /* RSブロックを初期化する */
    CreateRSBlock(&rsinfo, &rsblock, code);

    /* RS符号エンコーダ/デコーダを初期化する。*/
    init_rs(rsinfo.ErrcorCodeNum / rsinfo.RSBlockNum, 0);
    /* 各RSブロックに対してエンコードする。*/
    for (i = 0; i < rsblock.nblocks; i++) {
        /* RSブロック内のdataをdataに転送する。*/
        rs = &rsblock.rsb[i];
        al = rs->length;
        dl = rs->datalen;
        el = al - dl;
        if (el != E) {
            // term_rs();
            return false;   /* 使用可能誤り訂正数をチェックする。*/
        }
        for (j = 0; j < dl; j++)
            data[j] = rs->data[dl - 1 - j];
        for (j = dl; j < NN - el; j++)
            data[j] = 0;
        for (j = NN - el; j < NN; j++)
            data[j] = rs->data[al - 1 - (j - (NN - el))];
         /* RSエンコードする。*/
        encode_rs(data, data + NN - el);
        /* エンコード済みデータdataをRSブロック内のdataに転送する。*/
        for (j = 0; j < dl; j++)
            rs->data[dl - 1 - j] = data[j];
        for (j = NN - el; j < NN; j++)
            rs->data[al - 1 - (j - (NN - el))] = data[j];
    }
    /* RS符号エンコーダ/デコーダを終了する。*/
    // term_rs();

    /* データコード語+誤り訂正コード語をRSブロックからcodeに代入する */
    SetRsBlock(&rsblock, &rsinfo, false);

    /* RSブロックを解放する */
    DestroyRSBlock(&rsinfo, &rsblock);

#ifdef DEBUG_LOG
    {
        FILE *fp = fopen("datacode2.txt", "w");
        int32_t i;
        for (i = 0; i < ncodes; i++)
            fprintf(fp, "%02x ", code[i]);
        fprintf(fp, "\n");
        for (i = 0; i < ncodes; i++)
            fprintf(fp, "%d ", code[i]);
        fclose(fp);
    }
#endif

    return true;
}

/****************************************************************
 * 内部関数
 ****************************************************************/
/*
 * 関数名 SetRsBlock
 * 機能   RSブロックにコード語の内容を、またはRSブロックの内容をコード語に
 *          転送する。
 * 引数   rsblock(IN/OUT)  RSブロックのポインタ
 *        rsinfo(IN)       誤り訂正レベル毎のデータへのポインタ
 *        torsblock(INT)   trueのときRSブロックにコード語の内容を設定
 *                         falseのときコード語にRSブロックの内容を設定
 * 戻り値 常に0以外を返す
 * 説明   RSブロックにコード語の内容を、またはRSブロックの内容をコード語に
 *          転送する。
 *          例）
 *          コード語: D1D4D2D5D3D6
 *          RSブロック数: 2
 *          RS#1: D1D2D3
 *          RS#2: D4D5D6
 */
static bool SetRsBlock(RSBLOCK *rsblock, ERRCORUNIT *rsinfo, bool torsblock)
{
    int32_t allcount, datacount;
    int32_t i, j, k, l;

    allcount = rsinfo->HiblockCount * rsinfo->BlockUnit[0].AllCodeNum
               + rsinfo->LoblockCount * rsinfo->BlockUnit[1].AllCodeNum;
    datacount = rsinfo->HiblockCount * rsinfo->BlockUnit[0].DataCodeNum
                + rsinfo->LoblockCount * rsinfo->BlockUnit[1].DataCodeNum;
    /* データコード語をセットする。*/
    for (i = 0; i < datacount; i++) {
        j = i % rsinfo->RSBlockNum; /* RSblock index */
        k = i / rsinfo->RSBlockNum; /* data index */
        if (i >= rsinfo->RSBlockNum * rsinfo->BlockUnit[0].DataCodeNum)
            j += rsinfo->HiblockCount;
        if (torsblock)
            rsblock->rsb[j].data[k] = rsblock->data[i];
        else
            rsblock->data[i] = rsblock->rsb[j].data[k];
    }
    /* 誤り訂正コード語をセットする。*/
    l = 0;
    for (; i < allcount; i++) {
        j = l % rsinfo->RSBlockNum; /* RSblock index */
        k = l / rsinfo->RSBlockNum; /* data index */
        if (j < rsinfo->HiblockCount)
            k += rsinfo->BlockUnit[0].DataCodeNum;
        else
            k += rsinfo->BlockUnit[1].DataCodeNum;
        if (torsblock)
            rsblock->rsb[j].data[k] = rsblock->data[i];
        else
            rsblock->data[i] = rsblock->rsb[j].data[k];
        l++;
    }
    return true;
}

/*
 * 関数名 CreateRSBlock
 * 機能   型番・形式情報からRSブロックを作成する
 * 引数   rsinfo(IN)                ERRCORUNITのポインタ
 *        rsblock(OUT)              RSブロック（データ列）
 *        code(IN)                  データコード語
 * 戻り値 常に0以外を返す
 * 説明   型番・形式情報からRSブロックを作成する。
 *          RSブロック内のデータは0でクリアされる。
 */
static bool CreateRSBlock(ERRCORUNIT *rsinfo, RSBLOCK *rsblock,
                           uint8_t *code)
{
    int32_t rsbidx;
    int32_t blockcount;
    int32_t codeidx;

    rsblock->nblocks = (int32_t)rsinfo->RSBlockNum;
    rsblock->rsb
        = (SINGLERSBLOCK *)QRE_Malloc(rsblock->nblocks * sizeof(SINGLERSBLOCK));
    rsbidx = 0;
    /* 上段処理 */
    for (blockcount = 0; blockcount < (int32_t)rsinfo->HiblockCount;
         blockcount++) {
        MemoryAllocate(rsblock->rsb + rsbidx, rsblock->nblocks, rsbidx,
                       (int32_t)rsinfo->BlockUnit[HighBlock].AllCodeNum,
                       (int32_t)rsinfo->BlockUnit[HighBlock].DataCodeNum);
        rsbidx++;
    }
    /* 下段処理 */
    for (blockcount = 0; blockcount < (int32_t)rsinfo->LoblockCount;
         blockcount++) {
        MemoryAllocate(rsblock->rsb + rsbidx, rsblock->nblocks, rsbidx,
                       (int32_t)rsinfo->BlockUnit[LowBlock].AllCodeNum,
                       (int32_t)rsinfo->BlockUnit[LowBlock].DataCodeNum);
        rsbidx++;
    }
    /* データコード語をRSブロックに格納する。*/
    codeidx = 0;
    rsbidx = 0;
    for (blockcount = 0; blockcount < (int32_t)rsinfo->HiblockCount;
         blockcount++) {
        int32_t i;
        for (i = 0; i < rsblock->rsb[rsbidx].datalen; i++) {
            rsblock->rsb[rsbidx].data[i] = code[codeidx];
            codeidx++;
        }
        rsbidx++;
    }
    for (blockcount = 0; blockcount < (int32_t)rsinfo->LoblockCount;
         blockcount++) {
        int32_t i;
        for (i = 0; i < rsblock->rsb[rsbidx].datalen; i++) {
            rsblock->rsb[rsbidx].data[i] = code[codeidx];
            codeidx++;
        }
        rsbidx++;
    }
    rsblock->data = code;

    return true;
}

/*
 * 関数名 MemoryAllocate
 * 機能   RSブロックの各ブロックを作成する
 * 引数   RSindex(IN)            RSBlockインデックス
 *        nblocks(IN)            ブロック単位
 *        num(IN)                ブロック内での位置
 *        rsblock(IN/OUT)        RSブロック（データ列）
 *        codelen(IN)            総コード語数
 *        datacodelen(IN)        データコード語数
 * 戻り値 常に0以外を返す
 * 説明   RSブロックの各ブロックを作成する。
 *          各ブロックのデータは0で初期化する。
 *
 */
static bool MemoryAllocate(SINGLERSBLOCK *rsb,
                           int32_t nblocks, int32_t num, int32_t codelen, int32_t datacodelen)
{
    rsb->data = (uint8_t *)QRE_Malloc(codelen);
    memset(rsb->data, 0, codelen);
    rsb->length = codelen;
    rsb->datalen = datacodelen;
    rsb->nblock = nblocks;
    rsb->num = num;

    return true;
}

/*
 * 関数名 DestroyRSBlock
 * 機能   RSブロックを破棄する
 * 引数   rsinfo(IN)                ERRCORUNITのポインタ
 *        rsblock(OUT)            RSブロック（データ列）
 * 戻り値 常に0以外をを返す
 * 説明   RSブロックを破棄する。
 *
 */
static bool DestroyRSBlock(ERRCORUNIT *rsinfo, RSBLOCK *rsblock)
{
    int32_t i = 0;
    for (i = 0; i < rsinfo->HiblockCount + rsinfo->LoblockCount; i++)
        if (!MemoryFree(rsblock->rsb + i))
            return false;
    QRE_Free(rsblock->rsb);
    rsblock->rsb = NULL;

    return true;
}

/*
 * 関数名 MemoryFree
 * 機能   RSブロックの各ブロックを解放する
 * 引数   RSindex(IN)              RSBlockインデックス
 *        rsblock(IN/OUT)          RSブロック（データ列）
 * 戻り値 常に0以外を返す
 * 説明   RSブロックの各ブロックを解放する。
 *
 */
static bool MemoryFree(SINGLERSBLOCK *rsb)
{
    QRE_Free(rsb->data);
    rsb->data = NULL;
    return true;
}
