﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_Result.h>
#include <nn/ec/system/ec_TicketApi.h>
#include <nn/ldn/ldn_MonitorApi.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ResultMenu.h>
#include <nn/ns/ns_ApplicationRightsApi.h>
#include <nn/ns/ns_DynamicRightsApi.h>
#include <nn/ns/ns_DynamicRightsSystemApi.h>
#include <nn/ns/ns_TicketApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../DevMenu_Result.h"
#include "DevMenu_LauncherLibraryAppletApis.h"
#include "DevMenu_LauncherRightsCheckApis.h"

#include "DevMenuCommand_ErrorContextUtil.h"
#include "DevMenuCommand_HashUtil.h"

namespace devmenu { namespace launcher {

namespace {

    using RightsOnClient = nn::ns::ApplicationRightsOnClient;
    using RightsListOnClient = std::array< RightsOnClient, 2 >;
    using RightsOnServer = nn::ns::ApplicationRightsOnServer;
    using RightsListOnServer = std::array< RightsOnServer, 2 >;

    /**
     * @brief   インターネット利用要求を出して結果を返します。
     *
     * @param[in]   pNetworkConnection  ネットワーク接続を扱うクラスのポインタ
     * @param[in]   pIsCancelRequested  処理のキャンセル要求の有無
     *
     * @return   処理の結果を返します。利用要求が受理された場合は ResultSuccess が返ります。
     *
     * @details
     *  本関数は長時間ブロックすることがあります。少なくとも利用要求受の検証が完了するまでブロックします。
     *  利用要求が却下された場合は、インターネット設定やエラービューア等のライブラリアプレットが起動されることがあります。
     *  ライブラリアプレットが起動された場合は、ユーザー操作により終了するのを待って処理を再開します。
     *
     *  利用要求が却下されてエラーハンドリングを行った結果、エラーが解消したと思われる場合は自動的に再度利用要求を提出します。
     */
    const nn::Result SubmitNetworkRequestAndWait( nn::nifm::NetworkConnection* pNetworkConnection, bool* pIsCancelRequested ) NN_NOEXCEPT
    {
        pNetworkConnection->SubmitRequest();

        while ( NN_STATIC_CONDITION( true ) )
        {
            if ( !pNetworkConnection->IsRequestOnHold() )
            {
                break;
            }

            if ( *pIsCancelRequested )
            {
                NN_RESULT_SUCCESS;
            }

            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
        }

        if ( !pNetworkConnection->IsAvailable() )
        {
            DEVMENU_LOG( "Network is not available.\n" );

            NN_RESULT_TRY( HandleNetworkRequestResult( pNetworkConnection ) )
                NN_RESULT_CATCH( nn::nifm::ResultErrorHandlingCompleted )
            {
                // 再度ネットワーク利用要求を出す
                NN_RESULT_DO( SubmitNetworkRequestAndWait( pNetworkConnection, pIsCancelRequested ) );
            }
            NN_RESULT_END_TRY
        }

        NN_RESULT_SUCCESS;
    }

#if 0
    /**
     * @brief   ネットワークサービスアカウントを利用可能か判別します。(未使用)
     *
     * @param[in]  userId  チェック対象の Uid
     *
     * @return  利用可能であれば true を返します。
     *
     * @details
     *  本関数はライブラリアプレットが表示されてユーザー操作が必要になり、長時間ブロックする可能性があります。
     *  NSA が使用できない状態の場合、使用できるようにするためにライブラリアプレットが起動されます。
     */
    bool IsNetworkServiceAccountAvailable( const nn::account::Uid& userId ) NN_NOEXCEPT
    {
        NN_ASSERT( nn::account::InvalidUid != userId );

        // userId のユーザーの NSA 可用性をチェック
        nn::account::NetworkServiceAccountManager manager;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::GetNetworkServiceAccountManager( &manager, userId ) );
        const auto availability = manager.CheckNetworkServiceAccountAvailability();
        NN_ABORT_UNLESS( availability.IsSuccess() || nn::account::ResultNetworkServiceAccountUnavailable::Includes( availability ) );

        if ( nn::account::ResultNetworkServiceAccountUnavailable::Includes( availability ) )
        {
            // アカウント管理アプレットを起動し、NSA の有効化を試みる
            // 有効化された場合のみ先に進む
            const auto result = EnsureNetworkServiceAccountAvailable( userId );

            // TORIAEZU: ModalView の切り替えのために待ち時間を設ける
            //           CanCheckLaunchRightsOnServer() の実行前にユーザー選択画面が消えるようにするには、
            //           NSA 可用性チェックを非同期にしつつ、ローカル通信エラー時の StartModal() はメインスレッドから
            //           呼ばれるようにする必要がありそう
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 300 ) );

            return result.IsSuccess();
        }
        return true;
    }
#endif

    /**
     * @brief   アプリケーションを使用する権利情報リスト（アプリケーションと追加コンテンツ）をサーバーから取得します
     *
     * @param[out]  pOutCount               有効な権利の数の出力先ポインタ
     * @param[out]  pOutRightsOnServerList  権利リストの出力先ポインタ
     * @param[in]   pIsCancelRequested      処理のキャンセル要求の有無
     * @param[in]   applicationId           チェック対象の ApplicationId
     * @param[in]   userId                  チェック対象のユーザー
     *
     * @return  処理の結果が返ります。
     */
    const nn::Result RequestApplicationRightsListOnServer(
        int* pOutCount, RightsListOnServer* pOutRightsList, bool* pIsCancelRequested,
        const nn::ncm::ApplicationId& applicationId, const nn::account::Uid& userId = nn::account::InvalidUid ) NN_NOEXCEPT
    {
        nn::ns::AsyncApplicationRightsOnServerList async;
        if ( nn::account::InvalidUid == userId )
        {
            NN_RESULT_DO( nn::ns::RequestApplicationRightsOnServer( &async, applicationId ) );
        }
        else
        {
            NN_RESULT_DO( nn::ns::RequestApplicationRightsOnServer( &async, applicationId, userId ) );
        }

        while ( NN_STATIC_CONDITION( true ) )
        {
            if ( async.TryWait() )
            {
                break;
            }

            if ( *pIsCancelRequested )
            {
                async.Cancel();
                NN_RESULT_SUCCESS;
            }

            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
        }
        NN_RESULT_DO( devmenuUtil::Get( pOutCount, pOutRightsList->data(), static_cast< int >( pOutRightsList->size() ), &async ) );
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   サーバーから取得した権利情報リストからアプリケーションの権利を取得します。
     *
     * @param[in]  rightsList   サーバーから取得した権利情報リスト
     * @param[in]  rightsCount  リスト上の有効な権利の数
     *
     * @return  アプリケーションの権利情報を返します。
     */
    const RightsOnServer GetApplicationRights( const RightsListOnServer& rightsList, int rightsCount ) NN_NOEXCEPT
    {
        const auto endList = rightsList.begin() + rightsCount;

        const auto rights =
            std::find_if( rightsList.begin(), endList, []( const RightsOnServer& rights ) NN_NOEXCEPT
            {
                return nn::ns::ApplicationContentType::Application == rights.type;
            }
        );

        NN_ABORT_UNLESS_NOT_EQUAL( rights, endList );

        return *rights;
    }

    /**
     * @brief   サーバーから取得した権利情報リストから追加コンテンツの権利を取得します。
     *
     * @param[in]  rightsList   サーバーから取得した権利情報リスト
     * @param[in]  rightsCount  リスト上の有効な権利の数
     *
     * @return  追加コンテンツの権利情報を返します。
     */
    const RightsOnServer GetAddOnContentRights( const RightsListOnServer& rightsList, int rightsCount ) NN_NOEXCEPT
    {
        const auto endList = rightsList.begin() + rightsCount;

        const auto rights =
            std::find_if( rightsList.begin(), endList, []( const RightsOnServer& rights ) NN_NOEXCEPT
            {
                return nn::ns::ApplicationContentType::AddOnContent == rights.type;
            }
        );

        NN_ABORT_UNLESS_NOT_EQUAL( rights, endList );

        return *rights;
    }

    /**
     * @brief   サーバーから取得した権利情報リストを参照し、権利を取得可能かを判別します。
     *
     * @param[in]  rights   権利情報
     *
     * @return  権利を取得可能な場合は true を返します。
     */
    bool IsRightsDownloadable( const RightsOnServer& rights ) NN_NOEXCEPT
    {
        return rights.HasAvailableRights() && ( rights.RecommendAssignRights() || rights.RecommendSyncTicket() );
    }

    /**
     * @brief   サーバーからアプリケーションの権利を取得できない理由を Result 値で返します。
     *
     * @param[in]  applicationRights   アプリケーションの権利
     *
     * @pre
     *  - IsRightsDownloadable( applicationRights ) が false を返す
     *
     * @return  取得できない理由を返します。
     */
    const nn::Result GetApplicationRightsUnavalilableReason( const RightsOnServer& applicationRights ) NN_NOEXCEPT
    {
        // アプリケーションの場合はサーバーに有効な権利のみがある状態でないと起動できない
        if ( !applicationRights.HasAvailableRights() )
        {
            NN_ABORT_UNLESS( applicationRights.HasUnavailableRights() );

            // エラーの優先度は Ocean に準拠
            if ( applicationRights.reason.SystemUpdateRequired() )
            {
                return ResultSystemUpdateRequiredToCheckLaunchRights();
            }
            else if ( applicationRights.reason.AssignableRightsLimitExceeded() )
            {
                return ResultAssignableRightsLimitExceeded();
            }
            else if ( applicationRights.reason.HasDeviceLinkedRightsOnlyContent() )
            {
                return ResultHasDeviceLinkedRightsOnlyContent();
            }
            else if ( applicationRights.reason.NotReleased() )
            {
                return ResultNotReleasedContent();
            }
            else if ( applicationRights.reason.NoRights() )
            {
                return ResultNoLaunchRightsOnServer();
            }

            NN_ASSERT( "Must not come here" );
        }

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   サーバーから追加コンテンツの権利を取得できない理由を Result 値で返します。
     *
     * @param[in]  addOnContentRights   追加コンテンツの権利
     *
     * @pre
     *  - IsRightsDownloadable( addOnContentRights ) が false を返す
     *
     * @return  取得できない理由を返します。
     */
    const nn::Result GetAddOnContentRightsUnavalilableReason( const RightsOnServer& addOnContentRights ) NN_NOEXCEPT
    {
#if 0   // 追加コンテンツの権利の有無で起動可否は変わらない。ひとまず無効化しておく
        if ( addOnContentRights.HasUnavailableRights() )
        {
            // ログ出力など
        }
#else
        NN_UNUSED( addOnContentRights );
#endif

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   サーバーから権利を取得します。
     *
     * @param[in]  rightsList          サーバーから取得した権利情報リスト
     * @param[in]  rightsCount         リスト上の有効な権利の数
     * @param[in]  pIsCancelRequested  キャンセル要求の有無
     *
     * @return  処理の結果が返ります。 取得が成功した場合や ResultSuccess が返ります。
     *
     * @details
     *   本関数は権利の取得処理で長時間ブロックする可能性があります。
     *   ResultSuccess が返ったとしても、実際に権利を取得できとすることはできません。
     *   権利の取得状態は nn::ns::GetApplicationRightsOnClient() 系の API で別途確認する必要があります。
     */
    const nn::Result DownloadRights( const RightsListOnServer& rightsList, int rightsCount, bool* pIsCancelRequested ) NN_NOEXCEPT
    {
        const auto endList = rightsList.begin() + rightsCount;

        // 非同期処理が完了するの待って結果を取得する
        auto waitAndGetResult = []( nn::ns::AsyncResult* pAsyncResult, bool* pIsCancelRequested ) NN_NOEXCEPT -> nn::Result
        {
            while ( NN_STATIC_CONDITION( true ) )
            {
                if ( pAsyncResult->TryWait() )
                {
                    break;
                }

                if ( *pIsCancelRequested )
                {
                    pAsyncResult->Cancel();
                    NN_RESULT_SUCCESS;
                }

                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
            }
            NN_RESULT_DO( devmenuUtil::Get( pAsyncResult) );
            NN_RESULT_SUCCESS;
        };

        if ( std::any_of( rightsList.begin(), endList,
            []( const RightsOnServer& rights ) NN_NOEXCEPT { return rights.RecommendAssignRights(); } ) )
        {
            // 権利取得
            nn::ns::AsyncResult asyncResult;
            NN_RESULT_DO( nn::ns::RequestAssignRights( &asyncResult, rightsList.data(), rightsCount ) );
            NN_RESULT_DO( waitAndGetResult( &asyncResult, pIsCancelRequested ) );
            if ( *pIsCancelRequested )
            {
                NN_RESULT_SUCCESS;
            }
        }

        if ( std::any_of( rightsList.begin(), endList,
            []( const RightsOnServer& rights ) NN_NOEXCEPT { return rights.RecommendSyncTicket(); } ) )
        {
            // 権利同期
            nn::ns::AsyncResult asyncResult;
            NN_RESULT_DO( nn::ns::RequestSyncRights( &asyncResult ) );
            NN_RESULT_DO( waitAndGetResult( &asyncResult, pIsCancelRequested ) );
            if ( *pIsCancelRequested )
            {
                NN_RESULT_SUCCESS;
            }
        }

        NN_RESULT_SUCCESS;
    }

} // ~namespace devmenu::accounts::<anonymous>

const nn::Result CheckContentsAvailability( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    NN_RESULT_DO( nn::ns::CheckApplicationLaunchVersion( applicationId ) );
    NN_RESULT_DO( devmenuUtil::VerifyApplicationAndPatchCombination( applicationId ) );
    NN_RESULT_SUCCESS;
}

const nn::Result CheckApplicationRightsOnClientForApplication(
    bool* pOutHasLaunchRights, const nn::ncm::ApplicationId& applicationId,
    const nn::account::Uid& userId ) NN_NOEXCEPT
{
    bool hasLaunchRights = false;
    NN_UTIL_SCOPE_EXIT
    {
        *pOutHasLaunchRights = hasLaunchRights;
    };

    RightsOnClient rights;
    if ( nn::account::InvalidUid == userId )
    {
        NN_RESULT_DO( nn::ns::GetApplicationRightsOnClientForApplication( &rights, applicationId ) );
    }
    else
    {
        NN_RESULT_DO( nn::ns::GetApplicationRightsOnClientForApplication( &rights, applicationId, userId ) );
    }
    hasLaunchRights = rights.HasAvailableRights() && !rights.HasUnavailableRights();

    NN_RESULT_SUCCESS;
}

bool IsLocalCommunicationRunning() NN_NOEXCEPT
{
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::InitializeMonitor();
    NN_UTIL_SCOPE_EXIT
    {
        nn::ldn::FinalizeMonitor();
    };

    return nn::ldn::GetNetworkInfoForMonitor( &networkInfo ).IsSuccess();
}

void AskToDisconnectLocalCommunication(
    const std::function< void( ModalView* ) >& startModalFunction,
    const std::function< void() >& disconnectCallback,
    const std::function< void() >& cancelCallback ) NN_NOEXCEPT
{
    auto pMessageView = new MessageView( false );
    pMessageView->AddMessage(
        "Cannot launch the application\nsince local network communication is runnning." );
    pMessageView->AddMessage( "Disconnect the communication?" );

    pMessageView->AddButton(
        "Cancel",
        [ cancelCallback ]( void* pParam, nn::TimeSpan& timespan )
        {
            NN_UNUSED( pParam );
            timespan = nn::TimeSpan::FromMilliSeconds( 100 );
            cancelCallback();
        }
    );

    pMessageView->AddButton(
        "Disconnect",
        [ disconnectCallback ]( void* pParam, nn::TimeSpan& timespan )
        {
            NN_UNUSED( pParam );
            timespan = nn::TimeSpan::FromMilliSeconds( 100 );
            disconnectCallback();
        }
    );

    startModalFunction( pMessageView );
}

const nn::Result CheckLaunchRightsOnServer(
    RightsState* pOutRightsState,
    AsyncTaskWithProgressView::ExitReason* pOutExitReason,
    bool* pIsCancelRequested,
    const nn::ncm::ApplicationId& applicationId,
    const nn::account::Uid& userId ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutRightsState );
    NN_SDK_REQUIRES_NOT_NULL( pOutExitReason );
    NN_SDK_REQUIRES_NOT_NULL( pIsCancelRequested );

    auto rightsState = RightsState_None;
    NN_UTIL_SCOPE_EXIT
    {
        *pOutRightsState = rightsState;
    };

    auto exitReason = AsyncTaskWithProgressView::ExitReason_Failure;
    NN_UTIL_SCOPE_EXIT
    {
        *pOutExitReason = exitReason;
    };

    auto& isCancelRequested = *pIsCancelRequested;
    NN_ASSERT( false == isCancelRequested );

    // ネットワーク利用要求を出す
    nn::util::TypedStorage
        < nn::nifm::NetworkConnection, sizeof( nn::nifm::NetworkConnection ), NN_ALIGNOF( nn::nifm::NetworkConnection ) > networkConnectionStorage;
    auto* pNetworkConnection = new ( &networkConnectionStorage )nn::nifm::NetworkConnection();

    nn::nifm::SetRequestRequirementPreset(
        pNetworkConnection->GetRequestHandle(),
        nn::nifm::RequirementPreset_InternetForApplet
    );

    NN_RESULT_TRY( SubmitNetworkRequestAndWait( pNetworkConnection, &isCancelRequested ) )
        NN_RESULT_CATCH_ALL
        {
            // ユーザー操作で解決しなかったケースを想定し、DevMenu でもメッセージ表示のためにエラーハンドリングする
            exitReason = AsyncTaskWithProgressView::ExitReason_CancelOnLibraryApplet;
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY

    NN_UTIL_SCOPE_EXIT
    {
        pNetworkConnection->CancelRequest();
    };

    if ( isCancelRequested )
    {
        exitReason = AsyncTaskWithProgressView::ExitReason_CancelOnModalView;
        NN_RESULT_SUCCESS;
    }

    // サーバーに全ユーザーの機器認証の権利を問い合わせる
    RightsListOnServer deviceAuthenticatedRightsList;
    int rightsCount;

    NN_RESULT_DO( RequestApplicationRightsListOnServer(
        &rightsCount, &deviceAuthenticatedRightsList, pIsCancelRequested, applicationId ) );

    if ( isCancelRequested )
    {
        exitReason = AsyncTaskWithProgressView::ExitReason_CancelOnModalView;
        NN_RESULT_SUCCESS;
    }

    if ( IsRightsDownloadable( GetApplicationRights( deviceAuthenticatedRightsList, rightsCount ) ) )
    {
        // 権利をダウンロードする
        NN_RESULT_DO( DownloadRights( deviceAuthenticatedRightsList, rightsCount, pIsCancelRequested ) );

        if ( isCancelRequested )
        {
            exitReason = AsyncTaskWithProgressView::ExitReason_CancelOnModalView;
            NN_RESULT_SUCCESS;
        }

        // クライアント上の権利を再度チェックする
        bool hasLaunchRights;
        NN_RESULT_DO( CheckApplicationRightsOnClientForApplication( &hasLaunchRights, applicationId ) );

        if ( !hasLaunchRights )
        {
            return ResultNoLaunchRightsOnServer();
        }

        rightsState = RightsState_DeviceAuthenticated;
    }
    else // 機器認証されたアプリケーションの使用権利が無い場合は、選択したユーザーのアプリケーションと追加コンテンツの権利をサーバーに問い合わせる
    {
        RightsListOnServer timeLimitedRightsList;
        rightsCount = 0;

        // Note: ここで NSA 可用性チェックが必要かもしれないが、Ocean の仕様書からは確実に必要だと読み取れないので保留

        NN_RESULT_DO( RequestApplicationRightsListOnServer(
            &rightsCount, &timeLimitedRightsList, pIsCancelRequested, applicationId, userId ) );

        if ( isCancelRequested )
        {
            exitReason = AsyncTaskWithProgressView::ExitReason_CancelOnModalView;
            NN_RESULT_SUCCESS;
        }

        const auto endList = timeLimitedRightsList.begin() + rightsCount;
        const auto applicaitonRights = GetApplicationRights( timeLimitedRightsList, rightsCount );
        const auto addOnContentRights = GetAddOnContentRights( timeLimitedRightsList, rightsCount );

        // アプリケーションの権利を取得可能かチェックする
        if ( !IsRightsDownloadable( applicaitonRights ) )
        {
            NN_RESULT_DO( GetApplicationRightsUnavalilableReason( applicaitonRights ) );
        }

        // 追加コンテンツの権利をチェックする
        if ( !IsRightsDownloadable( addOnContentRights ) )
        {
            NN_RESULT_DO( GetAddOnContentRightsUnavalilableReason( addOnContentRights ) );
        }

        // 権利をダウンロードする
        NN_RESULT_DO( DownloadRights( timeLimitedRightsList, rightsCount, pIsCancelRequested ) );

        if ( isCancelRequested )
        {
            exitReason = AsyncTaskWithProgressView::ExitReason_CancelOnModalView;
            NN_RESULT_SUCCESS;
        }

        // クライアント上の権利を再度チェックする
        bool hasLaunchRights;
        NN_RESULT_DO( CheckApplicationRightsOnClientForApplication( &hasLaunchRights, applicationId, userId ) );

        if ( !hasLaunchRights )
        {
            return ResultNoLaunchRightsOnServer();
        }

        // 利用できるユーザーが制限されている権利があるかチェック
        if ( std::any_of( timeLimitedRightsList.begin(), endList,
            []( const RightsOnServer& rights ) NN_NOEXCEPT { return rights.IsAccountRestrictedRights(); } ) )
        {
            // 権利利用可能者を登録する
            NN_RESULT_DO( nn::ns::RegisterUserOfAccountRestrictedRights( userId ) );
        }

        rightsState = RightsState_TimeLimited;
    }

    exitReason = AsyncTaskWithProgressView::ExitReason_Success;
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

}} // ~namespace devmenu::launcher
