﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/err/err_ShowErrorApi.h>
#include <nn/err/err_ShowErrorApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/swkbd/swkbd_Api.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Utf8StringUtil.h>

#include "../DevMenu_Common.h"
#include "DevMenu_LauncherLibraryAppletApis.h"

namespace devmenu { namespace launcher {

namespace {
    // DevMenu による LA 起動有無
    bool g_IsLibraryAppletLaunchedByDevMenu = false;
}

bool IsLibraryAppletLaunchedByDevMenu() NN_NOEXCEPT
{
    return g_IsLibraryAppletLaunchedByDevMenu;
}

const nn::Result EnsureNetworkServiceAccountAvailable( const nn::account::Uid& userId ) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::GetNetworkServiceAccountManager( &manager, userId ) );

    g_IsLibraryAppletLaunchedByDevMenu = true;
    auto result = manager.EnsureNetworkServiceAccountAvailable();
    g_IsLibraryAppletLaunchedByDevMenu = false;
    return result;
}

const nn::Result ShowApplicationLegalInformationPage( nn::web::OfflineHtmlPageReturnValue* pOutOfflineHtmlPageReturnValue, const nn::web::ShowApplicationLegalInformationPageArg& showApplicationLegalInformationPageArg ) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    g_IsLibraryAppletLaunchedByDevMenu = true;
    auto result = nn::web::ShowApplicationLegalInformationPage( pOutOfflineHtmlPageReturnValue, showApplicationLegalInformationPageArg );
    g_IsLibraryAppletLaunchedByDevMenu = false;
    return result;
#else
    NN_UNUSED( pOutOfflineHtmlPageReturnValue );
    NN_UNUSED( showApplicationLegalInformationPageArg );
    NN_RESULT_SUCCESS;
#endif
}

void ShowError( const nn::Result& result ) NN_NOEXCEPT
{
    g_IsLibraryAppletLaunchedByDevMenu = true;
    nn::err::ShowError( result );
    g_IsLibraryAppletLaunchedByDevMenu = false;
}

void ShowErrorRecord( const void* pRecordedErrorParam, size_t recordedErrorParamSize ) NN_NOEXCEPT
{
    g_IsLibraryAppletLaunchedByDevMenu = true;
    nn::err::ShowErrorRecord( pRecordedErrorParam, recordedErrorParamSize );
    g_IsLibraryAppletLaunchedByDevMenu = false;
}

const nn::Result HandleNetworkRequestResult( nn::nifm::NetworkConnection* pNetworkConnection ) NN_NOEXCEPT
{
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    nn::applet::TakeScreenShotOfOwnLayerIntoCallerAppletBuffer( true );
#endif

    g_IsLibraryAppletLaunchedByDevMenu = true;
    auto result = pNetworkConnection->HandleNetworkRequestResult();
    g_IsLibraryAppletLaunchedByDevMenu = false;
    return result;
}

const nn::Result LaunchSoftwareKeyboardAndGetString( char* pOutString, size_t outSize, const char* pInitialText, const KeyboardConfig& keyboardConfig ) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )

    // Keyborad settings
    nn::swkbd::ShowKeyboardArg keyboardArg;
    if ( keyboardConfig.preset == KeyboardConfig::Preset::Preset_UserName )
    {
        nn::swkbd::MakePreset( &( keyboardArg.keyboardConfig ), nn::swkbd::Preset_UserName );
    }
    else
    {
        nn::swkbd::MakePreset( &( keyboardArg.keyboardConfig ), nn::swkbd::Preset_Default );
    }
    if ( keyboardConfig.keyboardMode == KeyboardConfig::KeyboardMode::KeyboardMode_Ascii )
    {
        keyboardArg.keyboardConfig.keyboardMode = nn::swkbd::KeyboardMode_ASCII;
    }
    else
    {
        keyboardArg.keyboardConfig.keyboardMode = nn::swkbd::KeyboardMode_Full;
    }
    keyboardArg.keyboardConfig.isPredictionEnabled = keyboardConfig.isPredictionEnabled;
    keyboardArg.keyboardConfig.isUseUtf8 = keyboardConfig.isUseUtf8;
    keyboardArg.keyboardConfig.isUseNewLine = keyboardConfig.isUseNewLine;
    keyboardArg.keyboardConfig.textMaxLength = keyboardConfig.textMaxLength;
    keyboardArg.keyboardConfig.textMinLength = keyboardConfig.textMinLength;
    nn::swkbd::SetHeaderTextUtf8( &( keyboardArg.keyboardConfig ), keyboardConfig.headerTextUtf8);
    nn::swkbd::SetGuideTextUtf8( &( keyboardArg.keyboardConfig ), keyboardConfig.guideTextUtf8);

    // Work buffer settings
    const auto workBufSize = nn::swkbd::GetRequiredWorkBufferSize( false );
    void* workBuf = MemoryAllocator::AllocAlignedMemory( nn::os::MemoryPageSize, workBufSize );
    NN_ABORT_UNLESS_NOT_NULL( workBuf );
    NN_ASSERT_ALIGNED( workBuf, nn::os::MemoryPageSize );
    keyboardArg.workBufSize = workBufSize;
    keyboardArg.workBuf = workBuf;

    // Output buffer settings
    const auto outBufSize = nn::swkbd::GetRequiredStringBufferSize();
    char* outBuf = reinterpret_cast< char* >( MemoryAllocator::AllocAlignedMemory( nn::os::MemoryPageSize, outBufSize ) );
    NN_ABORT_UNLESS_NOT_NULL( outBuf );
    NN_ASSERT_ALIGNED( outBuf, nn::os::MemoryPageSize );
    nn::swkbd::String outBufString;
    outBufString.ptr = outBuf;
    outBufString.bufSize = outBufSize;

    nn::swkbd::SetInitialTextUtf8( &keyboardArg, pInitialText );
    g_IsLibraryAppletLaunchedByDevMenu = true;
    const auto result = nn::swkbd::ShowKeyboard( &outBufString, keyboardArg );
    g_IsLibraryAppletLaunchedByDevMenu = false;

    if ( result.IsSuccess() )
    {
        nn::util::CopyUtf8String( pOutString, outSize, outBuf, outBufSize,
            nn::util::GetCodePointCountOfUtf8String( outBuf, nn::util::Strnlen( outBuf, outBufSize ) ) );
    }

    MemoryAllocator::FreeAlignedMemory( outBuf );
    MemoryAllocator::FreeAlignedMemory( workBuf );

    return result;

#else

    NN_RESULT_SUCCESS;

#endif
}

}} // ~namespace devmenu::launcher
