﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/eupld/eupld_Control.h>
#include <nn/util/util_StringUtil.h>
#include <nn/time.h>
#include <nn/time/time_CalendarTime.h>
#include "DevMenu_ErrorSettings.h"
#include "../DevMenu_RootSurface.h"
#include "../Applications/DevMenu_ApplicationsCommon.h"
#include "../Common/DevMenu_CommonSettingsApi.h"

#include <glv_ScissorBoxView.h>

using namespace nn;

namespace devmenu { namespace error {

namespace {

template <typename T>
T LoadFwdbg(const char* name, const char* key) NN_NOEXCEPT
{
    T value = {};
    GetFixedSizeFirmwareDebugSettingsItemValue(&value, name, key);
    return value;
}

} // ~devmenu::error::<anonymous>

ErrorSettingsScene::ErrorSettingsScene(devmenu::Page* pParent, glv::Rect rect) NN_NOEXCEPT
    : devmenu::Scene(pParent, rect, true)
    , m_Layout("<,", 8, 16)
    // 非 PROD MODE では「fwdbg 設定の操作 + Share Error Information の操作」を、PROD MODE では「Share Error Infomation の操作」のみを行うので、実態に合わせて表記を変える。
    , m_UploadCheckBox(devmenu::IsDebugModeEnabled() ? "Enable Upload" : "Share Error Information")
#if defined( NN_DEVMENUSYSTEM )
    , m_AutoUploadWithoutOptedInAccountCheckBox("Enable Auto Upload without opted-in Nintendo Account")
#endif
    , m_StatusLabel("", CommonValue::DefaultLabelSpec)
{
    m_IsUploadEnabledInCurrentBoot = LoadFwdbg<bool>("eupld", "upload_enabled");
    auto permission = settings::system::GetErrorReportSharePermission();
    m_UploadCheckBox.SetValue(m_IsUploadEnabledInCurrentBoot && permission == settings::system::ErrorReportSharePermission_Granted);
    m_UploadCheckBox.SetCallback([](const glv::Notification& n)->void { n.receiver<ErrorSettingsScene>()->UpdateUploadCheckBox(); }, this);
    m_Layout << m_UploadCheckBox;

#if defined( NN_DEVMENUSYSTEM )
    if( devmenu::IsDebugModeEnabled() )
    {
        m_AutoUploadWithoutOptedInAccountCheckBox.SetValue(LoadFwdbg<bool>("eclct", "analytics_override"));
        m_AutoUploadWithoutOptedInAccountCheckBox.SetCallback([](const glv::Notification& n)->void { n.receiver<ErrorSettingsScene>()->UpdateAutoUploadWithoutOptedInAccountCheckBox(); }, this);
        m_Layout << m_AutoUploadWithoutOptedInAccountCheckBox;
    }
#endif

    if( m_IsUploadEnabledInCurrentBoot )
    {
        RefreshStatusLabel();
        m_Layout << m_StatusLabel;
    }

    m_Layout.arrange().fit(false);
    m_Layout.disable(glv::Property::HitTest);
    *this << m_Layout;
}

void ErrorSettingsScene::Activate() NN_NOEXCEPT
{
    if( m_IsUploadEnabledInCurrentBoot )
    {
        RefreshStatusLabel();
        m_StatusRefreshTimer.StartPeriodic(nn::TimeSpan::FromSeconds(1), nn::TimeSpan::FromSeconds(1));
    }
}

void ErrorSettingsScene::Deactivate() NN_NOEXCEPT
{
    if( m_IsUploadEnabledInCurrentBoot )
    {
        m_StatusRefreshTimer.Stop();
    }
}

glv::View* ErrorSettingsScene::GetFocusableChild() const NN_NOEXCEPT
{
    return nullptr;
}

void ErrorSettingsScene::Refresh() NN_NOEXCEPT
{
    if( m_IsUploadEnabledInCurrentBoot &&  m_StatusRefreshTimer.TryWait() )
    {
        RefreshStatusLabel();
    }
}

void ErrorSettingsScene::UpdateUploadCheckBox() NN_NOEXCEPT
{
    auto value = m_UploadCheckBox.GetValue();
    nn::settings::system::SetErrorReportSharePermission(value ? settings::system::ErrorReportSharePermission_Granted : settings::system::ErrorReportSharePermission_Denied);
    if( devmenu::IsDebugModeEnabled() )
    {
        SetFixedSizeFirmwareDebugSettingsItemValue("eupld", "upload_enabled", value);
        GetRootSurfaceContext()->DisplayRebootRequestText();
    }
}

#if defined( NN_DEVMENUSYSTEM )
void ErrorSettingsScene::UpdateAutoUploadWithoutOptedInAccountCheckBox() NN_NOEXCEPT
{
    auto value = m_AutoUploadWithoutOptedInAccountCheckBox.GetValue();
    SetFixedSizeFirmwareDebugSettingsItemValue("eclct", "analytics_override", value);
    GetRootSurfaceContext()->DisplayRebootRequestText();
}
#endif

void ErrorSettingsScene::RefreshStatusLabel() NN_NOEXCEPT
{
    if( !m_IsUploadEnabledInCurrentBoot )
    {
        return;
    }
    bool isEnabled;
    uint32_t expireTickSeconds;
    eupld::Control c;

    auto getResult = c.GetAutoUpload(&isEnabled, &expireTickSeconds);
    if( getResult.IsFailure() )
    {
        DEVMENU_LOG("ErrorSettingsScene::Refresh: Failed to get error report upload status: %03d-%04d (0x%08x)\n",
            getResult.GetModule(), getResult.GetDescription(), getResult.GetInnerValueForDebug());
        return;
    }

#if defined( NN_DEVMENUSYSTEM )

    char nextCheckMessage[64];
    auto nextCheckTimeSpan = nn::TimeSpan::FromSeconds(static_cast<int64_t>(expireTickSeconds)) - os::GetSystemTick().ToTimeSpan();
    if( nextCheckTimeSpan.GetSeconds() <= 0 )
    {
        util::Strlcpy(nextCheckMessage, "Now Checking...", static_cast<int>(sizeof(nextCheckMessage)));
    }
    else
    {
        time::PosixTime currentTime;
        time::StandardUserSystemClock::GetCurrentTime(&currentTime);
        auto nextCheckTime = currentTime += nextCheckTimeSpan;

        time::CalendarTime calendarTime;
        time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, nextCheckTime));

        util::SNPrintf(nextCheckMessage, sizeof(nextCheckMessage), "%04d-%02d-%02d %02d:%02d:%02d (%lld seconds later)",
            calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.month, calendarTime.second, nextCheckTimeSpan.GetSeconds());
    }
    char statusMessage[128];
    util::SNPrintf(statusMessage, sizeof(statusMessage), "Auto Upload: %s.\nNext NintendoAccouht Check: %s",
        isEnabled ? "Enabled" : "Disabled",
        nextCheckMessage);
    m_StatusLabel.setValue(static_cast<const char*>(statusMessage));
#else
    char statusMessage[128];
    util::SNPrintf(statusMessage, sizeof(statusMessage), "Auto Upload: %s\n", isEnabled ? "Enabled" : "Disabled");
    m_StatusLabel.setValue(static_cast<const char*>(statusMessage));
#endif
}

}} // ~namespace devmenu::error, ~namespace devmenu
